/*
PointInLineRRAL v1.0.0 by Sheldor.  05/27/2013
A NanoBot with multimode movement and RRAL targeting.
Codesize: 249 bytes without any colors

Point-in-line is a fencing technique.  https://en.wikipedia.org/wiki/Fencing_terminology#P
The "RRAL" stands for Randomized Rolling Average Linear targeting; it's a reference to John Cleland's nanobot naming scheme.

Credits:
Many thanks to John Cleland (nz.jdc) for his Neophyte series of nanobots, from which I drew much inspiration.
Also, a general thanks to all open source bot authors and contributors to the RoboWiki.

PointInLineRRAL is open source and released under the terms of the RoboWiki Public Code License (RWPCL) - Version 1.1.  http://robowiki.net/wiki/RWPCL
*/

package sheldor.nano;
import robocode.*;
import robocode.util.Utils;

public class PointInLineRRAL extends AdvancedRobot
{
	//Constants
	static final int NUMBER_OF_AVERAGES = 100;
	
	//Global variables
	static double[] averageEnemyLateralVelocities = new double[NUMBER_OF_AVERAGES + 1];	
	static double   enemyEnergy;
	static double   movementDirection;
	static double   movementMode                  = -1;
	
	//En garde!
	public void run()
	{
		//Set the gun to turn independently of the robot body.
		setAdjustGunForRobotTurn(true);
		
		//Start spinning radar and initialize movementDirection.
		setTurnRadarRightRadians(movementDirection = 240);
	}
	
	public void onScannedRobot(ScannedRobotEvent e)
	{
		//Local variables
		double absoluteBearing      = e.getBearingRadians() + getHeadingRadians();
		double enemyLateralVelocity = e.getVelocity() * Math.sin(e.getHeadingRadians() - absoluteBearing);
		
		//Keep rolling averages of the enemy's lateral velocity at every depth between 0 and NUMBER_OF_AVERAGES.
		int i = NUMBER_OF_AVERAGES;
		do
		{
			averageEnemyLateralVelocities[i] = ((i * averageEnemyLateralVelocities[i]) + enemyLateralVelocity) / (i + 1);
		}
		while (--i > 0);
		
		//Pick one of the rolling averages at random and use it to aim.
		setTurnGunRightRadians(Utils.normalRelativeAngle(Math.asin(averageEnemyLateralVelocities[(int)(Math.random() * NUMBER_OF_AVERAGES)] / 14) +
			absoluteBearing - getGunHeadingRadians()));
		
		//Randomized stop and go/oscillating movement
		if ((char)((enemyEnergy - 1.0999 - (enemyEnergy = e.getEnergy()))) < 2)
		{
  			setAhead((movementDirection *= movementMode) * (Math.random() + 0.15));
    	}

		//Stay mostly perpendicular to the enemy, but try to maintain a distance of 250 pixels.
		//The absolute bearing variable is re-used here to represent the enemy distance.  This saves one byte.
    	setTurnRightRadians(Math.cos(e.getBearingRadians() - ((absoluteBearing = e.getDistance()) - 250) * movementDirection / 108000));
		
		//Fire medium power bullets most of the time, but use full power at very close range.	
		setFire(2 + (100 / (int)absoluteBearing));
		
		//Pseudo-infinite radar lock.
		setTurnRadarLeftRadians(getRadarTurnRemaining());
	}
	
	public void onHitWall(HitWallEvent e)
	{
		//When the bot hits a wall, reverse movement direction and start moving immediately.
   		setAhead(movementDirection = -movementDirection);
	}
	
	public void onHitByBullet(HitByBulletEvent e)
	{
		//When the bot is hit by an enemy bullet, alternate between one-way and reversing orbit movements.
    	movementMode = -movementMode;
	}
}																									