package pl.Enemy;

/*******************************************************************************
 * A class for keeping track of an enemy. It is useful for storing information
 * from ScannedRobotEvents, especially if you want to use information outside of
 * the onScannedRobot() method. All angle-related variables are kept in radians.
 ******************************************************************************/
public class Enemy {
	/**
	 * The enemy's bearing in radians
	 */
	public double bearing;
	/**
	 * The enemy's distance
	 */
	public double distance;
	/**
	 * The enemy's energy
	 */
	public double energy;
	/**
	 * The enemy's heading in radians
	 */
	public double heading;
	/**
	 * The enemy's velocity
	 */
	public double velocity;
	/**
	 * The enemy's location's x value
	 */
	public double x;
	/**
	 * The enemy's location's y value
	 */
	public double y;
	/**
	 * Whether or not the enemy is still alive
	 */
	public boolean isDead;
	/**
	 * The enemy's name
	 */
	public String name;

	/**
	 * Default constructor. Resets all information.
	 */
	public Enemy() {
		reset();
		isDead = false;
		name = "";
	}

	/**
	 * Constructor using a ScannedRobotEvent and an AdvancedRobot. Resets all
	 * information then updates it from the event and the AdvancedRobot.
	 * 
	 * @param e ScannedRobotEvent
	 * @param r AdvancedRobot that's calling this
	 */
	public Enemy(robocode.ScannedRobotEvent e, robocode.AdvancedRobot r) {
		reset();
		update(e, r);
	}

	/**
	 * Updates the information from a ScannedRobotEvent and an AdvancedRobot
	 * 
	 * @param e ScannedRobotEvent
	 * @param r AdvancedRobot that's calling this
	 */
	public void update(robocode.ScannedRobotEvent e, robocode.AdvancedRobot r) {
		bearing = e.getBearingRadians();
		distance = e.getDistance();
		energy = e.getEnergy();
		heading = e.getHeadingRadians();
		velocity = e.getVelocity();
		name = e.getName();
		double absBearing = r.getHeadingRadians() + e.getBearingRadians();
		x = r.getX() + Math.sin(absBearing) * e.getDistance();
		y = r.getY() + Math.cos(absBearing) * e.getDistance();
		isDead = false;
	}

	/**
	 * Resets all information except the name.
	 */
	public void reset() {
		bearing = 0.0;
		distance = Double.POSITIVE_INFINITY;
		energy = 0.0;
		heading = 0.0;
		velocity = 0.0;
		x = Double.NaN;
		y = Double.NaN;
	}

	/**
	 * Figures out if this enemy has no information.
	 * 
	 * @return true if there's no information, false if there is
	 */
	public boolean none() {
		return name.length() == 0;
	}

	/**
	 * Sets the enemy as dead.
	 */
	public void isNowDead() {
		isDead = true;
		reset();
	}
}