package lancel.components;
import java.awt.geom.*;
import java.util.*;
import robocode.*;
import robocode.util.Utils;
import static java.lang.Math.*; // By using this we can omit the 'Math' prefix from math function 
                                // calls.

/**************************************************************************************************
 Lynx - A duelist, based on Dynamic Clustering (DC) and WaveSurfing
 Author: lancel
 Current Version: 1.09
 Date: 6/12/2010
 License: Open Source under the RoboWiki Public Code License RWPCL (http://robowiki.net/wiki/RWPCL)
 Robocode Version: 1.6.1.4 (because of RoboRumble)
 Techniques this bot is based on:
    Gun: a k-nearest neighbors search with a kd-tree using a simple kernel density estimation 
         (dynamic clustering)
    Movement: basic WaveSurfing using the closest wave as a surfing wave
 CREDITS: Much of the ideas and code is based on various sources. A few to mention:
    wiki.BasicGFSurfer by Voidius, Pez and Bayen: initial structure for Lynx and the implementation
        of WaveSurfing
    Komarious by Voidius: the segmentation for WaveSurfing
    ABC: inventor of WaveSurfing
    Rednaxela: the implementation of the kd-tree
    rozu: precise prediction used in the Lynx's movement system
    DrussGT by Skilgannon: normalizing and weighting "features" that describe states for the gun
        ("segmentation" in DC) 
    Raiko by Jamougha: choosing a bullet power
    PEZ: Simple Iterative Wall Smoothing
    Voidious and others: good tutorials for getting into robocoding
**************************************************************************************************/

public class Movement {
//-------------------------------------------------------------------------------------------------
	final static int BINS = 47;
    final double WALL_STICK = 160;
    double oppSurroundingCircleRadius = 400.0; // WallSmoothing is done also for circle that
                                               // surrounds the opponent to keep distance.
  	double surfStats[][][] = new double[4][5][BINS];
    State state;
    double goAngle;
    public OppWave surfWave;
    public ArrayList<OppWave> oppWaves        = new ArrayList<OppWave>();
    public ArrayList<Integer> surfDirections  = new ArrayList<Integer>();
    public ArrayList<Double>  surfAbsBearings = new ArrayList<Double>();
    
    Rectangle2D.Double fieldRectangle = new Rectangle2D.Double(18, 18, 764, 564);
    double oppPrevEnergy   = 100;
    double oppPrevVelocity = 0;
    double oppPrevDistance = 0;
    double myPrevVelocity  = 0;
    Point2D.Double oppPrevLocation;
    Point2D.Double myPrevLocation;
    double advancingVelocityOpp;

    // For counting opponent's hit rate:
    int hitsByOpp         = 0;
    int oppBulletsFired   = 0;

    // For a graphical debugging in the DebugPaint class:
    Point2D.Double predictedPositionMinus = new Point2D.Double();
    Point2D.Double predictedPositionPlus  = new Point2D.Double();
    double debugMinusDanger = 0;
    double debugPlusDanger  = 0;

    public Movement (State state) {
    //---------------------------------------------------------------------------------------------
        this.state = state; // Attach game environment's state to the movement system.
    }

    public double onScannedRobot() {
    //---------------------------------------------------------------------------------------------
        double angleOffset = Utils.normalAbsoluteAngle(state.oppHeading - state.bearingToOpp);
        double oppLateralVelocity = state.oppVelocity * sin(angleOffset);
        surfDirections.add(0, new Integer((oppLateralVelocity >= 0) ? 1 : -1));
        surfAbsBearings.add(0, new Double(state.bearingToOpp + PI));
        double oppBulletPower = oppPrevEnergy - state.oppEnergy;
        double wallDamage = 0;

        if (abs(state.oppVelocity) == 0 && abs(oppPrevVelocity) > 2.0) {
            // The opponent hit the wall.
            wallDamage = max(0, abs(oppPrevVelocity) / 2 - 1);
        }

        double oppEnergyLoss = oppPrevEnergy - state.oppEnergy - wallDamage;
        if (oppEnergyLoss < 3.01 && oppEnergyLoss > 0.09 && surfDirections.size() > 2) {
            // The opponent has fired.
            oppBulletsFired++;
            OppWave ow          = new OppWave();
            ow.fireTime         = state.time - 1;
            ow.bulletVelocity   = Rules.getBulletSpeed(oppEnergyLoss);
            ow.distanceTraveled = Rules.getBulletSpeed(oppEnergyLoss);
            ow.direction        = (surfDirections.get(2)).intValue();
            ow.directAngle      = (surfAbsBearings.get(2)).doubleValue();
            ow.fireLocation     = (Point2D.Double)oppPrevLocation.clone();
            
            //-------------------------------------------------------------------------------------
            // CREDITS: Segmentation used here is from Komarious by Voidiuous
            //          (http://robowiki.net/?Komarious)
            //-------------------------------------------------------------------------------------            
            ow.waveGuessFactors = surfStats[(int)(min((ow.fireLocation.distance(myPrevLocation)+50)
                                  /200, 3))][(int)((abs(myPrevVelocity)+1)/2)];
            
            oppWaves.add(ow);
        }

        updateWaves();

        // Set "previous" variables
        oppPrevEnergy   = state.oppEnergy;
        oppPrevLocation = state.oppLocation;
        oppPrevVelocity = state.oppVelocity;
        oppPrevDistance = state.oppDistance;
        myPrevVelocity  = state.myVelocity;
        myPrevLocation  = state.myLocation;

        surfWave = getClosestSurfableWave();
        if (surfWave != null) {
            double dangerRight = debugPlusDanger = checkDanger(1);
            double dangerLeft  = debugMinusDanger = checkDanger(-1);

            goAngle = MyUtils.absoluteBearing(surfWave.fireLocation, state.myLocation);
            if (dangerLeft < dangerRight) {
                goAngle = doSmoothing(state.myLocation, goAngle - (PI/2), -1);
            } else {
                goAngle = doSmoothing(state.myLocation, goAngle + (PI/2), 1);
            }
        }

        return goAngle;
    }
    
    public static void setTurnTo(AdvancedRobot myBot, double goAngle) {
    //---------------------------------------------------------------------------------------------
    // aka setBackAsFront
        double angle =
            Utils.normalRelativeAngle(goAngle - myBot.getHeadingRadians());
        if (abs(angle) > (PI/2)) {
            if (angle < 0) {
                myBot.setTurnRightRadians(PI + angle);
            } else {
                myBot.setTurnLeftRadians(PI - angle);
            }
            myBot.setBack(100);
        } else {
            if (angle < 0) {
                myBot.setTurnLeftRadians(-1*angle);
           } else {
                myBot.setTurnRightRadians(angle);
           }
            myBot.setAhead(100);
        }
    }

    private double doSmoothing(Point2D.Double location, double angle, int orientation) {
    //---------------------------------------------------------------------------------------------
        // Before WallSmoothing another smoothing is done. A virtual circle is created at
        // opponents location and smoothing is done in respect to that circle to keep distance to
        // the opponent.

        final int MAX_TRIES = 350; // maximum iterations for smoothing
        double angleOffset = Utils.normalAbsoluteAngle(state.oppHeading - state.bearingToOpp);
        advancingVelocityOpp = -state.oppVelocity * cos(angleOffset);
        int diff;
        if (advancingVelocityOpp > 6.5)
            diff = 15;
        else
            diff = 30;
        oppSurroundingCircleRadius = min(400, state.myLocation.distance(state.oppLocation)+WALL_STICK-diff);
        Point2D.Double stickLoc = MyUtils.project(location, angle, WALL_STICK);        
        // Do smoothing for the circle surrounding the opponent to keep distance.
        int tries = 0; // limit tries
        if (location.distance(state.oppLocation) > (oppSurroundingCircleRadius-WALL_STICK)) {
            Ellipse2D.Double oppEllipse =
                new Ellipse2D.Double(state.oppLocation.x - oppSurroundingCircleRadius,
                state.oppLocation.y - oppSurroundingCircleRadius, oppSurroundingCircleRadius*2,
                oppSurroundingCircleRadius*2);
            tries = 0; // limit tries
            while (oppEllipse.contains(stickLoc.x, stickLoc.y) && tries < MAX_TRIES) {
                angle += orientation * 0.1;
                stickLoc = MyUtils.project(location, angle, WALL_STICK);
                tries++;
            }
        }
        
        //-------------------------------------------------------------------------------------
        // CREDITS: Simple Iterative Wall Smoothing (by PEZ) 
        //          (http://robowiki.net/wiki/Wall_Smoothing/Implementations)
        //------------------------------------------------------------------------------------- 
        stickLoc = MyUtils.project(location, angle, WALL_STICK);
        while (!fieldRectangle.contains(stickLoc.x, stickLoc.y)) {
            angle += orientation * 0.1;
            stickLoc = MyUtils.project(location, angle, WALL_STICK);
            tries++;
        }

        return angle;
    }

    private void logHit(OppWave ow, Point2D.Double targetLocation) {
    //---------------------------------------------------------------------------------------------
    // Given the OppWave that the bullet was on, and the point where we
    // were hit, update our stat array to reflect the danger in that area.
        int index = MyUtils.getFactorIndex(ow, targetLocation, BINS);
        for (int x = 0; x < BINS; x++) {
            double newValue = 1.0 / (pow(index - x, 2) + 1);
            ow.waveGuessFactors[x] = 0.6 * ow.waveGuessFactors[x] + 0.4 * newValue;
        }
    }

    public void onHitByBullet(HitByBulletEvent e) {
    //---------------------------------------------------------------------------------------------
        hitsByOpp++;
        // If the oppWaves collection is empty, we must have missed the
        // detection of this wave somehow.
        if (!oppWaves.isEmpty()) {
            Point2D.Double hitBulletLocation = new Point2D.Double(
                e.getBullet().getX(), e.getBullet().getY());
            OppWave hitWave = null;

            // look through the oppWaves, and find one that could've hit us.
            for (int x = 0; x < oppWaves.size(); x++) {
                OppWave ow = oppWaves.get(x);

                if (abs(ow.distanceTraveled -
                    state.myLocation.distance(ow.fireLocation)) < 50
                    && abs(Rules.getBulletSpeed(e.getBullet().getPower())
                        - ow.bulletVelocity) < 0.001) {
                    hitWave = ow;
                    break;
                }
            }

            if (hitWave != null) {
                logHit(hitWave, hitBulletLocation);

                // We can remove this wave now, of course.
                oppWaves.remove(oppWaves.lastIndexOf(hitWave));
            }
        }
    }

    public void onBulletHitBullet(BulletHitBulletEvent e) {
    //---------------------------------------------------------------------------------------------
        // If the oppWaves collection is empty, we must have missed the
        // detection of this wave somehow.
        if (!oppWaves.isEmpty()) {
            Point2D.Double hitBulletLocation = new Point2D.Double(
                e.getBullet().getX(), e.getBullet().getY());
            OppWave hitWave = null;

            // look through the OppWaves, and find one that could've hit bullet.
            for (int x = 0; x < oppWaves.size(); x++) {
                OppWave ow = oppWaves.get(x);
                if (abs(ow.distanceTraveled - hitBulletLocation.distance(ow.fireLocation)) < 20) {
                    hitWave = ow;
                    break;
                }
            }

            if (hitWave != null) {
                logHit(hitWave, hitBulletLocation);

                // We can remove this wave now, of course.
                oppWaves.remove(oppWaves.lastIndexOf(hitWave));
            }
        }
    }

    private double checkDanger(int direction) {
    //---------------------------------------------------------------------------------------------
    // CREDITS: mini sized predictor from Apollon, by rozu (http://robowiki.net?Apollon)
    //---------------------------------------------------------------------------------------------
        Point2D.Double predictedPosition = (Point2D.Double)state.myLocation.clone();
        double predictedHeading = state.myHeading;
        double predictedVelocity = state.myVelocity;
        double maxTurning;
        double moveAngle;
        double moveDir;
        double lastPredictedDistance;

        // - actual distance traveled so far needs +bullet_velocity,
        //    because it's detected one after being fired
        // - another +bullet_velocity b/c bullet advances before collisions
        // ...So start the counter at 2.
        // - another +bullet_velocity to approximate a bot's half width
        // ...So start the counter at 3.
        int counter = 3;

        do {
            moveDir = 1;
            moveAngle = doSmoothing(predictedPosition, surfWave.absoluteBearing(
                        predictedPosition) + (direction * (PI/2)), direction)
                        - predictedHeading;
            if (cos(moveAngle) < 0) {
                moveAngle += PI;
                moveDir = -1;
            }

            // calculate the new predicted position
            predictedPosition = MyUtils.project(predictedPosition,
                predictedHeading = Utils.normalRelativeAngle(predictedHeading +
                    MyUtils.limit(-(maxTurning = Rules.getTurnRateRadians(abs(predictedVelocity))),
                        Utils.normalRelativeAngle(moveAngle), maxTurning)),
                    (predictedVelocity = MyUtils.limit(-8,
                    predictedVelocity + (predictedVelocity * moveDir < 0 ? 2*moveDir : moveDir),
                    8)));

            if (direction == -1)
                predictedPositionMinus = predictedPosition;
            else if (direction == 1)
                predictedPositionPlus = predictedPosition;

        } while (
                (lastPredictedDistance = surfWave.distanceToPoint(predictedPosition)) >=
                surfWave.distanceTraveled + ((++counter) * surfWave.bulletVelocity));

        int index;
        //return (surfStats[index = MyUtils.getFactorIndex(surfWave, predictedPosition, BINS)]);
        
        return (surfWave.waveGuessFactors[index = MyUtils.getFactorIndex(surfWave, predictedPosition, BINS)]);
            //+ .01 / (Math.abs(index - GF_ZERO) + 1))
            /// Math.pow(Math.min(_enemyLocation.distance(predictedPosition), lastPredictedDistance), 4);
    }

    private void updateWaves() {
    //---------------------------------------------------------------------------------------------
        for (int x = 0; x < oppWaves.size(); x++) {
            OppWave ow = oppWaves.get(x);

            ow.distanceTraveled = (state.time - ow.fireTime) * ow.bulletVelocity;
            if (ow.distanceTraveled >
                state.myLocation.distance(ow.fireLocation) + 50) {
                oppWaves.remove(x);
                x--;
            }
        }
    }

    private OppWave getClosestSurfableWave() {
    //---------------------------------------------------------------------------------------------
        double closestDistance = 50000; // just a some very big number
        OppWave closestSurfWave = null;

        for (OppWave ow : oppWaves) {
            double distance = state.myLocation.distance(ow.fireLocation) - ow.distanceTraveled;

            if (distance > ow.bulletVelocity && distance < closestDistance) {
                closestSurfWave = ow;
                closestDistance = distance;
            }
        }
        return closestSurfWave;
    }

    public class OppWave {
    //---------------------------------------------------------------------------------------------
        double bulletVelocity;
        double directAngle;
        double distanceTraveled;
        int    direction;
        long   fireTime;
        Point2D.Double fireLocation;
        double[] waveGuessFactors;

        public double absoluteBearing(Point2D.Double target) {
        //-----------------------------------------------------------------------------------------
            return atan2(target.x - fireLocation.x, target.y - fireLocation.y);
        }

        public double distanceToPoint(Point2D.Double p) {
        //-----------------------------------------------------------------------------------------
            return fireLocation.distance(p);
        }
    }
}
