package ags.rougedc.robots;

import java.util.Map;

import ags.rougedc.base.Rules;
import ags.utils.points.*;

import robocode.Event;
import robocode.ScannedRobotEvent;
import robocode.RobotDeathEvent;
import robocode.BulletHitEvent;
import robocode.HitByBulletEvent;
import robocode.HitRobotEvent;

/**
 * Representation of an enemy robot
 * 
 * @author Alexander Schultz
 */
public class EnemyRobot implements VirtualRobot, java.lang.Cloneable {
    private final Map<String, Integer> enemyinfo;
    
    private final String name;
    private final StatusRobot status;
    private final Rules rules;
    
    private ScannedRobotEvent lastScan;
    private AbsolutePoint location;
    private AbsolutePoint lastLocation;
    private RelativePoint relativeLocation;
    private RelativePoint velocity;
    private double acceleration;
    private double angularVelocity;
    private double energyChange;
    private double explainedEnergyChange;
    private double firepower;
    private double gunheat; // Double.NaN when unknown
    private long coolticks;
    private boolean alive;
    
    private int timeSinceAccel;
    private int timeSinceDecel;
    
    private long scanage;
    
    // Cloning constructor
    private EnemyRobot(EnemyRobot source) {
        enemyinfo = source.enemyinfo;
        
        name = source.name;
        status = source.status;
        rules = source.rules;
        
        lastScan = source.lastScan; // Never changes so we don't need to worry about cloning
        if (source.location != null)
            location = (AbsolutePoint)source.location.clone();
        if (source.lastLocation != null)
            lastLocation = (AbsolutePoint)source.lastLocation.clone();
        if (source.relativeLocation != null)
            relativeLocation = (RelativePoint)source.relativeLocation.clone();
        if (source.velocity != null)
            velocity = (RelativePoint)source.velocity.clone();
        acceleration = source.acceleration;
        angularVelocity = source.angularVelocity;
        energyChange = source.energyChange;
        explainedEnergyChange = source.explainedEnergyChange;
        firepower = source.firepower;
        gunheat = source.gunheat;
        alive = source.alive;
    }
    
    @Override
    public EnemyRobot clone() {
        return new EnemyRobot(this);
    }
    
    // Main constructor
    public EnemyRobot(Rules rules, StatusRobot status, String name, Map<String, Integer> enemyinfo) {
        this.enemyinfo = enemyinfo;
        this.name = name;
        this.rules = rules;
        this.status = status;
        if (3.0 >= ((status.getTime()-1) * rules.GUN_COOLING_RATE))
            gunheat = 3.0 - ((status.getTime()-1) * rules.GUN_COOLING_RATE);
        else
            gunheat = Double.NaN;
        alive = true;
    }
    
    public String getName() {
        return name;
    }
    
    public int getHitCount() {
        if (!enemyinfo.containsKey(name))
            return 0;
        return enemyinfo.get(name);
    }
    
    public void handleEvents(Iterable<Event> i) {
        boolean collision = false;
        
        // HACK: Add the old energy to the explained change, and later subtract it.
        // This simulates the exact rounding of the Robocode engine and avoids small
        // differences in rounding!
        double oldenergy = 0;
        if (lastScan != null)
            oldenergy = lastScan.getEnergy(); 
        explainedEnergyChange = oldenergy;
        
        if (location != null)
            lastLocation = (AbsolutePoint)location.clone();
        
        for (Event e : i) {
            if (e instanceof ScannedRobotEvent) { // IF IT'S A SCAN
                ScannedRobotEvent scan = (ScannedRobotEvent)e;
                //If it's not ours, toss it
                if (scan.getName() != name)
                    continue;
                
                relativeLocation = RelativePoint.fromDM(scan.getBearingRadians()+status.getHeading(), scan.getDistance());
                location = status.getLocation().addRelativePoint(relativeLocation);
                velocity = RelativePoint.fromDM(scan.getHeadingRadians(), scan.getVelocity());
                
                // If we had a scan last tick, we can get other things too
                if (lastScan != null) {
                    acceleration = scan.getVelocity() - lastScan.getVelocity();
                    angularVelocity = scan.getHeadingRadians() - lastScan.getHeadingRadians();
                    energyChange = scan.getEnergy() - lastScan.getEnergy();
                    
                    if (Math.abs(scan.getVelocity()) > Math.abs(lastScan.getVelocity()))
                        timeSinceAccel = 0;
                    else
                        timeSinceAccel++;
                    
                    if (Math.abs(scan.getVelocity()) < Math.abs(lastScan.getVelocity()))
                        timeSinceDecel = 0;
                    else
                        timeSinceDecel++;
                }
                
                lastScan = scan;
                scanage = 1;
            } else if (e instanceof RobotDeathEvent) { // IF IT'S A ROBOT DYING
                // If it's not us dying, ignore it
                if (name != ((RobotDeathEvent)e).getName())
                    continue;
                alive = false;
            } else if (e instanceof BulletHitEvent) { // IF IT'S A ROBOT BEING HIT
                BulletHitEvent hit = (BulletHitEvent)e;
                // If someone else is being hit, ignore it
                if (name != hit.getName())
                    continue;
                
                // Include the bullet hit in the explained energy change
                explainedEnergyChange -= rules.getBulletDamage(hit.getBullet().getPower());  // Power to damage formula 
            } else if (e instanceof HitRobotEvent) { // A COLLISON!
                HitRobotEvent hit = (HitRobotEvent)e;
                // If someone else is being hit, ignore it
                if (name != hit.getName())
                    continue;
                
                // Collision damage
                explainedEnergyChange -= rules.ROBOT_HIT_DAMAGE;
                collision = true;
            } else if (e instanceof HitByBulletEvent) { // A HIT!
                HitByBulletEvent hit = (HitByBulletEvent)e;
                // If someone else is being hit, ignore it
                if (name != hit.getName())
                    continue;
                
                // The vampire-effect!
                explainedEnergyChange += 3 * hit.getPower();
            }
        }
        
        if (gunheat == 0)
            coolticks++;
        gunheat = Math.max(0, gunheat - rules.GUN_COOLING_RATE);
        
        // No scan! Invalidate data.
        if (scanage > 1) {
            lastScan = null;
            location = null;
            relativeLocation = null;
            velocity = null;
            acceleration = 0;
            angularVelocity = 0;
            energyChange = 0;
            gunheat = Double.NaN;
            timeSinceAccel = 0;
            timeSinceDecel = 0;
        }
        
        // Handle wall collisions!
        if (scanage <= 1 && !collision && lastScan.getVelocity() == 0 && Math.abs(acceleration) > 2) {
            // Well there wasn't a collision event but we suddenly stopped
            // Must have been a wall collision!
            double hitVelocity = Math.min(Math.abs(velocity.magnitude+acceleration), 8.0);
            explainedEnergyChange -= rules.getWallHitDamage(hitVelocity); 
        }
        
        // HACK: Take the old energy away again, this makes the rounding work more like the robocode engine
        explainedEnergyChange -= oldenergy;
        
        // Get the power we apparently fired with
        firepower = explainedEnergyChange - energyChange;
        
        if (firepower < 0.01) // 0.01 is the minimum energy a robot could ever have and still be living
            firepower = 0;
        else if (firepower > rules.MAX_BULLET_POWER)
            firepower = rules.MAX_BULLET_POWER;
    }
    
    public int getTimeSinceAccel() {
        return timeSinceAccel;
    }
    public int getTimeSinceDecel() {
        return timeSinceDecel;
    }
    public double getEnergy() {
        return lastScan.getEnergy();
    }
    public double getEnergyChange() {
        return energyChange;
    }
    public double getExplainedEnergyChange() {
        return explainedEnergyChange;
    }
    public double getFirePower() {
        return firepower;
    }
    public AbsolutePoint getLocation() {
        return location;
    }
    public AbsolutePoint getLastLocation() {
        return lastLocation;
    }
    public RelativePoint getRelativeLocation() {
        return relativeLocation;
    }
    public RelativePoint getVelocity() {
        return velocity;
    }
    public double getAcceleration() {
        return acceleration;
    }
    public double getAngularVelocity() {
        return angularVelocity;
    }
    public double getGunheat() {
        return gunheat;
    }
    public boolean isAlive() {
        return alive;
    }
    public long getScanAge() {
        return scanage;
    }
    public boolean isScanned() {
        return (lastScan != null);
    }
    
    // A bullet was fired a few ticks ago!
    public void firedBullet(double power, int ticksago) {
        gunheat = rules.getGunHeat(power) - ticksago * rules.GUN_COOLING_RATE;
        coolticks = 0;
    }
    // Return how many ticks the gun has been cool for, 
    public long getCoolticks() {
        return coolticks;
    }
}
