package ar.horizon.stats;

import static ar.horizon.util.Util.*;

import java.util.*;

/**
 * @author Aaron Rotenberg
 */
public class DynamicClusteringStatBuffer<K, V> implements StatBuffer<K, V> {
	private static final int MAX_DEPTH = 500;

	private final DCStatBufferParams<K> locationConverter;
	private int size = 0;
	private final Node rootNode = new Node(MAX_DEPTH);
	private final double[] lowerBounds;
	private final double[] upperBounds;

	public DynamicClusteringStatBuffer(DCStatBufferParams<K> params) {
		this.locationConverter = params;

		this.lowerBounds = new double[params.getDimensions()];
		this.upperBounds = new double[params.getDimensions()];
	}

	public void add(Pair<K, V> data) {
		resetBounds();
		rootNode.add(data);
	}

	public Collection<Pair<K, V>> getNearestNeighbors(K key) {
		resetBounds();
		Cluster cluster =
				new Cluster(locationConverter.getLocation(key),
						locationConverter.getMaxClusterSize(this.size));
		rootNode.addToCluster(cluster);
		return cluster.getContents();
	}

	private void resetBounds() {
		for (int i = 0; i < locationConverter.getDimensions(); i++) {
			lowerBounds[i] = 0.0;
			upperBounds[i] = 1.0;
		}
	}

	private static class DistanceCachingPair<K, V> implements
			Comparable<DistanceCachingPair<K, V>> {
		private final Pair<K, V> pair;
		private final double distance;

		public DistanceCachingPair(Pair<K, V> pair, double distance) {
			this.pair = pair;
			this.distance = distance;
		}

		public Pair<K, V> getPair() {
			return pair;
		}

		public double getDistance() {
			return distance;
		}

		public int compareTo(DistanceCachingPair<K, V> o) {
			if (distance > o.distance) {
				return -1;
			} else if (distance < o.distance) {
				return 1;
			} else {
				return 0;
			}
		}
	}

	private class Cluster {
		private final double[] center;
		private final int maxScansInCluster;
		private final PriorityQueue<DistanceCachingPair<K, V>> points =
				new PriorityQueue<DistanceCachingPair<K, V>>();

		public Cluster(double[] center, int maxScansInCluster) {
			this.center = center;
			this.maxScansInCluster = maxScansInCluster;
		}

		public double[] getCenter() {
			return center;
		}

		public void considerAdding(Pair<K, V> point) {
			final double distance =
					locationConverter.getDistance(center,
							locationConverter.getLocation(point.getKey()));
			final DistanceCachingPair<K, V> pointWithDistance =
					new DistanceCachingPair<K, V>(point, distance);

			// Add the point to the cluster if it fits, or if it is better than
			// the current worst element.
			if (points.size() < maxScansInCluster) {
				points.add(pointWithDistance);
			} else if (distance < points.peek().getDistance()) {
				points.remove();
				points.add(pointWithDistance);
			}
		}

		public boolean isViable() {
			if (points.size() < maxScansInCluster) {
				return true;
			} else {
				double[] testPoint =
						new double[locationConverter.getDimensions()];
				for (int i = 0; i < locationConverter.getDimensions(); i++) {
					testPoint[i] =
							limit(lowerBounds[i], center[i], upperBounds[i]);
				}

				return points.peek().getDistance() > locationConverter.getDistance(
						testPoint, center);
			}
		}

		public Collection<Pair<K, V>> getContents() {
			Collection<Pair<K, V>> contents = new LinkedList<Pair<K, V>>();
			for (DistanceCachingPair<K, V> pair : points) {
				contents.add(pair.getPair());
			}
			return contents;
		}
	}

	private class Node {
		private static final int MAX_DENSITY = 10;

		private final int maxDepth;
		private boolean internal = false;
		private Node leftChild = null;
		private Node rightChild = null;
		private Queue<Pair<K, V>> data = new LinkedList<Pair<K, V>>();

		public Node(int maxDepth) {
			this.maxDepth = maxDepth;
		}

		public void add(Pair<K, V> point) {
			int dimension = maxDepth % locationConverter.getDimensions();
			double median =
					(lowerBounds[dimension] + upperBounds[dimension]) / 2;

			if (internal) {
				double[] location =
						locationConverter.getLocation(point.getKey());
				if (location[dimension] < median) {
					upperBounds[dimension] = median;
					if (leftChild == null) {
						leftChild = new Node(maxDepth - 1);
					}

					leftChild.add(point);
				} else {
					lowerBounds[dimension] = median;
					if (rightChild == null) {
						rightChild = new Node(maxDepth - 1);
					}

					rightChild.add(point);
				}
			} else {
				if (data.size() < MAX_DENSITY) {
					data.add(point);
					size++;
				} else if (maxDepth == 1) {
					data.add(point);
					data.poll();
				} else {
					for (Pair<K, V> existingPoint : data) {
						if (locationConverter.getLocation(existingPoint.getKey())[dimension] < median) {
							if (leftChild == null) {
								leftChild = new Node(maxDepth - 1);
							}

							leftChild.data.add(existingPoint);
						} else {
							if (rightChild == null) {
								rightChild = new Node(maxDepth - 1);
							}

							rightChild.data.add(existingPoint);
						}
					}

					data = null;
					internal = true;
					add(point);
				}
			}

		}

		public void addToCluster(Cluster addTo) {
			if (internal) {
				int dimension = maxDepth % locationConverter.getDimensions();
				double median =
						(lowerBounds[dimension] + upperBounds[dimension]) / 2;
				boolean leftFirst = addTo.getCenter()[dimension] < median;
				addChildToCluster(addTo, dimension, median, leftFirst);
				addChildToCluster(addTo, dimension, median, !leftFirst);
			} else {
				for (Pair<K, V> point : data) {
					addTo.considerAdding(point);
				}
			}
		}

		private void addChildToCluster(Cluster addTo, int dimension,
				double median, boolean left) {
			if (left) {
				if (leftChild != null) {
					double original = upperBounds[dimension];
					upperBounds[dimension] = median;
					if (addTo.isViable()) {
						leftChild.addToCluster(addTo);
					}
					upperBounds[dimension] = original;
				}
			} else {
				if (rightChild != null) {
					double original = lowerBounds[dimension];
					lowerBounds[dimension] = median;
					if (addTo.isViable()) {
						rightChild.addToCluster(addTo);
					}
					lowerBounds[dimension] = original;
				}
			}
		}
	}
}
