package tjk.defense;

import robocode.*;
import robocode.util.Utils;
import robocode.Rules;
import tjk.universe.*;
import tjk.universe.BracketHist.HistType;
import tjk.utils.FastTrig;
import tjk.utils.HitTracker;
import tjk.utils.tjkUtil;
import java.awt.geom.Point2D;

import java.util.ArrayList;
// Testing MovementPredictor
import tjk.utils.MovementPredictor;
import tjk.utils.MovementPredictor.*;

// paint debug
import java.awt.Color;
import java.awt.Graphics2D;

/**
 * Wheels - a class by tkiesel
 * Copyright (c) 2012 Tom Kiesel (Tkiesel @ Robowiki)
 * 
 * This software is provided 'as-is', without any express or implied warranty. In no event will the authors be held liable for any damages arising from the use of this software.
 * 
 * Permission is granted to anyone to use this software for any purpose, including commercial applications, and to alter it and redistribute it freely, subject to the following restrictions:
 * 
 *     1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. 
 *        If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
 * 
 *     2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
 * 
 *     3. This notice may not be removed or altered from any source distribution.
 * 
 *  In addition:
 *
 *     4. Use of this software in any programming competition, including the RoboRumble, without the written permission of the author is forbidden.
 * 
 */

public class Wheels
{
	
	public static final int REAL = 0;
	public static final int FLAT = 1;
	public static final int RAND = 2;
		
	public static final double DEFAULT_HALF_LIFE = 20.0;
	
	public static double HALF_LIFE = DEFAULT_HALF_LIFE;
	public static double[] HL_ARRAY;
	
	public final boolean surfTwoWaves = true;
	
	public boolean flat = false;
	
	AdvancedRobot robot = null;
	
	Universe universe = null;
	
	Bot enemy = null;
	
	Bot me = null;
	
	WaveStats realHits = null;
	
	WaveStats enemyStats = null;
	
	ArrayList<Wave> realWaves = null;
	
	ArrayList<Wave> hitByWaves = null;
	
	Point2D.Double enemyFuture = new Point2D.Double(0.0,0.0);
	
	Wave closestWave = null;
	
	Wave secondClosestWave = null;
	
	HitTracker hitTracker = null;
	
	MyOrbitDistanceController orbitControl = new MyOrbitDistanceController();
	
	public Wheels(AdvancedRobot ar, Universe u)
	{
		this.robot = ar;
		this.universe = u;
		this.realHits = new WaveStats();
		this.realHits.addStartNullHits();
		this.me = universe.getMe();
		this.enemy = universe.getEnemy();
		this.realWaves = new ArrayList<Wave>(7);
		this.hitByWaves = new ArrayList<Wave>(2);
		this.enemyStats = enemy.getStats();
		this.enemyStats.addStartNullHits();
		this.hitTracker = new HitTracker();
		Wheels.HL_ARRAY = tjkUtil.halfLifeArray(DEFAULT_HALF_LIFE);
	}

	public void onScannedRobot() 
	{
		enemyFuture = enemyProject();
		
		// Enemy fires waves.
		Wave firedWave = enemy.checkFireWave();
		if ( firedWave != null )
		{
			realWaves.add(firedWave);
			loadStats(firedWave);
		}
		else //if ( universe.getTime() % 3 == 0)
		{
			enemy.fireOldWave(false, 2.0);
		}
		
		// Wave cleanup
		Wave deleteMe = null;
		ArrayList<Wave> allWaves = enemy.getWaves();
		for ( Wave w : realWaves )
		{
			if ( !allWaves.contains(w)  )
			{
				deleteMe = w;
				break;
			}
		}
		if ( deleteMe != null )
		{
			realWaves.remove(deleteMe);
			hitTracker.addMiss();
		}
	
		// Wait for waves that hit me to pass before loading into stat buffer.
		if ( !hitByWaves.isEmpty() )
		{
			boolean reLoad = false;
			for ( int i = 0; i < hitByWaves.size(); i++ )
			{
				Wave w = hitByWaves.get(i);
				if ( w.getState() == Wave.DEAD )
				{
					realHits.addWave(w);
					hitByWaves.remove(w);
					i--;
					reLoad = true;
					hitTracker.addHit(w);
				}
			}
			if ( reLoad )
			{
				for ( Wave w : realWaves )
				{
					reLoadRealStats(w);
				}
			}
		}
		
		// Wave surfing...
		getClosestSurfableWave();
		
		// Rammer!
		if ( isRamming() )
		{
			antiRamMove();
			return;
		}
		
		if ( closestWave != null )
		{
			surf();
		}
		else
		{
			// No wave?  Sit still?
			//System.out.println("No Surfable wave");
			noWaveMove();
		}
		
	}

	public void surf()
	{
		loadStats(closestWave);
				
		PredictionStatus startPoint = newPredStat(me);
		WaveEndPredictionCondition ender = new WaveEndPredictionCondition(closestWave.getLocation(), closestWave.getRadius(), closestWave.getVelocity() );
		int current_clockwise = enemy.enemyCW() >= 0 ? 1 : -1;
		
		PredictionStatus stopPoint = MovementPredictor.predictWaveStopPoint(startPoint, ender, current_clockwise, closestWave.getLocation());
		PredictionStatus CWPoint = MovementPredictor.predictWaveSurfpoint(startPoint, ender, orbitControl, 1, closestWave.getLocation());
		PredictionStatus CCWPoint = MovementPredictor.predictWaveSurfpoint(startPoint, ender, orbitControl, -1, closestWave.getLocation());
		
		double dangerStop = getDanger(closestWave, stopPoint);
		double dangerCW = getDanger(closestWave, CWPoint);
		double dangerCCW = getDanger(closestWave, CCWPoint);
		
		// Surf second wave.
		if ( surfTwoWaves && secondClosestWave != null )
		{
			loadStats(secondClosestWave);
			
			//System.out.println("Surfing second wave!");
			
			if ( least(dangerCW,dangerCCW,dangerStop) )
			{
				dangerCW += minDangerOnWave(secondClosestWave, CWPoint, 1);
				double min_danger = dangerCW;
				if ( dangerStop < min_danger )
				{
					dangerStop += minDangerOnWave(secondClosestWave, stopPoint, current_clockwise);
					if ( dangerStop < min_danger)
					{
						min_danger = dangerStop;
					}
				}
				if ( dangerCCW < min_danger )
				{
					dangerCCW += minDangerOnWave(secondClosestWave, CCWPoint, -1);
				}
			}
			else if ( least(dangerStop,dangerCW,dangerCCW) )
			{
				dangerStop += minDangerOnWave(secondClosestWave, CWPoint, current_clockwise);
				double min_danger = dangerStop;
				if ( dangerCCW < min_danger )
				{
					dangerCCW += minDangerOnWave(secondClosestWave, CCWPoint, -1);
					if ( dangerCCW < min_danger)
					{
						min_danger = dangerCCW;
					}
				}
				if ( dangerCW < min_danger )
				{
					dangerCW += minDangerOnWave(secondClosestWave, CWPoint, 1);
				}
			}
			else 
			{
				dangerCCW += minDangerOnWave(secondClosestWave, CCWPoint, -1);
				double min_danger = dangerCCW;
				if ( dangerStop < min_danger )
				{
					dangerStop += minDangerOnWave(secondClosestWave, stopPoint, current_clockwise);
					if ( dangerStop < min_danger)
					{
						min_danger = dangerStop;
					}
				}
				if ( dangerCW < min_danger )
				{
					dangerCW += minDangerOnWave(secondClosestWave, CWPoint, 1);
				}
			}
		}
		
		//System.out.println("Dangers: " + dangerCW + ", " + dangerStop + ", " + dangerCCW);
		
		double goAngle = FastTrig.atan2(me.getX()-closestWave.getX(), me.getY()-closestWave.getY());
		double orbitRadius = me.getLocation().distance(closestWave.getLocation());
		
		if ( least(dangerCW,dangerCCW,dangerStop) )
		{
			goAngle += Math.PI/2.0 + orbitControl.angleAdjust(startPoint, orbitRadius);
			goAngle = MovementPredictor.fastSmooth(me.getLocation(), goAngle, 1, orbitRadius);
			setBackAsFront(goAngle,100.0);
			//System.out.println("Going CW");
		}
		else if ( least(dangerStop,dangerCW,dangerCCW) )
		{
			if ( surfTwoWaves && secondClosestWave != null && me.getV() == 0.0 && orbitRadius - closestWave.getRadius() < 70.0 )
			{
				// If we're (most likely) stopping on this wave, might as well get perpendicular to the next wave.
				goAngle = FastTrig.atan2(me.getX()-secondClosestWave.getX(), me.getY()-secondClosestWave.getY()) + ((double)current_clockwise)*Math.PI/2.0;
			}
			else
			{
				goAngle += ((double)current_clockwise)*Math.PI/2.0;
			}
			goAngle = MovementPredictor.fastSmooth(me.getLocation(), goAngle, current_clockwise, orbitRadius);
			setBackAsFront(goAngle,0.0);
			//System.out.println("Stopping");
		}
		else if ( least(dangerCCW,dangerCW,dangerStop) )
		{
			goAngle -= Math.PI/2.0 + orbitControl.angleAdjust(startPoint, orbitRadius);
			goAngle = MovementPredictor.fastSmooth(me.getLocation(), goAngle, -1, orbitRadius);
			setBackAsFront(goAngle,100.0);
			//System.out.println("Going CCW");
		}
		else
		{
			// Tie?  Still, move.
			if ( (Universe.getTime() / 100 ) % 2 == 0 )
			{
				goAngle += Math.PI/2.0 + orbitControl.angleAdjust(startPoint, orbitRadius);
				goAngle = MovementPredictor.fastSmooth(me.getLocation(), goAngle, 1, orbitRadius);
				setBackAsFront(goAngle,100.0);
				//System.out.println("Going CW");
			}
			else
			{
				goAngle -= Math.PI/2.0 + orbitControl.angleAdjust(startPoint, orbitRadius);
				goAngle = MovementPredictor.fastSmooth(me.getLocation(), goAngle, -1, orbitRadius);
				setBackAsFront(goAngle,100.0);
				//System.out.println("Going CCW");
			}
		}
		
	}

	public double minDangerOnWave(Wave w, PredictionStatus location, int clockwise)
	{
		WaveEndPredictionCondition ender = new WaveEndPredictionCondition(w.getLocation(), w.getRadius() + w.getVelocity()*location.time , w.getVelocity() );
		
		PredictionStatus stopPoint = MovementPredictor.predictWaveStopPoint(location, ender, clockwise, w.getLocation());
		PredictionStatus CWPoint = MovementPredictor.predictWaveSurfpoint(location, ender, orbitControl, 1, w.getLocation());
		PredictionStatus CCWPoint = MovementPredictor.predictWaveSurfpoint(location, ender, orbitControl, -1, w.getLocation());
		
		double dangerStop = getDanger(w, stopPoint);
		double dangerCW = getDanger(w, CWPoint);
		double dangerCCW = getDanger(w, CCWPoint);
		
		return Math.min(dangerStop,Math.min(dangerCW,dangerCCW));
	}

	public void antiRamMove()
	{
		// Enemy in 6 ticks. Very rough prediction.
		double futureEnemyX = enemy.getX() + 6.0*enemy.getV()*FastTrig.sin(enemy.getHeading());
		double futureEnemyY = enemy.getY() + 6.0*enemy.getV()*FastTrig.cos(enemy.getHeading());
		
		//If close to wall, just go directly from enemy.
		if ( me.getX() < 100 || me.getY() < 100 || me.getX() > Universe.bfW() - 100 || me.getY() > Universe.bfH() - 100 )
		{
			futureEnemyX = enemy.getX();
			futureEnemyY = enemy.getY();
		}
		
		double goAngle = FastTrig.atan2(me.getX()-futureEnemyX, me.getY()-futureEnemyY) + 0.25*((Universe.getTime()/50)%3);
		double orbitRadius = me.getLocation().distance(enemy.getLocation());
		
		double ram_wall_margin = 55.0;
		
		double CWgo = MovementPredictor.fastSmooth(me.getLocation(), goAngle, 1, orbitRadius, ram_wall_margin);
		double CCWgo = MovementPredictor.fastSmooth(me.getLocation(), goAngle, -1, orbitRadius, ram_wall_margin);
		
		double goingNow = Utils.normalAbsoluteAngle((me.getV()< 0.0 ? -1.0 : 1.0) * me.getHeading());
		
		if ( Math.abs(Utils.normalRelativeAngle(CWgo - goingNow)) < Math.abs(Utils.normalRelativeAngle(CCWgo - goingNow)) )
		{
			goAngle = CWgo;
		}
		else
		{
			goAngle = CCWgo;
		}
		
		setBackAsFront(goAngle,200.0);
		
	}

	public void noWaveMove()
	{
		double goAngle = FastTrig.atan2(enemy.getX()-me.getX(), enemy.getY()-me.getY())+FastTrig.HALF_PI;
		double dist = (Universe.getTime()%50 - 25)*(Universe.getRoundNum()%2==0?1.0:-1.0);
		setBackAsFront(goAngle,dist);
	}

	public void onRoundEnded(RoundEndedEvent e)
	{
		System.out.println("Enemy total hit rate: " + 100*hitTracker.totalHitRate() + "%");
		System.out.println("Enemy rolling hit rate, long: " + 100*hitTracker.rollingHitRateLong() + "%");
		System.out.println("Enemy rolling hit rate, medium: " + 100*hitTracker.rollingHitRateMedium() + "%");
		System.out.println("Enemy rolling hit rate, short: " + 100*hitTracker.rollingHitRateShort() + "%");
		if ( flat )
		{
			System.out.println("Flattener on");
			System.out.println("Flattener half life = " + HALF_LIFE + " rounds");
		}
		else
		{
			System.out.println("Flattener off");
		}
	}

	public void onBulletHitBullet(BulletHitBulletEvent e)
	{
		Point2D.Double hitLocation = new Point2D.Double(e.getBullet().getX(),e.getBullet().getY());
		double minDistance = Double.POSITIVE_INFINITY;
		Wave bulletHitWave = null;
		for ( Wave w : realWaves )
		{
			if ( w.real )
			{
				double testD = Math.abs( w.getRadius() - hitLocation.distance(w.getLocation()) );
				if ( testD < minDistance )
				{
					minDistance = testD;
					bulletHitWave = w;
				}
			}
		}
		if (bulletHitWave != null)
		{
			//System.out.println("Logging bullet hit bullet.");
			realHits.addBulletHit(bulletHitWave, hitLocation);
			//enemyStats.addBulletHit(bulletHitWave, hitLocation);
			realWaves.remove(bulletHitWave);
			
			for ( Wave w : realWaves )
			{
				reLoadRealStats(w);
			}
		}
	}

	public void onHitByBullet(HitByBulletEvent e)
	{
		
		// Starting out with Wave Surfing Tutorial code here. :)
		if ( realWaves.isEmpty() )
		{
			return;
		}
		Wave hitWave = null;
		
		for ( int i = 0; i < realWaves.size(); i++ ) 
		{
			Wave checkWave = realWaves.get(i);
			
			if ( Math.abs(checkWave.getRadius() - me.getLocation().distance(checkWave.getLocation())) < 50 
				&& Math.abs(checkWave.getPower() - e.getBullet().getPower()) < 0.001 )
			{
				hitWave = checkWave;
				break;
			}
			
		}
		if ( hitWave != null )
		{
			hitByWaves.add(hitWave);
			realWaves.remove(hitWave);
		}
		
	}

	public double getDanger(Wave w, PredictionStatus location)
	{
		double hitrateMin = 100.0*Math.min(Math.min(hitTracker.rollingHitRateMedium() , hitTracker.rollingHitRateLong()),hitTracker.rollingHitRateShort());
		
		double hitrateMax = 100.0*Math.max(Math.max(hitTracker.rollingHitRateMedium() , hitTracker.rollingHitRateLong()),hitTracker.rollingHitRateShort());
		
		if ( !flat && Universe.getRoundNum() > 2 && hitrateMin > 12.0 )
		{
			flat = true;
			//System.out.println("Flattener on.");
		}
		else if ( flat && hitrateMax < 12.0 )
		{
			flat = false;
			//System.out.println("Flattener off.");
		}
				
		double flattener = 0.0;
		double real = 1.0;
		
		if ( !w.real )
		{
			flattener = 1.0;
			real = 0.0;
		}
		else if ( flat )
		{
			//double hR = hitrate - 11.0;
			//flattener = 10.0 * hR / (hR+2.0);
			flattener = 3.5;
			//real = 1.0 - flattener;
		}
				
		// RealHits data.
		double answer = 0.0;
		if ( Double.compare(real,0.0) > 0.0 )
		{
			answer += real*getStatArea(w, location, REAL);
		}
		// Full wave data. (FLATTENER)
		if ( Double.compare(flattener,0.0) > 0.0 )
		{
			answer += flattener*getStatArea(w, location, FLAT);
		}
	
		// Dive protection.	
		double bearingToWave = Math.abs(Utils.normalRelativeAngle(FastTrig.atan2(enemy.getX()-location.getX(),enemy.getY()-location.getY()) - location.heading) );
		if ( bearingToWave <= 0.30*FastTrig.PI || bearingToWave >= 0.70*FastTrig.PI )
		{
			if ( bearingToWave <= 0.15*FastTrig.PI || bearingToWave >= 0.85*FastTrig.PI )
			{
				answer *= 1.080;
			}
			else
			{
				answer *= 1.20;
			}
		}
				
		//Time factor
		
		if ( surfTwoWaves && secondClosestWave != null )
		{
			answer *= Rules.getBulletDamage(w.getPower()) / Math.abs(w.getLocation().distanceSq(me.getLocation()) - w.getRadius()*w.getRadius());
		}
		
		return answer;
	}
	
	public void reLoadRealStats(Wave w)
	{
		if ( w == null )
		{
			return;
		}
		if ( realHits.getSize() > 0 )
		{
			ArrayList<GFBracket> kNN = realHits.getNearestGFBrackets(w, closestK(realHits), false).get(0);
			BracketHist realBH = new BracketHist(HistType.MOVEMENT, kNN, 12.0, 1.0, false, BracketHist.nullB);
			w.setRealBracketHist(realBH);
		}
	}

	public void loadStats(Wave w)
	{
		
		//TODO: adaptive halflife.
		
		if ( w == null )
		{
			return;
		}
		
		if ( w.getRealBracketHist() == null )
		{
			if ( realHits.getSize() > 0 )
			{
				ArrayList<GFBracket> kNN = realHits.getNearestGFBrackets(w, closestK(realHits), false).get(0);
				BracketHist realBH = new BracketHist(HistType.MOVEMENT, kNN, 12.0, 1.0, false, BracketHist.nullB);
				w.setRealBracketHist(realBH);
			}
		}
		
		if ( w.getMainBracketHist() == null )
		{
			if ( enemyStats.getSize() > 0 )
			{
				ArrayList<GFBracket> kNN = enemyStats.getNearestGFBrackets(w, closestK(enemyStats), false).get(0);
				BracketHist flatBH = new BracketHist(HistType.MOVEMENT, kNN, 12.0, 1.0, false, BracketHist.nullB);
				w.setMainBracketHist(flatBH);
			}
		}
		
	}
	
			
	public void paint(Graphics2D g)
	{
		if ( closestWave != null )
		{
			PredictionStatus startPoint = newPredStat(me);
			WaveEndPredictionCondition ender = new WaveEndPredictionCondition(closestWave.getLocation(), closestWave.getRadius(), closestWave.getVelocity() );
			int current_clockwise = enemy.enemyCW() >= 0 ? 1 : -1;
			
			PredictionStatus stopPoint = MovementPredictor.predictWaveStopPoint(startPoint, ender, current_clockwise, closestWave.getLocation());
			PredictionStatus CWPoint = MovementPredictor.predictWaveSurfpoint(startPoint, ender, 1, closestWave.getLocation());
			PredictionStatus CCWPoint = MovementPredictor.predictWaveSurfpoint(startPoint, ender, -1, closestWave.getLocation());
			
			g.setColor(Color.RED);
			g.fillRect((int)stopPoint.getX()-10,(int)stopPoint.getY()-10,20,20);
			g.setColor(Color.GREEN);
			g.fillRect((int)CWPoint.getX()-10,(int)CWPoint.getY()-10,20,20);
			g.setColor(Color.ORANGE);
			g.fillRect((int)CCWPoint.getX()-10,(int)CCWPoint.getY()-10,20,20);						
		}
	
		Point2D.Double enemyFuture = enemyProject();
		
		g.setColor(Color.WHITE);
		g.fillRect((int)enemyFuture.getX()-10,(int)enemyFuture.getY()-10,20,20);
	}
	
	private int closestK(WaveStats w)
	{
		return (int)Math.round(Math.min(Math.max(2,w.getSize()/20),150));
	}
	
	public void getClosestSurfableWave()
	{
		if ( closestWave != null && closestWave.getState() == Wave.DEAD )
		{
			this.closestWave = null;
		}
		if ( realWaves == null || realWaves.isEmpty() )
		{
			this.secondClosestWave = null;
			this.closestWave = null;
			
			return;
		}
		
		double closestTime = Double.POSITIVE_INFINITY;
		Wave newClosestWave = null;
		double secondClosestTime = Double.POSITIVE_INFINITY;
		Wave newSecondClosestWave = null;
		for ( Wave w : realWaves )
		{
			if ( w.getState() != Wave.DEAD )
			{
				double distance = me.getLocation().distance(w.getLocation());
				if ( Double.compare(w.getRadius()+w.getVelocity(),distance ) < 0 )
				{
					double timeToHit = (distance - w.getRadius() - w.getVelocity()) / w.getVelocity();
					if ( timeToHit < closestTime )
					{
						if ( newClosestWave != null )
						{
							newSecondClosestWave = newClosestWave;
							secondClosestTime = closestTime;
						}
						newClosestWave = w;
						closestTime = timeToHit;
					}
					else if ( timeToHit < secondClosestTime )
					{
						newSecondClosestWave = w;
						secondClosestTime = timeToHit;
					}
				}
			}
		}
		this.closestWave = newClosestWave;
		this.secondClosestWave = newSecondClosestWave;
	}
	
	public double getStatArea(Wave w, Point2D.Double loc, int bracket)
	{
		double GFangle = w.absToGF(FastTrig.atan2(loc.getX()-w.getX(), loc.getY()-w.getY()) );
		
		BracketHist h =  null;
		if ( bracket == REAL )
		{
			h = w.getRealBracketHist();
		}
		else if ( bracket == FLAT )
		{
			h = w.getMainBracketHist();
		}
		else if ( bracket == RAND )
		{
			h = w.getRandBracketHist();
		}
		
		if ( h == null )
		{
			System.out.println("Warning! Trying to read a nonexistent bracket!");
			return 0.0000000000000001;
		}
		
		return h.GFtoArea(GFangle);
	}
	
	public PredictionStatus newPredStat(Bot b)
	{
		return new PredictionStatus(b.getX(),b.getY(),b.getHeading(),b.getV(),0);
	}
	
	public boolean least(double a, double b, double c)
	{
		return Double.compare(a,b) < 0 && Double.compare(a,c) < 0;
	}
	
	public boolean isRamming()
	{
		double eFutureDist = enemyFuture.distance(me.getLocation());
		double eNowDist = enemy.getLocation().distance(me.getLocation());
		return (eFutureDist < 40 && eNowDist < 250) || eFutureDist/eNowDist < 0.05 || eNowDist < 80;
	}
	
	// Crazy rough future projection to tag rammers
	public Point2D.Double enemyProject()
	{
		//Point2D.Double e1 = enemy.getMeOneAgo().getLocation();
		Point2D.Double e1 = enemy.getLocation();
		Point2D.Double e2 = enemy.getMeTwoAgo().getLocation();
		
		double angle12 = FastTrig.atan2(e1.getX()-e2.getX(), e1.getY()-e2.getY());
		
		double dist = me.getLocation().distance(enemy.getLocation());
		
		if ( Math.abs(enemy.getV()) < 0.5 )
		{
			dist = 0.0;
		}
		
		double x12 = enemy.getX() + dist*FastTrig.sin(angle12);
		
		double y12 = enemy.getY() + dist*FastTrig.cos(angle12);
		
		return new Point2D.Double(x12,y12);
		
	}
	
	public void setBackAsFront(double goAngle, double distance) {
        double angle = Utils.normalRelativeAngle(goAngle - robot.getHeadingRadians());
        if (Math.abs(angle) > FastTrig.HALF_PI ) {
            if (angle < 0) {
                robot.setTurnRightRadians(Math.PI + angle);
            } else {
                robot.setTurnLeftRadians(Math.PI - angle);
            }
            robot.setBack(distance);
        } else {
            if (angle < 0) {
                robot.setTurnLeftRadians(-1*angle);
           } else {
                robot.setTurnRightRadians(angle);
           }
            robot.setAhead(distance);
        }
    }
	
	/**
	 * Doesn't adjust the angle at all.
	 */
	public class MyOrbitDistanceController implements OrbitDistanceController {
		
		public static final double maxD = 425.0; //575
		public static final double minD = 250.0; //225
		//public static final double MAX_ADJUST = FastTrig.QUARTER_PI/2.5;
		
		public double angleAdjust(PredictionStatus p, double distance)
		{
			
			if ( distance > maxD )
			{
				return 0.001;
			}
			else if ( distance < minD )
			{
				return -0.001;
			}
			
			return 0.0;
		}
	}
	
}																																																																																																								