package nz.jdc.nano;

import robocode.AdvancedRobot;
import robocode.HitByBulletEvent;
import robocode.HitWallEvent;
import robocode.Rules;
import robocode.ScannedRobotEvent;
import robocode.util.Utils;

/**
 * NeophytePRAL (predictive rolling average linear)
 * 
 * My second nano bot. Close sibling to NeophyteSRAL.
 * Firing:  Simple prediction with rolling average linear velocity and linear targeting.
 *   The prediction is segmented on lateral velocity and last latVel and recorded 
 *   against the firing tick so it has memory/prediction of behaviour over the bullet 
 *   flight time.
 * Movement:  Bullet dodge. Toggles between uni-directional and reversing moves.
 * 
 * Credits: Pretty much everybody, especially those who contributed helpful articles to the
 * wiki, particularly the stuff on flat movement, rolling averages and nano-linear targetting.
 * Movement is based on the very elegant movement strategy of dft.Freddie 1.32 by Alcatraz271.
 * Also thanks to Michael Dorgan for the codesize ideas.
 *  
 * Version 1.0: Initial version. Hopefully not too sucky. Unfortunately testing shows it is
 *   weak against the plethora of pattern gun / random movement bots out there.
 *   Codesize 248.
 *
 * Version 1.1: Minor tweaks to a few params.
 *   Codesize 249.
 *   
 * Version 1.2: Change movement to one derived from dft.Freddie by Alcatraz271.  
 *   Codesize 249.
 * 
 * Version 1.3: Altered averaging, tuned params.
 *   Codesize 246. 
 *  
 * Version 1.4: Stripped some if the bullet power control and trimmed to
 *   squeeze in a second segment.
 *   Codesize 249.
 *
 * @author John Cleland
 */
public class NeophytePRAL extends AdvancedRobot {
  // constants
  // tunable constants
  // for movement
  private final static double MOVE_DISTANCE        = 240.0;
  private final static double NON_RAND             = 0.15;
  private final static double PREFERRED_RANGE      = 220.0;
  private final static double CLOSE_FCT            = 430 * MOVE_DISTANCE;
 
  // for gun
  private final static double ROLL_FACTOR         = 13;
  
  private final static double V_DIVISOR           = 1.39;                  // must be > 1 else P_OFFS will be > 7 and we
  private final static int    P_OFFS              = (int) (8 / V_DIVISOR); // will overflow the allocated bits
  private final static int    V_MAG_BITS          = 4;                     // 4 bits, max 15, so latvel index max -7 to 7 -> 0-14  
  private final static int    P_BINS              = 256;                   // 2 * 4 = 8 bits = 256
  private final static double BULLET_POWER_FACTOR = 450;//539;
  
  // attributes
  // for movement
  private static double   enemyEnergy;
  private static double   dir                     = MOVE_DISTANCE;
  private static double   moveMode                = 1;
  
  // for gun    
  private static int      lvPartIndexPrevious;
  private static int      plvLvIndexFiring;
  private static double[] rAvgLateralVelocity = new double[P_BINS]; // bit offset to act as 2 dimensional array

  /**
   * Bot run method.
   */
  public void run() {
    // testparam = Double.parseDouble(System.getProperty("RUMBLETESTER_PARAM"));
    
    // start our infinite radar loop
    turnRadarRightRadians(Double.POSITIVE_INFINITY);
  }
  
  /**
   * Scan handler.
   * This is where it all happens.
   */
  public void onScannedRobot(ScannedRobotEvent e) {
    // register vars, should be the most commonly used
    int    plvLvIndexCurrent;
    double rd;
    
    // other vars
    double latVel;
  
    // radar scan
    setTurnRadarLeftRadians(getRadarTurnRemaining());
    
    // bullet dodge a random distance
    if ( (char) ((enemyEnergy - 1.0999 - (enemyEnergy = e.getEnergy()))) < 2) {
      setAhead((dir *= (double) moveMode) * (Math.random() + NON_RAND));
    }
    
    // turn perpendicular with distance control
    setTurnRightRadians(Math.cos((rd = e.getBearingRadians()) - (e.getDistance() - PREFERRED_RANGE) * dir / CLOSE_FCT));
    // rd has enemy relative bearing, now updated to absolute
    rd += getHeadingRadians();
    
    // targeting
    // update our rolling average lateral velocity and get an integer index for predictive lookup
    // rd is updated to enemy absolute bearing
    rAvgLateralVelocity[plvLvIndexFiring] = (rAvgLateralVelocity[plvLvIndexFiring] * ROLL_FACTOR 
      + (latVel = (e.getVelocity() * Math.sin(e.getHeadingRadians() - rd)))) / (ROLL_FACTOR + 1.0);

    plvLvIndexCurrent = (lvPartIndexPrevious << V_MAG_BITS) + (lvPartIndexPrevious = P_OFFS + (int) (latVel / V_DIVISOR));
    
    // nano linear targetting fn
    setTurnGunRightRadians(Utils.normalRelativeAngle(rd - getGunHeadingRadians()
      + (rAvgLateralVelocity[plvLvIndexCurrent] / Rules.getBulletSpeed(rd = BULLET_POWER_FACTOR / e.getDistance()) ) ));
   
    // firing and stat update
    if (setFireBullet(rd) != null) {
      plvLvIndexFiring = plvLvIndexCurrent;
    }
  } 
  
  /**
   * Hit a wall, reverse direction and start moving immediately.
   */
  public void onHitWall(HitWallEvent e) {
    setAhead(dir = -dir);
  }

  /**
   * Hit by bullet, change movement strategy.
   */
  public void onHitByBullet(HitByBulletEvent e) {
    moveMode = -moveMode;
  }
}
