package tjk.utils;

import robocode.Rules;
import robocode.util.Utils;
import robocode.AdvancedRobot;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;
 
/**
 * Movement Predictor, also known as Precise Predictor
 *
 * Original Code by Nat @ RoboWiki, provided with no license (Public Domain) 
 *  at http://robowiki.net/wiki/User:Nat/Free_code#Movement_Predictor
 *
 * Copyright (c) 2012 by: 
 *     Nat @ Robowiki : http://robowiki.net/wiki/User:Nat
 *     Tom Kiesel (Tkiesel @ RoboWiki) : http://robowiki.net/wiki/User:Tkiesel
 * 
 * 
 * This software is provided 'as-is', without any express or implied warranty. In no event will the authors be held liable for any damages arising from the use of this software.
 * 
 * Permission is granted to anyone to use this software for any purpose, including commercial applications, and to alter it and redistribute it freely, subject to the following restrictions:
 * 
 *     1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. 
 *        If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
 * 
 *     2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
 * 
 *     3. This notice may not be removed or altered from any source distribution.
 * 
 */
 
public class MovementPredictor {

	// The wall stick.  Set to whatever you wish as long as it's 123+.
	// 150 and 160 seem to be popular values.
	public static final double WALL_STICK = 150.0;
	
	public static final double DEFAULT_WALL_MARGIN = 18.15;
	
	// Most common battlefield size.
	// Call setBFSize(w,h) in your robot initialization
	//	to set this up for other battlefields.
	private static double fieldWidth = 800.0;
	private static double fieldHeight = 600.0;

	/**
	 * Represents a predicted robot position in the future.
	 */
	public static class PredictionStatus extends Point2D.Double {
		private static final long serialVersionUID = 4116202515905711057L;
		public final double heading, velocity;
		public final long time;
 
		public PredictionStatus(double x, double y, double h, double v, long t)
		{
			super(x, y);
			heading = h;
			velocity = v;
			time = t;
		}
		
		public PredictionStatus(AdvancedRobot r) {
			this(r.getX(),r.getY(),r.getHeadingRadians(),r.getVelocity(),0);
		}
		
		public PredictionStatus copy() {
			return new PredictionStatus(getX(), getY(), heading, velocity, time);
		}
	}
	
	/**
	 * Determines how to adjust the orbital distance.
	 */
	public static interface OrbitDistanceController {
		/**
		 * Returns how much to adjust the orbit angle when orbiting clockwise.
		 */
		double angleAdjust(PredictionStatus p, double distance);
	}
	
	/**
	 * Doesn't adjust the angle at all.
	 */
	public static class NullOrbitDistanceController implements OrbitDistanceController {
		public double angleAdjust(PredictionStatus p, double distance)
		{
			return 0.0;
		}
	}
	
	/**
	 * Determines whether to end the movement prediction and return.
	 * This interface allows the user to write custom triggers
	 * to end the prediction loop.  The most common trigger for a 
	 * wave surfing bot will probably be a wave passing over the 
	 * predicted bot position.
	 */
	public static interface EndPredictionCondition {
		boolean endNow(PredictionStatus p);
	}
	
	/**
	 * A simple example EndPredictionCondition. 
	 * This one is just a sanity check of 20 time ticks for debugging. 
	 */
	public static class SanityEndPredictionCondition implements EndPredictionCondition {
		public boolean endNow(PredictionStatus p) {
			return p.time > 20;
		}
	}
	
	/**
	 * A simple example EndPredictionCondition for a wave surfer.
	 * <p> 
	 * Point2D.Double position  (wave launch position)
	 * double radius (wave radius at the current tick)
	 * double velocity (wave velocity)
	 */	
	public static class WaveEndPredictionCondition implements EndPredictionCondition {
		
		private final Point2D.Double position;
		
		private final double radius,velocity;
		
		public WaveEndPredictionCondition(Point2D.Double pos, double r, double v) {
			position = pos;
			radius = r;
			velocity = v;
		}
		
		public boolean endNow(PredictionStatus p) {
			return Double.compare(p.distanceSq(position),sqr(radius + velocity*(p.time-1))) < 0.0;
		}
		
		private double sqr(double a) {
			return a*a;
		}
	}
	
	////////////////////////  Truesurfing section ////////////////////////

	/**
	 * Workhorse method for predicting TrueSurfing Points.
	 *    This one has no orbital distance control.
	 */
	public static PredictionStatus predictWaveSurfpoint(PredictionStatus status, EndPredictionCondition ender, int clockwise, Point2D.Double wave) {
		OrbitDistanceController dControl = new NullOrbitDistanceController();
		return predictWaveSurfpoint(status, ender, dControl, clockwise, wave);
	}
			
	/**
	 * Workhorse method for predicting TrueSurfing Points.
	 */
	public static PredictionStatus predictWaveSurfpoint(PredictionStatus status, EndPredictionCondition ender, OrbitDistanceController dControl, int clockwise, Point2D.Double wave) {
			
		PredictionStatus thisPredictionStatus = status.copy();
		// Distance from attacker we're escaping.
		double attackDist = 0.0;
		
		double goAngle = 0.0;
		
		double doubleClock = (double)clockwise;
		
		// Update status until the EndPredictionCondition says we're done.
		do {
			attackDist = wave.distance(thisPredictionStatus);
			
			// Smooth along wall.
			goAngle = Utils.normalAbsoluteAngle(FastTrig.atan2(thisPredictionStatus.getX()-wave.getX(), thisPredictionStatus.getY()-wave.getY()) + doubleClock*(Math.PI/2.0 + dControl.angleAdjust(thisPredictionStatus, attackDist) ));
			goAngle = fastSmooth(thisPredictionStatus, goAngle, clockwise, attackDist);
			// Predict next tick.
			thisPredictionStatus = predict(thisPredictionStatus,goAngle);
		} while ( !ender.endNow(thisPredictionStatus) );
		
		return thisPredictionStatus;
	}
	
	public static PredictionStatus predictWaveStopPoint(PredictionStatus status, EndPredictionCondition ender, int clockwise, Point2D.Double wave) {
			
		PredictionStatus thisPredictionStatus = status.copy();
		// Calculate distance from attacker we're escaping.
		double attackDist = wave.distance(status);

		// Only need to update attack distance while it is 
		// less than the wall stick.
		boolean updateD = Double.compare(attackDist,WALL_STICK) < 0.0;		
		
		double goAngle = FastTrig.atan2(thisPredictionStatus.getX()-wave.getX(), thisPredictionStatus.getY()-wave.getY()) + ((double)clockwise)*Math.PI/2.0;
		
		// Update status until the EndPredictionCondition says we're done.
		do {
			// Update distance from attacker for proper wall smoothing.
			if ( updateD ) { 
				attackDist = wave.distance(thisPredictionStatus);
				updateD = Double.compare(attackDist,WALL_STICK) < 0.0;
			}
			// Smooth along wall.
			goAngle = FastTrig.atan2(thisPredictionStatus.getX()-wave.getX(), thisPredictionStatus.getY()-wave.getY()) + ((double)clockwise)*Math.PI/2.0;
			goAngle = fastSmooth(thisPredictionStatus, goAngle, clockwise, attackDist);
			// Predict next tick.
			thisPredictionStatus = predictStop(thisPredictionStatus,goAngle);
		} while ( !ender.endNow(thisPredictionStatus) );
		
		return thisPredictionStatus;		
	}		
	
	///////////////////////   END Truesurfing section ////////////////////
	
	////////////////////////  BEGIN MAXIMUM ESCAPE ANGLE SECTION /////////
	
	/**
	 * Calculates and returns the head-on angle and two MEA angles (relative to head-on)
	 * for a bot at 'status' from a bot at 'attacker'. 
	 * Prediction ended by 'ender'.
	 *
	 * MEA angles are returned as relative angles (-PI to PI).
	 * HeadOn angle is returned as absolute (0 to 2PI)
	 *
	 * angles returned are {CW, HeadOn, CCW}
	 *
	 * Use this if you want your MEA's as a pair of angles.
	 */
	public static double[] predictMEAangles(PredictionStatus status, EndPredictionCondition ender, Point2D.Double attacker) {
	
		double headOn = Utils.normalAbsoluteAngle( FastTrig.atan2(status.getX()-attacker.getX(), status.getY()-attacker.getY()) );
	
		double angleCW = Utils.normalAbsoluteAngle(headOn + Math.PI/2.0);
		
		double angleCCW = Utils.normalAbsoluteAngle(angleCW + Math.PI);
		
		// Clockwise result
		PredictionStatus cwpoint = predictMEApoint(status,angleCW,ender,1,attacker);
		
		// Counter-clockwise result
		PredictionStatus ccwpoint = predictMEApoint(status,angleCCW,ender,-1,attacker);
		
		double[] answer = new double[3];
		
		double answerCW = Utils.normalAbsoluteAngle(FastTrig.atan2(cwpoint.getX()-attacker.getX(), cwpoint.getY()-attacker.getY()));
		double distanceCW = cwpoint.distance(attacker);
		answerCW += halfBotWidth( answerCW, distanceCW );
		
		// Clockwise point first
		answer[0] = Utils.normalRelativeAngle( answerCW - headOn );
		
		// Head on second...
		answer[1] = headOn;
		
		double answerCCW = Utils.normalAbsoluteAngle(FastTrig.atan2(ccwpoint.getX()-attacker.getX(), ccwpoint.getY()-attacker.getY()));
		double distanceCCW = ccwpoint.distance(attacker);
		answerCCW -= halfBotWidth( answerCCW, distanceCCW );
		
		// Counter-Clockwise point third
		answer[2] = Utils.normalRelativeAngle( answerCCW - headOn );
		
		return answer;
	}
	
	/**
	 * Calculates and returns the two MEA points for a bot at 'status'
	 * from a bot at 'attacker'. Prediction ended by 'ender'.
	 *
	 * Use this if you want your MEAs as a pair of points on the battlefield.
	 */
	public static List<PredictionStatus> predictMEApoints(PredictionStatus status, EndPredictionCondition ender, Point2D.Double attacker) {
	
		ArrayList<PredictionStatus> answer = new ArrayList<PredictionStatus>(2);
	
		double headOn = Utils.normalAbsoluteAngle( FastTrig.atan2(status.getX()-attacker.getX(), status.getY()-attacker.getY()) );
	
		double angleCW = Utils.normalAbsoluteAngle(headOn + Math.PI/2.0);
		
		double angleCCW = Utils.normalAbsoluteAngle(angleCW + Math.PI);
		
		// Clockwise result
		answer.add( predictMEApoint(status,angleCW,ender,1,attacker) );
		
		// Counter-clockwise result
		answer.add( predictMEApoint(status,angleCCW,ender,-1,attacker) );
		
		return answer;
	}
 
	/**
	 * Private workhorse method for predicting MEAs.
	 */
	private static PredictionStatus predictMEApoint(PredictionStatus status, double goAngle, EndPredictionCondition ender, int clockwise, Point2D.Double attacker) {
		
		// Make a copy of status, so we don't influence other predictions.
		PredictionStatus firstPredictionStatus = status.copy();
		
		// Straight Line prediction.
		// Update status until the EndPredictionCondition says we're done.
		do {
			// Predict next tick.
			firstPredictionStatus = predict(firstPredictionStatus,goAngle);
		} while ( !ender.endNow(firstPredictionStatus) );
		
		// If straight line didn't go off-map, return.
		if ( firstPredictionStatus.getX() >= 18.0 && firstPredictionStatus.getY() >= 18.0 && firstPredictionStatus.getX() <= fieldWidth - 18.0 && firstPredictionStatus.getY() <= fieldHeight - 18.0) {
			return firstPredictionStatus;
		}
		
		// If we went off map, do a second prediction smoothed!
		PredictionStatus secondPredictionStatus = status.copy();
		// Calculate distance from attacker we're escaping.
		double startAttackDist = attacker.distance(status);
		// copy of original angle.
		//double newGoAngle = goAngle;
		
		//iterate thrice
		for ( int i = 0; i < 3; i++) {
			secondPredictionStatus = status.copy();
			double attackDist = startAttackDist;
			// Only need to update attack distance while it is 
			// less than the wall stick.
			boolean updateD = Double.compare(attackDist,WALL_STICK) < 0.0;		
			// Update status until the EndPredictionCondition says we're done.
			do {
				// Update distance from attacker for proper wall smoothing.
				if ( updateD ) { 
					attackDist = attacker.distance(secondPredictionStatus);
					updateD = Double.compare(attackDist,WALL_STICK) < 0.0;
				}
				// Smooth along wall.
				goAngle = fastSmooth(secondPredictionStatus, goAngle, clockwise, attackDist);
				// Predict next tick.
				secondPredictionStatus = predict(secondPredictionStatus,goAngle);
			} while ( !ender.endNow(secondPredictionStatus) );
			
			// Adjust goAngle to new target and reset secondPredictionStatus.
			goAngle = FastTrig.atan2(secondPredictionStatus.getX()-status.getX(), secondPredictionStatus.getY()-status.getY());
		}
		
		return secondPredictionStatus;
	}

	/*
	 * Precise angular half-width of a bot at some absolute angle and distance away.
	 */
	private static double halfBotWidth( double angle, double distance )
	{
		return Math.abs(FastTrig.atan2( 18.0 + Math.abs(7.45584412271571087843*FastTrig.sin(2.0*angle)), distance));
	}
 
	////////////////////////  END MAXIMUM ESCAPE ANGLE SECTION ///////////
 
	/**
	 * Calculate next tick prediction status. This always simulate accelerate to
	 * max velocity.
	 * 
	 * @param status
	 *            beginning status
	 * @param goAngle
	 *            angle to move, in radians, absolute
	 * @return predicted state next tick
	 */
	public static PredictionStatus predict(PredictionStatus status, double goAngle) {
		return predict(status, goAngle, Rules.MAX_VELOCITY);
	}

	/**
	 * stop predictor.
	 */
	public static PredictionStatus predictStop(PredictionStatus status, double goAngle) {
 
		if (FastTrig.cos(goAngle - status.heading) < 0) {
			goAngle += Math.PI;
		}
 
		return _predict(status, goAngle, Rules.MAX_VELOCITY, 0.0);
	}
 
	/**
	 * Calculate next tick prediction status. This always simulate accelerate to
	 * max velocity.
	 * 
	 * @param status
	 *            beginning status
	 * @param goAngle
	 *            angle to move, in radians, absolute
	 * @param maxVelocity
	 *            max allowed velocity of robot
	 * @return predicted state next tick
	 */
	public static PredictionStatus predict(PredictionStatus status, double goAngle, double maxVelocity) {
 
		int moveDir = 1;
 
		if (FastTrig.cos(goAngle - status.heading) < 0) {
			moveDir = -1;
		}
 
		return _predict(status, goAngle, maxVelocity, Double.POSITIVE_INFINITY * moveDir);
	}
 
	/**
	 * Calculate predicted status for every ticks before it reach its
	 * destination.
	 * 
	 * @param status
	 *            beginning status
	 * @param goAngle
	 *            angle to move, in radians, absolute
	 * @param maxVelocity
	 *            max allowed velocity of robot
	 * @param distanceRemaining
	 *            remain distance before stop
	 * @return list of predicted status
	 */
	public static List<PredictionStatus> predict(PredictionStatus status, double goAngle, double maxVelocity, double distanceRemaining) {
		List<PredictionStatus> predicted = new ArrayList<PredictionStatus>(20);
		predicted.add(status);
 
		while (distanceRemaining > 0) {
			status = _predict(status, goAngle, maxVelocity, distanceRemaining);
			predicted.add(status);
 
			// Deduct the distance remaining by the velocity
			distanceRemaining -= status.velocity;
		}
 
		return predicted;
	}
 
	/**
	 * Calculate predicted status for every ticks until timer run out.
	 * 
	 * @param status
	 *            beginning status
	 * @param tick
	 *            time available to move
	 * @param goAngle
	 *            angle to move, in radians, absolute
	 * @param maxVelocity
	 *            max allowed velocity of robot
	 * @return list of predicted status
	 */
	public static List<PredictionStatus> predict(PredictionStatus status,
			int tick, double goAngle, double maxVelocity) {
		List<PredictionStatus> predicted = new ArrayList<PredictionStatus>(
				tick + 2);
		predicted.add(status);
 
		while (tick-- > 0) {
			status = predict(status, goAngle, maxVelocity);
			predicted.add(status);
		}
 
		return predicted;
	}
 
	/**
	 * Calculate predicted status for every ticks before it reach its
	 * destination, or until timer run out.
	 * 
	 * @param status
	 *            beginning status
	 * @param tick
	 *            time available to move
	 * @param goAngle
	 *            angle to move, in radians, absolute
	 * @param maxVelocity
	 *            max allowed velocity of robot
	 * @param distanceRemaining
	 *            remain distance before stop
	 * @return list of predicted status
	 */
	public static List<PredictionStatus> predict(PredictionStatus status,
			int tick, double goAngle, double maxVelocity,
			double distanceRemaining) {
		List<PredictionStatus> predicted = new ArrayList<PredictionStatus>(
				tick + 2);
		predicted.add(status);
 
		while (distanceRemaining > 0 && tick-- > 0) {
			status = _predict(status, goAngle, maxVelocity, distanceRemaining);
			predicted.add(status);
 
			// Deduct the distance remaining by the velocity
			distanceRemaining -= status.velocity;
		}
 
		return predicted;
	}
 
	/**
	 * Calculate next tick prediction status. This always simulate accelerate to
	 * max velocity.
	 * 
	 * @param status
	 *            beginning status
	 * @param goAngle
	 *            angle to move, in radians, absolute
	 * @param maxVelocity
	 *            max allowed velocity of robot
	 * @param distanceRemaining
	 *            the remaining distance
	 * @return predicted state next tick
	 */
	private static PredictionStatus _predict(PredictionStatus status,
			double goAngle, double maxVelocity, double distanceRemaining) {
		double x = status.x;
		double y = status.y;
		double heading = status.heading;
		double velocity = status.velocity;
 
		// goAngle here is absolute, change to relative bearing
		goAngle -= heading;
 
		// If angle is at back, consider change direction
		if (FastTrig.cos(goAngle) < 0) {
			goAngle += Math.PI;
		}
 
		// Normalize angle
		goAngle = Utils.normalRelativeAngle(goAngle);
 
		// Max turning rate, taken from Rules class
		double maxTurning = Math.toRadians(10d - 0.75 * velocity);
		heading += limit(-maxTurning, goAngle, maxTurning);
 
		// Get next velocity
		velocity = getVelocity(velocity, maxVelocity, distanceRemaining);
 
		// Calculate new location
		x += FastTrig.sin(heading) * velocity;
		y += FastTrig.cos(heading) * velocity;
 
		// return the prediction status
		return new PredictionStatus(x, y, heading, velocity, status.time + 1l);
	}
 
	/**
	 * This function return the new velocity base on the maximum velocity and
	 * distance remaining. This is copied from internal bug-fixed Robocode
	 * engine.
	 * 
	 * @param currentVelocity
	 *            current velocity of the robot
	 * @param maxVelocity
	 *            maximum allowed velocity of the robot
	 * @param distanceRemaining
	 *            the remaining distance to move
	 * @return velocity for current tick
	 */
	private static double getVelocity(double currentVelocity,
			double maxVelocity, double distanceRemaining) {
		if (distanceRemaining < 0) {
			return -getVelocity(-currentVelocity, maxVelocity,
					-distanceRemaining);
		}
 
		double newVelocity = currentVelocity;
 
		final double maxSpeed = Math.abs(maxVelocity);
		final double currentSpeed = Math.abs(currentVelocity);
 
		// Check if we are decelerating, i.e. if the velocity is negative.
		// Note that if the speed is too high due to a new max. velocity, we
		// must also decelerate.
		if (currentVelocity < 0 || currentSpeed > maxSpeed) {
			// If the velocity is negative, we are decelerating
			newVelocity = currentSpeed - Rules.DECELERATION;
 
			// Check if we are going from deceleration into acceleration
			if (newVelocity < 0) {
				// If we have decelerated to velocity = 0, then the remaining
				// time must be used for acceleration
				double decelTime = currentSpeed / Rules.DECELERATION;
				double accelTime = (1 - decelTime);
 
				// New velocity (v) = d / t, where time = 1 (i.e. 1 turn).
				// Hence, v = d / 1 => v = d
				// However, the new velocity must be limited by the max.
				// velocity
				newVelocity = Math.min(maxSpeed, Math.min(Rules.DECELERATION
						* decelTime * decelTime + Rules.ACCELERATION
						* accelTime * accelTime, distanceRemaining));
 
				// Note: We change the sign here due to the sign check later
				// when returning the result
				currentVelocity *= -1;
			}
		} else {
			// Else, we are not decelerating, but might need to start doing so
			// due to the remaining distance
 
			// Deceleration time (t) is calculated by: v = a * t => t = v / a
			final double decelTime = currentSpeed / Rules.DECELERATION;
 
			// Deceleration time (d) is calculated by: d = 1/2 a * t^2 + v0 * t
			// + t
			// Adding the extra 't' (in the end) is special for Robocode, and v0
			// is the starting velocity = 0
			final double decelDist = 0.5 * Rules.DECELERATION * decelTime
					* decelTime + decelTime;
 
			// Check if we should start decelerating
			if (distanceRemaining <= decelDist) {
				// If the distance < max. deceleration distance, we must
				// decelerate so we hit a distance = 0
 
				// Calculate time left for deceleration to distance = 0
				double time = distanceRemaining / (decelTime + 1); // 1 is added
				// here due
				// to the extra 't'
				// for Robocode
 
				// New velocity (v) = a * t, i.e. deceleration * time, but not
				// greater than the current speed
 
				if (time <= 1) {
					// When there is only one turn left (t <= 1), we set the
					// speed to match the remaining distance
					newVelocity = Math.max(currentSpeed - Rules.DECELERATION,
							distanceRemaining);
				} else {
					// New velocity (v) = a * t, i.e. deceleration * time
					newVelocity = time * Rules.DECELERATION;
 
					if (currentSpeed < newVelocity) {
						// If the speed is less that the new velocity we just
						// calculated, then use the old speed instead
						newVelocity = currentSpeed;
					} else if (currentSpeed - newVelocity > Rules.DECELERATION) {
						// The deceleration must not exceed the max.
						// deceleration.
						// Hence, we limit the velocity to the speed minus the
						// max. deceleration.
						newVelocity = currentSpeed - Rules.DECELERATION;
					}
				}
			} else {
				// Else, we need to accelerate, but only to max. velocity
				newVelocity = Math.min(currentSpeed + Rules.ACCELERATION, maxSpeed);
			}
		}
 
		// Return the new velocity with the correct sign. We have been working
		// with the speed, which is always positive
		return (currentVelocity < 0) ? -newVelocity : newVelocity;
	}		
				
	/**
	 * If you want your bot to work no matter what battlefield
	 * size it'll fight on, use this once when your bot starts up.
	 */
	public static void setBFSize(double w, double h) {
		fieldWidth = w;
		fieldHeight = h;
	}
	
	public static final double fastSmooth(Point2D.Double p, double angle, int direction, double c2pd) {
		return fastSmooth(p.getX(), p.getY(), angle, direction, c2pd);
	}
	
	public static final double fastSmooth(Point2D.Double p, double angle, int direction, double c2pd, double margin) {
		return fastSmooth(p.getX(), p.getY(), angle, direction, c2pd, margin);
	}
	
	public static final double fastSmooth(double px, double py, double angle, int direction, double c2pd) {
		return fastSmooth(px, py, angle, direction, fieldWidth, fieldHeight, c2pd);
	}
	
	public static final double fastSmooth(double px, double py, double angle, int direction, double c2pd, double margin) {
		return fastSmooth(px, py, angle, direction, fieldWidth, fieldHeight, c2pd, margin);
	}

	public static final double fastSmooth(double px, double py, double angle, int direction, double fw, double fh, double c2pd) {
		return fastSmooth(px, py, angle, direction, fw, fh, c2pd, DEFAULT_WALL_MARGIN);
	}
			
	//no object creation or method calling if we can help it, need this to stay fast and rather memory unintensive
	//px = position x
	//py = position y
	//angle = angle we are trying to travel
	//direction = orbit direction (> 0 = clockwise, anything else is counter-clockwise)
	//fw = field width
	//fh = field height
	//c2pd = orbit center to position distance
	public static final double fastSmooth(double px, double py, double angle, int direction, double fw, double fh, double c2pd, double margin) {
 
		double stick = WALL_STICK;
		if(c2pd < stick) stick = c2pd;
 
		double stickSq = stick*stick;
 
		double nx = px + stick*FastTrig.sin(angle);
		double ny = py + stick*FastTrig.cos(angle);
 
		if(nx >= margin && nx <= fw - margin && ny >= margin && ny <= fh - margin)
			return angle;
 
		/* TOP */
		if(ny > fh - margin || py > fh - stick - margin) {
			//System.out.println("top");
			/* RIGHT */
			if(nx > fw - margin || px > fw - stick - margin) {
				//System.out.println("right");
				if(direction > 0) {
					//smooth right
					stick = fw - margin - px;
					nx = fw - margin;
					ny = py - direction * Math.sqrt(stickSq - stick*stick);
					return FastTrig.atan2(nx-px, ny-py);
				} else {
					//smooth top
					stick = fh - margin - py;
					nx = px + direction * Math.sqrt(stickSq - stick*stick);
					ny = fh - margin;
					return FastTrig.atan2(nx-px, ny-py);
				}
			} else /* LEFT */ if(nx < margin || px < stick + margin) {
				//System.out.println("left");
				if(direction > 0) {
					//smooth top
					stick = fh - margin - py;
					nx = px + direction * Math.sqrt(stickSq - stick*stick);
					ny = fh - margin;
					return FastTrig.atan2(nx-px, ny-py);
				} else {
					//smooth left
					stick = px - margin;
					nx = margin;
					ny = py + direction * Math.sqrt(stickSq - stick*stick);
					return FastTrig.atan2(nx-px, ny-py);
				}
			}
			//smooth top
			stick = fh - margin - py;
			nx = px + direction * Math.sqrt(stickSq - stick*stick);
			ny = fh - margin;
			return FastTrig.atan2(nx-px, ny-py);
		} else /* BOTTOM */ if(ny < margin || py < stick + margin) {
			/* RIGHT */
			if(nx > fw - margin || px > fw - stick - margin) {
				if(direction > 0) {
					//smooth bottom
					stick = py - margin;
					nx = px - direction * Math.sqrt(stickSq - stick*stick);
					ny = margin;
					return FastTrig.atan2(nx-px, ny-py);
				} else {
					//smooth right
					stick = fw - margin - px;
					nx = fw - margin;
					ny = py - direction * Math.sqrt(stickSq - stick*stick);
					return FastTrig.atan2(nx-px, ny-py);
				}
			} else /* LEFT */ if(nx < margin || px < stick + margin) {
				if(direction > 0) {
					//smooth left
					stick = px - margin;
					nx = margin;
					ny = py + direction * Math.sqrt(stickSq - stick*stick);
					return FastTrig.atan2(nx-px, ny-py);
				} else {
					//smooth bottom
					stick = py - margin;
					nx = px - direction * Math.sqrt(stickSq - stick*stick);
					ny = margin;
					return FastTrig.atan2(nx-px, ny-py);
				}
			}
			//smooth bottom
			stick = py - margin;
			nx = px - direction * Math.sqrt(stickSq - stick*stick);
			ny = margin;
			return FastTrig.atan2(nx-px, ny-py);
		}
	 
		/* RIGHT */
		if(nx > fw - margin || px > fw - stick - margin) {
			stick = fw - margin - px;
			nx = fw-margin;
			ny = py - direction * Math.sqrt(stickSq - stick*stick);
			return FastTrig.atan2(nx-px, ny-py);
		} else /* LEFT */ if(nx < margin || px < stick + margin) {
			stick = px - margin;
			nx = margin;
			ny = py+direction * Math.sqrt(stickSq - stick*stick);
			return FastTrig.atan2(nx-px, ny-py);
		}
		System.err.println("Something is really messed up here... (check your wall smoothing code!)");
		return angle;
	}

	
	public static double limit( double min, double test, double max) {
		
		//error check. Removable if you're very careful with your code.
		if ( Double.compare(min,max) > 0 ) { return limit(max,test,min); }
		
		return Math.max(min,Math.min(test,max));
	}
	
}																		