package tzu.intel;

import robocode.*;
import tzu.util.*;
import tzu.gun.*;

/**
 * Describes and tracks an enemy bullet.
 */
public class EnemyBullet implements Constants {

    double          x;
    double          y;
    double          expectedHitX;
    double          expectedHitY;
    double          energy;
    double          speed;
    double          heading;
    long            lastUpdate;
    long            creationTime;
    AdvancedRobot   myRobot;
    String          name;
    int             aimMethod;

    /**
     * Create a new enemy bullet object.
     * @param x         Current x position of bullet.
     * @param y         Current y position of bullet.
     * @param energy    Bullet's energy.
     * @param heading   Absolute heading of bullet.
     * @param ar        Reference to your AdvancedRobot.
     */
    public EnemyBullet(
            String shooter,
            double x,
            double y,
            double expectedHitX,
            double expectedHitY,
            double energy,
            double heading,
            AdvancedRobot ar,
            int     aimMethod,
            long    creationTime) {

        this.name           = shooter;
        this.x              = x;
        this.y              = y;
        this.expectedHitX   = expectedHitX;
        this.expectedHitY   = expectedHitY;
        this.energy         = energy;
        this.speed          = BotMath.calcBulletSpeed(energy);
        this.heading        = heading;
        this.myRobot        = ar;
        this.aimMethod      = aimMethod;
        this.lastUpdate     = creationTime;
        this.creationTime   = creationTime;

        double bearing = BotMath.calcHeading(
                myRobot.getX(),
                myRobot.getY(),
                x, y);

        update();
    }

    /**
     * Update the bullet's x,y position.
     */
    public void update() {

        long time = myRobot.getTime() - lastUpdate;
        double distance = time * speed;

        if (distance != 0.0) {
            x = BotMath.calcX(x, heading, distance);
            y = BotMath.calcY(y, heading, distance);
            lastUpdate = myRobot.getTime();
        }
    }

    public String getName()         { return name; }
    public double getX()            { return x; }
    public double getY()            { return y; }
    public double getSpeed()        { return speed; }
    public double getHeading()      { return heading; }
    public double getEnergy()       { return energy; }
    public double getExpectedHitX() { return expectedHitX; }
    public double getExpectedHitY() { return expectedHitY; }
    public long   getCreationTime() { return creationTime; }
    public int    getAimMethod()    { return aimMethod; }


    /**
     * Return the distance this bullet is from your robot.
     */
    public double getDistance() {
        return BotMath.calcDistance(x, y, myRobot.getX(), myRobot.getY());
    }

    /**
     * Return the bearing from your robot to this bullet object.
     */
    public double getBearing() {
        return BotMath.calcHeading(myRobot.getX(), myRobot.getY(), x, y);
    }

    /**
     * Calculate the point the bullet will be at in the future.
     * @param ticks     amount of game ticks into the future.
     * @return          a point object.
     */
    public Point calcPositionIn(long ticks) {
        return new Point(
                BotMath.calcX(x, heading, ticks * speed),
                BotMath.calcY(y, heading, ticks * speed));
    }

    /**
     * Calculate the distance between the bullet and your robot in the future.
     * @param ticks     amount of game ticks into the future.
     * @return          distance.
     */
    public double calcDistanceIn(long ticks) {

        Point p = calcPositionIn(ticks);
        return BotMath.calcDistance(p.x, p.y, myRobot.getX(), myRobot.getY());
    }

    /**
     * Calculate the bearing from your robot to this bullet in the future.
     * @param ticks     amount of game ticks into the future.
     * @return          bearing from your robot to the bullet.
     */
    public double calcBearingIn(long ticks) {

        Point p = calcPositionIn(ticks);
        return BotMath.calcHeading(myRobot.getX(), myRobot.getY(), p.x, p.y);
    }

    /**
     * Estimate the time to impact of the bullet hitting our bot.  This
     * estimate assumes that our robot will be travelling at a right angle
     * to the bullet path when it hits.
     */
    public long estimateTimeToImpact(double yourX, double yourY) {

        double  bearingToRobot =
                BotMath.calcHeading(x, y, yourX, yourY);

        double  angle = BotMath.abs(
                        BotMath.plusMinus180(
                        heading,
                        bearingToRobot));

        if (angle >= 90) {
            return Long.MAX_VALUE;
        }

        return (long)(getDistance() / BotMath.cos(angle) / speed);
//        return (long)(getDistance() * BotMath.cos(angle) / speed);
    }

    /**
     * Estimates the distance from your robot to the point
     * where the bullet may impact your robot.
     * @return  distance.
     */
    public double estimateDistanceToImpact(double yourX, double yourY) {
        return calcDistanceIn(estimateTimeToImpact(yourX, yourY));
    }

    /**
     * Estimates the bearing from your robot to the point where the bullet
     * may impact your robot.
     * @return  bearing from your robot to the estimated impact point.
     */
    public double estimateBearingToImpact(double yourX, double yourY) {
        return calcBearingIn(estimateTimeToImpact(yourX, yourY));
    }

    public double estimateReverseBearingToImpact(double yourX, double yourY) {
        return (estimateBearingToImpact(yourX, yourY) + 180) % 360;
    }

    public Point calcAntiGravity(double yourX, double yourY) {

        Point p = new Point(0,0);
        double force;
        double direction;

//        if (isAThreat(yourX, yourY)) {
        if (isAThreat(yourX, yourY) &&
           (energy > 0.1 || myRobot.getEnergy() < 10)) {

            force = (BUFFER_BULLET - getDistance()) * ANTI_GRAVITY_BULLETS;

            if (force > 0) {
                direction = estimateReverseBearingToImpact(yourX, yourY);
                p.x = BotMath.calcX(0, direction, force);
                p.y = BotMath.calcY(0, direction, force);
            }
        }
        return p;
    }

    /**
     * Estimate point where bullet may hit your robot.
     * @return  estimated point of impact.
     */
    public Point estimatePointOfImpact(double yourX, double yourY) {

        long t = estimateTimeToImpact(yourX, yourY);
        return new Point(
                BotMath.calcX(x, heading, t * speed),
                BotMath.calcY(y, heading, t * speed));
    }

    /**
     * Updates the bullet position (through isOutOfBounds() call) and
     * returns true if the bullet could possibly hit our robot.
     */
    public boolean isAThreat(double yourX, double yourY) {

        return

            isOutOfBounds()          == false &&
            willMiss(yourX, yourY)   == false &&
            BotMath.abs(
            BotMath.plusMinus180(
            heading,
            BotMath.calcHeading(x, y, yourX, yourY))) < 90;
    }

    public boolean isNotAThreat(double yourX, double yourY) {
        return !isAThreat(yourX, yourY);
    }

    /**
     * Returns true if there is no chance that this bullet will
     * hit our robot due to the heading of the bullet and our
     * robot's distance from it.
     */
    public boolean willMiss(double yourX, double yourY) {

        Point p = estimatePointOfImpact(yourX, yourY);
        return ((BotMath.calcDistance(x, y, p.x, p.y) / speed) <
               ((BotMath.calcDistance(yourX, yourY,
                p.x, p.y) - BattleField.getRobotSize()) / MAX_SPEED));
    }

    /**
     * Updates the bullet position and returns true if the bullet
     * is outside the battlefield area.
     */
    public boolean isOutOfBounds() {

        update();
        return (x < 0 ||  y < 0 ||
                x > BattleField.getWidth() ||
                y > BattleField.getHeight());
    }

    public String toString() {
        return  "Bullet: "      +
                " name="        + name                          +
                " aimMethod="   + aimMethod                     +
                " x="           + BotMath.round2(x)             +
                " y="           + BotMath.round2(y)             +
                " energy="      + BotMath.round2(energy)        +
                " speed="       + BotMath.round2(speed)         +
                " heading="     + BotMath.round2(heading)       +
                " bearing="     + BotMath.round2(getBearing())  +
                " distance="    + BotMath.round2(getDistance()) +
                " creation="    + creationTime;
    }
}
