package apv.test;
import java.util.*;
import java.awt.geom.*;
import robocode.robocodeGL.EllipseGL;

/**
 * WaveProbability - by Albert Perez
 * Manages waves and statistics
 */
public class WaveProbability {
	
	//static final int MAX_HISTORY = 2000;
	static final double BANDW = 0.1; //0.15
	static final double DECAY = 1;
	static final int WINDOW = 200;
	static final double HISTO_PRECISION = 0.05; //0.01
	
	private int bfs; //Counts the number of waves that have hit
	
	private double[] timeb = new double[100000];
	private double[] distanceb = new double[100000]; //has the distance where the wave hit
	private double[] directionb = new double[100000]; //has the direction of the wave that hit
	private double[] powerb = new double[100000]; //has the power of the bullet corresponding to the wave
	private String[] segmentb = new String[100000]; //has the segment corresponding to the wave
	private double[] factorsb = new double[100000]; //has the bearing of the wave that hit
	
	private ArrayList waves = null;
	
	private double[] values = new double[WINDOW]; //used for calculations
	
	/* MANAGEMENT FUNCTIONS */
	
	public WaveProbability() { waves = new ArrayList(); bfs = 0; }
		
	public void addWave(Point2D origin, Point2D destination, double power, double time, double distance, double direction, String segment) {
		Wave w = new Wave(origin, destination, power, time, distance, direction, segment); 
		waves.add(w);	
	}
	
	public void updateProbability(Point2D currentTargetPosition, double currentTime) {
	//call it every turn to update statistics
		int i = 0;	
		while (i < waves.size()) {	
			Wave wm = (Wave) waves.get(i);
			if ( wm.bulletHits(currentTargetPosition, currentTime) ) { 
				timeb[bfs] = wm.getTime();
				distanceb[bfs] = wm.getDistance();
				directionb[bfs] = wm.getDirection();
				powerb[bfs] = wm.getPower();
				segmentb[bfs] = wm.getSegment();
				factorsb[bfs++] = wm.guessFactor(currentTargetPosition.getX(), currentTargetPosition.getY());
				waves.remove(i);
			} else if (wm.bulletMisses(currentTargetPosition, currentTime)) {
				waves.remove(i);
			} else { i++; } 
		}	
	}
	
	//call at the begining of a new round
	public void newRound() { waves.clear();	}
	
	/* INFORMATION FUNCTIONS */
	
	//distance range
	public double getProbability(double atBearing, double minDistance, double maxDistance) {
		int n = 0;
		for (int i=bfs-1; i>=0 && n<WINDOW; i--) {
			if (distanceb[i] >= minDistance && distanceb[i] <= maxDistance) values[n++] = factorsb[i];
		}
		return KernelDensity.densityAtPoint(values, n, BANDW, atBearing, DECAY, WINDOW);
	}
	
	//distance range and direction
	public double getProbability(double atBearing, double minDistance, double maxDistance, double direction) {
		int n = 0;
		for (int i=bfs-1; i>=0 && n<WINDOW; i--) {
			if (distanceb[i] >= minDistance && distanceb[i] <= maxDistance &&
			directionb[i] == direction) values[n++] = factorsb[i];
		}
		return KernelDensity.densityAtPoint(values, n, BANDW, atBearing, DECAY, WINDOW);
	}
	
	//distance range direction and segment
	public double getProbability(double atBearing, double minDistance, double maxDistance, double direction, String segment) {
		boolean found = false;
		int slen = segment.length();
		int n = 0;
		while (!found && slen>0) { 
			String sseg = segment.substring(0,slen);
			for (int i=bfs-1; i>=0 && n<WINDOW; i--) {
				if (distanceb[i] >= minDistance && distanceb[i] <= maxDistance &&
				directionb[i] == direction && 
				sseg.equals(segmentb[i].substring(0,slen))) values[n++] = factorsb[i];
			}
			if (n > 0) found = true; else slen--;
		}
		return KernelDensity.densityAtPoint(values, n, BANDW, atBearing, DECAY, WINDOW);
	}
	
	//get best max probability factor
	public double getBestFactor(double minDistance, double maxDistance) {
		int n = 0;
		for (int i=bfs-1; i>=0 && n<WINDOW; i--) {
			if (distanceb[i] >= minDistance && distanceb[i] <= maxDistance) values[n++] = factorsb[i];
		}
		double maxp = 0; double bestf = 0;
		for (double k=-1; k<=1; k+=HISTO_PRECISION) {
			//double p = getProbability(k, minDistance, maxDistance); 
			double p = KernelDensity.densityAtPoint(values, n, BANDW, k, DECAY, WINDOW);
			if (p>maxp) { maxp = p; bestf = k; }
		}
		return bestf;
	}
	
	//get best max probability factor (considering direction)
	public double getBestFactor(double minDistance, double maxDistance, double direction) {
		int n = 0;
		for (int i=bfs-1; i>=0 && n<WINDOW; i--) {
			if (distanceb[i] >= minDistance && distanceb[i] <= maxDistance &&
			directionb[i] == direction) values[n++] = factorsb[i];
		}
		double maxp = 0; double bestf = 0;
		for (double k=-1; k<=1; k+=HISTO_PRECISION) {
			double p = KernelDensity.densityAtPoint(values, n, BANDW, k, DECAY, WINDOW);
			if (p>maxp) { maxp = p; bestf = k; }
		}
		return bestf;
	}
	
	//get best max probability factor (considering direction && segment)
	public double getBestFactor(double minDistance, double maxDistance, double direction, String segment) {
		boolean found = false;
		int slen = segment.length();
		int n = 0;
		while (!found && slen>0) {
			String sseg = segment.substring(0,slen); 
			for (int i=bfs-1; i>=0 && n<WINDOW; i--) {
				if (distanceb[i] >= minDistance && distanceb[i] <= maxDistance &&
				directionb[i] == direction && 
				sseg.equals(segmentb[i].substring(0,slen))) values[n++] = factorsb[i];
			}
			if (n > 0) found = true; else slen--;
		}
		double maxp = 0; double bestf = 0;
		for (double k=-1; k<=1; k+=HISTO_PRECISION) {
			double p = KernelDensity.densityAtPoint(values, n, BANDW, k, DECAY, WINDOW);
			if (p>maxp) { maxp = p; bestf = k; }
		}
		return bestf;
	}
	
	//returns the last computed bearing, as a pair (time, bearing)
	public Point2D getLastBearing() {
		if (bfs>0) {
			return new Point2D.Double(timeb[bfs-1],factorsb[bfs-1]);
		} else return null;
		
	}
	
	/* DRAWING FUNCTIONS */
	
	//updates graph of the oldest wave (the first to hit)
	public void updateClosestWaveGraph(EllipseGL graph, double time) {
		if (waves.size() > 0) {
			Wave wm = (Wave) waves.get(1);
			wm.updateGraph(graph, time);
		}
	}
	
}
