package rc.yoda;

import robocode.*;
import java.awt.*;
import java.util.*;
import rc.yoda.utils.Laws;
import rc.yoda.utils.YUtils;
import rc.yoda.plugin.PluginManager;

/** 
 * Yoda - by Robert Codd (Gorded) 
 *
 * This code is here by released under the RoboWiki Public Code Licence (RWPCL),
 * datailed on: http://robowiki.net/?RWPCL
 * (Basically it means you must keep the code public if you base any bot on it)
 *
 * Yoda.java : v1.0 -- 2007/05/12
 */ 


/**
 * Yoda is an AdvancedRobot that 
 * provides basic robot operations.
 * Yoda has quick scan radar with a
 * one hundred percent lock on the enemy
 * and provides accurate energy tracking to
 * provide an extra event method for additional
 * high-level robot behaviours
 *
 * @author Robert Codd
 * @version v1.0
 */
public class Yoda extends AdvancedRobot
{ 
	/**
	 * Constant value given to variables without a current value
	 */
	private static final int NO_VALUE = -1000;	
	
	/**
	 * the resinating angle of this robots scan arc
	 */
	private static final double SCAN_ARC = Math.PI / 8;

	/**
	 * PluginManager controls all dynamic extendable behaviours of this robot
	 */
	private PluginManager manager;							
	
	/**
	 * Popular quotes from the Star Wars Series used when a round is won
	 */
	private static String[] yodaQuotes = { "Powerful you have become, the dark side I sense in you.",
		"Try not. Do or do not, there is no try.",
		"Size matters not.",
		"Strong am I with the force.",
		"When 900 years you reach, look as good, you will not.",
		"A Jedi's strength flows from the Force.",
		"Grave danger you are in. Impatient you are.",
		"Fear is the path to the dark side.\n Fear leads to anger.\n Anger leads to hate.\n Hate leads to suffering.",
		"You must unlearn what you have learned.",
		"That is why you fail.",
		"Told you I did. Reckless is he. Now, matters are worse."};  
	
	/**
	 * The amount of ticks into the current round
	 */ 
	private long time = 0;					
	
	/**
	 * The current bearing toward the enemy in radians, unknown until first 
	 * onScannedRobotEvent therefor is initialized without a value
	 */
	private double enemyBearing = NO_VALUE;
	
	/**
	 * The enemy's last energy level, robots begin round with 100 energy.
	 * Used to track when the enemy fires a bullet
	 */
	private double previousEnemyEnergy = 100.0;
	
	/**
	 * The enemy's last known velocity, robot begin round stopped.
	 * Used to estimate the collision damage when the enemy hits a wall
	 */
	private double previousEnemyVelocity = 0;	
	
	/**
	 * The gun heat of the enemy, enemy can only fire while this is 0
	 * robots begin round with a gun heat of 3.0
	 */
	private double enemyGunHeat = 3.0;		
	
	/**
	 * The energy level of this robot, robot begin round with 100 energy.
	 * Used to track when the inactivity timer has kicked in
	 */
	private double robotsEnergy = 100.0;
	
	/**
	 * The current velocity of this robot, robot begin round stopped
	 * Used to estimate collision damage when this robot hits a wall
	 */
	private double robotsVelocity = 0;
	
	/**
	 * The power of the bullet this robot fired this tick. Used
	 * to keep track of energy loss through firing
	 */
	private double firePower = 0;
		
	/**
	 * Did this robot collide with another robot last tick
	 */ 
	private boolean wasRammed = false;
	
	/**
	 * Yoda's default behavior, all actions in while loop
	 * execute every round
	 */
	public void run() {
		manager = new PluginManager(this);
		YUtils.setFieldAttr(this);
		
		setColors(new Color(125, 95, 65),new Color(124, 205, 124),new Color(124, 205, 124));

		startRound();
		
		System.gc();
		
		setAdjustRadarForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		setAdjustGunForRobotTurn(true);
		
		while(true) 
		{
			doScan();
		}
	}
	
	/**
	 * Event method called by Robocode when this robot's scanner
	 * passes over another robot
	 * 
	 * @param ScannedRobotEvent information about the scanned robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		double enemyEnergy = e.getEnergy(),
			enemyVelocity = e.getVelocity();
		enemyBearing = e.getBearingRadians() + getHeadingRadians();
		robotsVelocity = getVelocity();
		
		int deltaTime = (int)(getTime() - time);
		double deltaMyEnergy = robotsEnergy - getEnergy() - firePower,
			deltaEnergy = previousEnemyEnergy - enemyEnergy - deltaMyEnergy;
		if (Math.abs(previousEnemyVelocity) > 2 && enemyVelocity == 0 && !wasRammed)
		{
			deltaEnergy -= getWallHitDamage(deltaEnergy, previousEnemyVelocity);
		}
		if ((enemyGunHeat -= deltaTime * 0.1) <= 0 && deltaEnergy > 0.09 && deltaEnergy < 3.01)
		{
			manager.onRobotFire(deltaEnergy);
			enemyGunHeat = Laws.getGunHeat(deltaEnergy) - 0.1;
		}
		previousEnemyEnergy = enemyEnergy;
		previousEnemyVelocity = enemyVelocity;
		firePower = 0;
		
		manager.onScannedRobot(e);
		
		robotsEnergy -= deltaMyEnergy; 
		time += deltaTime;
		wasRammed = false; 
	}

	/**
	 * Event method called by Robocode when this robot
	 * gets hit by a bullet
	 * 
	 * @param HitByBulletEvent information about ther bullet
	 * that hit this robot
	 */
	public void onHitByBullet(HitByBulletEvent e) {
		double power = e.getPower();
		previousEnemyEnergy += Laws.getBulletHitBonus(power);
		robotsEnergy -= Laws.getBulletDamage(power);
		manager.onHitByBullet(e);
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired hits another robot
	 * 
	 * @param BulletHitEvent information about the robot 
	 * that got hit by the bullet
	 */	
	public void onBulletHit(BulletHitEvent e) {
		double power = e.getBullet().getPower();
		robotsEnergy += Laws.getBulletHitBonus(power);
		previousEnemyEnergy -= Laws.getBulletDamage(power);
		manager.onBulletHit(e);
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired hit a wall
	 *
	 * @param BulletMissedEvent information about the bullet
	 */
	public void onBulletMissed(BulletMissedEvent e) {
		manager.onBulletMissed(e);
	}

	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired collides with a bullet fired by another robot
	 *
	 * @param BulletHitBulletEvent information about the bullets
	 */
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		manager.onBulletHitBullet(e);
	}
	
	/**
	 * Event method called by Robocode when this robot hits a wall
	 * 
	 * @param HitWallEvent information about the wall
	 */
	public void onHitWall(HitWallEvent e) {
		robotsEnergy -= getWallHitDamage(robotsEnergy - getEnergy(), robotsVelocity);
		manager.onHitWall(e);
	}
	
	/**
	 * Event method called by Robocode when this robot 
	 * collides with another robot
	 *
	 * @param HitRobotEvent information about the collision and 
	 * the other robot in the crash
	 */
	public void onHitRobot(HitRobotEvent e) {
		previousEnemyEnergy -= Laws.ROBOT_HIT_DAMAGE;
		robotsEnergy -= Laws.ROBOT_HIT_DAMAGE;
		wasRammed = true;
		manager.onHitRobot(e);
	}

	/**
	 * Event method called by Robocode when a robot dies
	 *
	 * @param RobotDeathEvent name of decaesed robot
	 */
	public void onRobotDeath(RobotDeathEvent e) {
		manager.onRobotDeath(e);
	}
	
	/**
	 * Event method called by Robocode when this robot
	 * wins a round
	 *
	 * @param WinEvent
	 */
	public void onWin(WinEvent e) {
		enemyBearing = NO_VALUE;
		setColors(new Color(124, 205, 124),new Color(124, 205, 124),new Color(124, 205, 124));
		out.println(yodaQuotes[(int)(Math.random() * yodaQuotes.length)]);
		
		manager.onWin(e);
	}
	
	/**
	 * Event method called by Robocode when this robot dies
	 *
	 * @param DeathEvent 
	 */
	public void onDeath(DeathEvent e) {
		manager.onDeath(e);
		Iterator events = getAllEvents().iterator();
		while (events.hasNext()) {
			Object obj = events.next();
			if (obj instanceof HitByBulletEvent) onHitByBullet((HitByBulletEvent) obj);
		}	
	}

	/**
	 * Event method called by Robocode when this robot is
	 * allowed to draw debugging graphics to the screen
	 *
	 * @param Graphics2D graphics that provides drawing method for painting
	 */
	public void onPaint(Graphics2D g) {
		manager.onPaint(g);
	}
	
	/**
	 * fires a bullet with the specified power and keeps track of 
	 * lossed energy due to firing 
	 *
	 * @param double power of the bullet to fire
	 */
	public void setFire(double power) {
		if (getGunHeat() == 0) { firePower = Math.min(getEnergy(), YUtils.limit(0.1, power, 3.0)); }
		super.setFire(power);
	}
	
	/**
	 * Calculates the most accurate wall hit damahge based on last known
	 * velocity and the deltaEnergy of the robot
	 *
	 * @param double the amount of energy lost
	 * @param double the last velocity of the robot
	 * @return double the damage taken by the robot due to hiting the wall
	 */
	private double getWallHitDamage(double deltaEnergy, double velocity) {
		double damage = Math.min(Laws.getWallHitDamage(Math.abs(velocity) + Laws.ACCELERATION), 3);
		if (damage > deltaEnergy) { damage = Laws.getWallHitDamage(velocity); }
		if (damage > deltaEnergy) { damage = Laws.getWallHitDamage(Math.abs(velocity) - Laws.DECELERATION); }
		return damage;
	}
	
	/**
	 * Efficiently scans the battlefield by turning all robot's parts in
	 * the direction of the center of the field to maximize radar turn rate
	 */
	private void startRound() {
		int direction = YUtils.sign(getRadarHeadingRadians() -
			YUtils.normalRelativeAngle(Math.atan2(getX() - YUtils.battleFieldWidth, getY() - YUtils.battleFieldHeight)));
		while (enemyBearing == NO_VALUE)
		{
			setTurnRadarRightRadians(Double.POSITIVE_INFINITY * direction);
			setTurnGunRightRadians(Laws.GUN_TURN_RATE_RADIANS * direction);
			setTurnRightRadians(Laws.MAX_TURN_RATE_RADIANS * direction);
			execute();
		}
	}
	
	/**
	 * Corrects the scanner to keep a lock on our enemy
	 */
	private void doScan() {
		double radarOffset = Double.POSITIVE_INFINITY;
		if(enemyBearing != NO_VALUE)
		{
			radarOffset = YUtils.normalRelativeAngle(getRadarHeadingRadians() - enemyBearing);
			radarOffset += YUtils.sign(radarOffset) * SCAN_ARC;
		}
		setTurnRadarLeftRadians(radarOffset);
		execute();
	}
}	