package ags.rougedc.robots;

import robocode.Rules;
import ags.utils.points.*;

/**
 * A base of a precise predictor, made from scraps of Gwynfor and LunarTwins
 * 
 * @author Alexander Schultz
 */
public abstract class VirtualRobotPredictor implements VirtualRobot {
    private AbsolutePoint fieldcorner;
    private AbsolutePoint location;
    private RelativePoint velocity;
    private double energy;
    private long deltatime;
    private boolean hitWall;
    
    public VirtualRobotPredictor(VirtualRobot sourcerobot, AbsolutePoint fieldcorner) {
        this.fieldcorner = fieldcorner;
        if (sourcerobot != null)
            reset(sourcerobot);
    }
    
    public void reset(VirtualRobot sourcerobot) {
        location = (AbsolutePoint)sourcerobot.getLocation().clone();
        velocity = (RelativePoint)sourcerobot.getVelocity().clone();
        energy = sourcerobot.getEnergy();
        deltatime = 0;
        hitWall = false;
    }
    
    public boolean hitsWall() {
        return hitWall;
    }
    
    public AbsolutePoint move() {
        deltatime++;
        
        /* Get intended acceleration */
        double acceleration = getAcceleration();
        /* Check if we're accelerating or deccelerating, and use the appropriate acceleration limit */
        if (Math.abs(acceleration) > Math.abs(velocity.getMagnitude()) && (Math.signum(acceleration) == Math.signum(velocity.getMagnitude())))
            acceleration = Math.max(Math.min(acceleration, Rules.ACCELERATION), -Rules.ACCELERATION);
        else
            acceleration = Math.max(Math.min(acceleration, Rules.DECELERATION), -Rules.DECELERATION);
        
        /* Calculate angular velocity */
        double maxav = Math.toRadians(Rules.getTurnRate(Math.abs(velocity.getMagnitude())));
        //double maxav = (10 - .75 * velocity.getMagnitude())*(Math.PI/180);
        double angularvelocity = Math.max(Math.min(getAngularVelocity(), maxav), -maxav); 
        
        /* Rotate and accelerate the velocity */
        velocity = (RelativePoint)velocity.clone();
        velocity.setDirectionMagnitude(velocity.getDirection()+angularvelocity, Math.max(Math.min(velocity.getMagnitude()+acceleration,Rules.MAX_VELOCITY),-Rules.MAX_VELOCITY));
        
        /* Increment the location */
        location = location.addRelativePoint(velocity);
        
        /* Process potential wall collisions */
        hitWall = processWallCollision();
        
        return location;
    }
    
    /*
     * The following code is derived from robotcode.peer.RobotPeer.checkWallCollision()
     * Used to crop predictions to within the walls accurately
     */
    private static final int ROBOT_SIZE=18;
    public boolean processWallCollision() {
        boolean hitWall = false;
        double fixx = 0, fixy = 0;
    
        if (Math.round(location.getX()) > fieldcorner.x - ROBOT_SIZE) {
            hitWall = true;
            fixx = fieldcorner.x - ROBOT_SIZE - location.getX();
        }
        if (Math.round(location.getX()) < ROBOT_SIZE) {
            hitWall = true;
            fixx = ROBOT_SIZE - location.getX();
        }
        if (Math.round(location.getY()) > fieldcorner.y - ROBOT_SIZE) {
            hitWall = true;
            fixy = fieldcorner.y - ROBOT_SIZE - location.getY();
        }
        if (Math.round(location.getY()) < ROBOT_SIZE) {
            hitWall = true;
            fixy = ROBOT_SIZE - location.getY();
        }
        
        double tanHeading = Math.tan(velocity.direction);
        double fixxtanHeading = fixx / tanHeading;
        double fixytanHeading = fixy * tanHeading;
        
        // if it hits bottom or top wall
        if (fixx == 0) {
            fixx = fixytanHeading;
        } // if it hits a side wall
        else if (fixy == 0) {
            fixy = fixxtanHeading;
        } // if the robot hits 2 walls at the same time (rare, but just in case)
        else if (Math.abs(fixxtanHeading) > Math.abs(fixy)) {
            fixy = fixxtanHeading;
        } else if (Math.abs(fixytanHeading) > Math.abs(fixx)) {
            fixx = fixytanHeading;
        }
        
        location.setLocation(location.x+fixx, location.y+fixy);
        
        if (hitWall)
            velocity.setDirectionMagnitude(velocity.direction, 0);
        
        return hitWall;
    }
    
    public AbsolutePoint getLocation() {
        return location;
    }
    public RelativePoint getVelocity() {
        return velocity;
    }
    public double getEnergy() {
        return energy;
    }
    public long getDeltaTime() {
        return deltatime;
    }
    
    abstract public double getAcceleration();
    abstract public double getAngularVelocity();
}
