package ag.targeting;

import java.awt.BasicStroke;
import java.awt.Graphics2D;
import java.awt.Color;
import java.io.File;
import java.util.Iterator;
import java.util.Vector;

import robocode.BulletHitEvent;
import robocode.BulletMissedEvent;
import robocode.DeathEvent;
import robocode.ScannedRobotEvent;
import ag.Gir;
import ag.battledata.DataRobot;
import ag.battledata.DataVector;
import ag.neuralgir.NeuralNet;
import ag.neuralgir.NeuralNetData;
import ag.neuralgir.TrainDataSet;

/**
 * Handles the robot targeting
 * @author Andree Grosse
 *
 */
public class TargetingNeural {
	
	private Gir mGir;
	private DataRobot mTarget, mTeamTarget, mScanTarget;
		
	private NeuralNet mNeuralNet; 
	
	private Vector<TrainDataSet> mTrainSet;
	private Vector<TrainDataSet> mTrainSet2;
	private int mShootWait;
	
	private int mGirStatsUseRounds;
	
	private int mInputSize; 
	// Stores how many last datasets shall be used for the net
	private int mOverallInputSize; 					
	private int mTrainIntervall;
	private TrainDataSet mLastTrainSet;
	// -- painting --
	private Vector<DataVector> mPredictGraphics;
	private BasicStroke mStrokeMove, mStrokeCircle;
	private Color mColorMove, mColorCircle;
	
	/**
	 * Constructor
	 * @param gir Gir
	 */
	public TargetingNeural(Gir gir) {
		mGir = gir;
		mTarget = null;
		mScanTarget = null;
		mTeamTarget = null;
		
		mInputSize = 14;
		mOverallInputSize = 2*mInputSize + 3;
				
		mTrainIntervall = 1;
		
		mPredictGraphics = null;
		
		mGirStatsUseRounds = 5;
		mShootWait = 0;
		
		mNeuralNet = null;		
		mTrainSet = new Vector<TrainDataSet>();
		mTrainSet2 = new Vector<TrainDataSet>();
		mLastTrainSet = null;
		// painting

		mStrokeMove = new BasicStroke(2.0f);
		mStrokeCircle = new BasicStroke(1.2f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL, 
				10.0f, new float[]{4, 6}, .0f);
		mColorCircle = Color.CYAN;
		mColorMove = Color.BLUE;
		
	}

	/**
	 * Main method - should be called every frame.
	 */
	public void run() {
				
		if(mScanTarget == null)
			scan();
		else
			moveRadar();
				
		if(mTarget != null && mTarget.checkData(mGir.getTime(), 1) >= 0) {
	
			if(mTarget.getEnergy(0) == .0){
				rotateGunToward(mTarget.getPosition(0));
				mGir.setFire(0.1);
				mShootWait = 100;
			}
			// train the net and shoot if data is available
			else{
				int offset = mTarget.checkData(mGir.getTime(), mOverallInputSize + 3);
				if(offset >= 0){
					shoot(offset);	
				}else{
					// align gun directly towards the target
					rotateGunToward(mTarget.getPosition(0));
				}
			}		
		}
		
		//setTurnRadarRight(Math.PI / 4 * ((mTick % 2) * 2 - 1));		
		
	}
	// ####################### getter & setter #######################
	
	/**
	 * Returns current target
	 * @return Current target
	 */
	public DataRobot getTarget() {
		return mTarget;	
	}	
		
	/**
	 * Sets the target and read the weights for it (if existing)
	 * @param target New target
	 */
	private void setTarget(DataRobot target) {
		if(mTarget == target)
			return;
		
		if(target == null){
			mTarget = null;
			return;
		}
		
		// only accept the target if the team has not found a target yet
		// or we have found the team target
		if(mTeamTarget == null || target == mTeamTarget)
		{
			setScanTarget(target);
			mTarget = target;		
			if(mTarget != null){
				readNet(mTarget);
				writeGunDataFile(mTarget);
				/*
				try {
					System.out.println("TEAM: Message send: TARGET = " +  mTarget.getName());
					mGir.broadcastMessage(new String[]{"TARGET", mTarget.getName()});
				} catch (IOException ex) {
					System.out.println("Unable to send message: " + ex);
				}
				*/
			}
		}
		
	}
	
	private void setScanTarget(DataRobot scanned){
		if(mScanTarget == scanned)
			return;
			
		if(mTeamTarget == null || scanned != mTeamTarget || mGir.getOthers() == mGir.getNumTeammates() + 1){

			if(mScanTarget != null)
				mGir.getMovement().writeMoveFactors(mScanTarget);
			mScanTarget = scanned;
			if(mScanTarget != null)
				mGir.getMovement().readMoveFactors(mScanTarget);
		}
	
	}
	
	/**
	 * This method should be called when the team sends an information
	 * to set a new target
	 * @param teamTarget the new team target
	 */
	public void setTeamTarget(DataRobot teamTarget){
		if(mTeamTarget == teamTarget)
			return;
		// reset scan target so Gir can find
		// a correct scan target (i.e. a different
		// one than the team target)
		mTeamTarget = teamTarget;
		if(mTeamTarget == mScanTarget)
			setScanTarget(null);
		setTarget(mTeamTarget);
	}
	
	// ############################# private ##################################
	
	/**
	 * Scales the x-position to (0, 1)
	 */
	private double scalePositionX(double x){
		return x / mGir.getBattleFieldWidth();
	}	
	
	/**
	 * Scales the y-position to (0, 1)
	 */
	private double scalePositionY(double y){
		return y / mGir.getBattleFieldHeight();
	}
	
	/**
	 * Scales the distance to (0, 1)
	 */
	private double scaleDistance(double distance){
		return Math.min(distance, 1500) / 1500;
	}
		
	/**
	 * Scales the signed velocity to (0, 1)
	 */
	private double scaleVelocity(double velocity){
		return (velocity + 8) * 0.0625;
	}
	
	/**
	 * Uncales the signed velocity to (-8, 8).
	 */
	private double unscaleVelocity(double velocity){
		return (velocity * 16) - 8;
	}		

	/**
	 * Scales the change in the heading of a robot to (0, 1). Change
	 * must not be greater than 0.175 (approx. PI/18)
	 */
	private double scaleAngleChange(double angle){
		double maxAbsAngle = 0.175;
		return 0.5 + (0.5 * angle) / maxAbsAngle;
	}
	
	/**
	 * Uncales the change in the heading of a robot to (-PI/18, PI/18). Change
	 * must not be greater than 0.175 (approx. PI/18)
	 */
	private double unscaleAngleChange(double angle){
		double maxAbsAngle = 0.175;
		return (angle - 0.5) * maxAbsAngle * 2;
	}
			
	// ###########################################################
	
	/**
	 * Moves the radar toward the scan target.
	 */
	private void moveRadar(){
		if(mScanTarget.checkData(mGir.getTime(), 1) != 0){
			scan();
			return;
		}
			
		DataVector nextpos = mScanTarget.getPosition(0).plus(mScanTarget.getAbsDirection(0).multiple(
					(mGir.getTime() % 2)*0.02 + mScanTarget.getAbsVelocity(0)));
		setRobotInBattleField(nextpos);
		rotateRadarToward(nextpos);	
	}
	
	/**
	 * Performs a 440 degree scan
	 */
	private void scan(){
		mGir.setTurnRadarRightRadians(3 * Math.PI);		
	}
	
	/**
	 * Rotates the gun towards a target point.
	 */
	private double rotateGunToward(DataVector target) {	
		DataVector align = target.minus(mGir.getData().getPosition(0));
		align.normalize();
		return alignGun(align);
	}
	
	/**
	 * Aligns the gun towards the vector. dir must be normalised
	 */
	private double alignGun(DataVector dir) {
		DataVector gunDir = new DataVector(mGir.getGunHeadingRadians());
		double angleDiff = gunDir.angleTo(dir);
		mGir.setTurnGunRightRadians(angleDiff);
		return Math.abs(angleDiff);
	}			
	
	/**
	 * Rotates the radar towards a target point.
	 */
	private double rotateRadarToward(DataVector target) {	
		DataVector align = target.minus(mGir.getData().getPosition(0));
		align.normalize();
		return alignRadar(align);
	}
	
	/**
	 * Aligns the gun towards the vector. dir must be normalised
	 */
	private double alignRadar(DataVector dir) {
		DataVector radarDir = new DataVector(mGir.getRadarHeadingRadians());
		double angleDiff = radarDir.angleTo(dir);
		mGir.setTurnRadarRightRadians(angleDiff);		
		
		return angleDiff;
	}	
	
	/**
	 * Calculates the bulletvelocity
	 * @param power Power of the bullet fired
	 * @return Velocity of the bullet
	 */
	private double bulletVelocity(double power) {
		return (double) 20 - 3*power;
	}
	
	/**
	 * Selects the firepower according to distance an gunstats. mTarget
	 * must not be null.
	 * @param dist Distance to the target
	 * @return Firepower (between 1 and 4)
	 */
	private double selectFirePower(double dist) {
		// the closer the target, the more likely it will be hit -> greater power
		double power;		
		if(dist < 200){
			power = 3;
		}
		else{
			power =  1 + Math.max(2 - (dist - 200) * 
				(0.015-(mTarget.getAverageGirStats() + mTarget.getCurrentGirStats())*0.006), 0);
		}
		power = Math.min(power, mTarget.getEnergy(0) * 0.25);
		return Math.min(power, mGir.getEnergy() * 0.3);
	}
		
	/**
	 * Truncates a robot's position point so that it is in 
	 * the battlefield
	 * @param v Point to be corrected
	 */
	private void setRobotInBattleField(DataVector v) {
		if(v.x < 18)
			v.x = 18;
		else if(v.x > mGir.getBattleFieldWidth() - 18)
			v.x = mGir.getBattleFieldWidth() - 18;
		
		if(v.y < 18)
			v.y = 18;
		else if(v.y > mGir.getBattleFieldHeight() - 18)
			v.y = mGir.getBattleFieldHeight() - 18;
	}
		
	/**
	 * Selects the time to wait until next shoot according to
	 * gunstats and distance. mTarget must not be null.
	 * @return Time to wait (in ticks)
	 */
	private int selectShootWait(double dist){
		if(dist > 800) dist = 800;
		dist = Math.max(10, (dist - 200)*0.05);
		double rnd = Math.random() * Math.random()*20;
		// gunstats of 0.2 are good enough
		rnd = rnd * (0.5 - Math.min(0.4, 
				mTarget.getAverageGirStats() + mTarget.getCurrentGirStats()));
		return (int)Math.max(0, dist * rnd);
	}
		
	/**
	 * Creates a TrainDataSet with input Data that can be
	 * fed into a neural net. The DataRobot set for the target
	 * mus be greater than 5.
	 * @param bulletvel Planned bullet velocity
	 * @return
	 */
	private TrainDataSet createNeuralDataInput(){
		
		TrainDataSet dataset = new TrainDataSet(mOverallInputSize, 2, mGir.getTime());
		
		DataVector dir;
		DataVector nextdir = mTarget.getDirection(0);
		
		double[] set = dataset.getInput();		
		
		for(int i = 0; i < mInputSize; i++){	
			dir = mTarget.getDirection(-i-1);
			set[2*i] = scaleAngleChange(dir.angleTo(nextdir));
			nextdir = dir;
			// scale all inputs to 0-1
			set[2*i+1] = scaleVelocity(mTarget.getVelocity(-i));
		}		
		
		set[mOverallInputSize-3] = scalePositionX(mTarget.getPosition(0).x);
		set[mOverallInputSize-2] = scalePositionY(mTarget.getPosition(0).y);
		set[mOverallInputSize-1] = scaleDistance(mGir.getData().getPosition(0).minus(
											mTarget.getPosition(0)).length());
	
		/*
		System.out.println("## TRAINSET ##");
		for(int j = 0; j < mOverallInputSize; j++){
			System.out.println(dataset.getInput()[j]);
		}		
		*/
		
		return dataset;
	}
	
	/**
	 * Trains the neural network with a random train data set if there 
	 * are more than 30 train data sets stored.	 *
	 */
	private void train(TrainDataSet newdata){	
		
		
		if(mLastTrainSet == null){
			mLastTrainSet = new TrainDataSet(newdata);
			return;
		}
		
		if(mLastTrainSet.getTick() == mGir.getTime() - 1){

			/*
			System.out.println("---TRAINSET IN");
			for(int j = 0; j < mOverallInputSize; j++){
				System.out.println(mLastTrainSet.getInput()[j]);
			}
			System.out.println("--OUT");
			System.out.println(mLastTrainSet.getOutput()[0]);
			System.out.println(mLastTrainSet.getOutput()[1]);
			*/
			if(mGir.getTime() % mTrainIntervall == 0){

				
				mLastTrainSet.getOutput()[0] = scaleAngleChange(mTarget.getDirection(-1).angleTo(mTarget.getDirection(0)));
				mLastTrainSet.getOutput()[1] = scaleVelocity(mTarget.getVelocity(0));
	
				mTrainSet.add(mLastTrainSet);
				if(mGir.getWriteData() && mGir.isDataSpaceAvailable())
					mGir.getFileManager().appendDataSet(mTarget.getName() + "_gundata.dat", mLastTrainSet.getDataSet());
				
				if(mTrainSet.size() > 30){
		  			TrainDataSet dataset = mTrainSet.remove(
						(int)Math.round(Math.random()*(mTrainSet.size() - 1)));
		  			mTrainSet2.add(dataset);
					mNeuralNet.trainBackpropagation(dataset.getInput(), dataset.getOutput());
				}
		  		if(mTrainSet2.size() > 30){
		  			TrainDataSet dataset = mTrainSet2.remove(
						(int)Math.round(Math.random()*(mTrainSet2.size() - 1)));
					mNeuralNet.trainBackpropagation(dataset.getInput(), dataset.getOutput());
				}
			}

			mLastTrainSet = new TrainDataSet(newdata);
					
		}  		
	}
	
	/**
	 * Shoots a bullet.
	 */
	private void shoot(int offset){	
		TrainDataSet dataset = createNeuralDataInput();
			
		train(dataset);

		DataVector girpos = mGir.getData().getPosition(0);
		DataVector diff = mTarget.getPosition(0).minus(girpos);
		DataVector pos = mTarget.getPosition(0);
			
		double power = selectFirePower(diff.normalize());
		
		double bulletvel = bulletVelocity(power);
		
		DataVector dir = mTarget.getDirection(0);
		double velocity = mTarget.getVelocity(0);
				
		double dist;
		Vector<DataVector> predictedWay = new Vector<DataVector>();
		predictedWay.add(pos);
		
		for(int tick = 2 - offset; tick < 40; tick++){

			// Calculate the new position of the target
			pos = pos.plus(dir.multiple(velocity));
			setRobotInBattleField(pos);	
			// add to graphics		
			predictedWay.add(pos);
			
			diff = pos.minus(girpos);

			// dist is the predicted distance to the target in turn 'tick'
			dist = diff.normalize();			

			// if a bullet fired at the right angle would
			// at 'tick' hit the target, fire!
			if(dist / bulletvel < tick){
				
				alignGun(diff);
				if(mGir.getGunHeat() == .0 && (mShootWait <= 0 || dist < 50)){
					predictedWay.add(mGir.getData().getPosition(0));
					mPredictGraphics = predictedWay;
					mGir.setFire(power);
					mShootWait = selectShootWait(dist);
				}
				else
					mShootWait--;
				
				break;
			}
			// otherwise, predict new movement of the target
			else{
				dataset.setOutput(mNeuralNet.calcOutput(dataset.getInput()));
								
				dir = dir.rotate(unscaleAngleChange(dataset.getOutput()[0]));
				velocity = unscaleVelocity(dataset.getOutput()[1]);

				
				if(velocity > 8) velocity = 8;
				else if(velocity < -8) velocity = -8;

				if(velocity > 8) velocity = 8;
				else if(velocity < -8) velocity = -8;
				
				for(int i = 2*mInputSize - 1; i > 1; i--)
					dataset.getInput()[i] = dataset.getInput()[i-2];
				dataset.getInput()[1] = dataset.getOutput()[1];
				dataset.getInput()[0] = dataset.getOutput()[0];
					
				
				dataset.getInput()[mOverallInputSize-3] = scalePositionX(pos.x);
				dataset.getInput()[mOverallInputSize-2] = scalePositionY(pos.y);
				dataset.getInput()[mOverallInputSize-1] = scaleDistance(dist);
			}
		}
		
	}
	
	// ############################# public ##################################
	

	// ############################## reaction methods ##############################
	/**
	 * This method should be called when a bullet 
	 * hits another robot
	 */
	public void onBulletHit(BulletHitEvent e) {		
		if(mTarget != null && e.getName().equals(mTarget.getName()))
			mTarget.hitThisRobot();		
	}
	
	/**
	 * This method should be called when one of Girs 
	 * bullets misses (hits a wall) 
	 */
	public void onBulletMissed(BulletMissedEvent e){
		if(mTarget != null)
			mTarget.missedThisRobot();	
	}
	
	/**
	 * This method should be called when nother robot dies 
	 */
	public void onRobotDeath(DataRobot dead) {
		if(mTarget != null && mTarget == dead) {
			writeNet(mTarget);
			mTarget = null;
		}
		if(mScanTarget != null && mScanTarget == dead){
			setScanTarget(null);
		}
	}
	
	/**
	 * This method should be called when Gir sees another robot 
	 */	
	public void onScannedRobot(ScannedRobotEvent e){
		// don't bother about teammates
		if(mGir.isTeammate(e.getName()))
			return;
		
		if(mTarget == null){
			setTarget(mGir.getDataBattle().getRobotByName(e.getName()));
		}
		
		if(mScanTarget == null){			
			setScanTarget(mGir.getDataBattle().getRobotByName(e.getName()));
		}
			
	}
	
	/**
	 * This method should be called when Gir dies 
	 */
	public void onDeath(DeathEvent e){
		if(mTarget != null){
			writeNet(mTarget);
		}
		if(mScanTarget != null){
			mGir.getMovement().writeMoveFactors(mScanTarget);
		}
	}
	
	/**
	 * This method should be called for painting.
	 */
	public void onPaint(Graphics2D g){
		if(mPredictGraphics == null)
			return;
		DataVector vec1, vec2 = null;
		g.setStroke(mStrokeMove);
		g.setColor(mColorMove);
		for(int i = 0; i < mPredictGraphics.size() - 2; i++){
			vec1 = mPredictGraphics.get(i);
			vec2 = mPredictGraphics.get(i+1);
			
			g.drawLine((int)Math.round(vec1.x), (int)Math.round(vec1.y), 
					(int)Math.round(vec2.x), (int)Math.round(vec2.y));
		}

		g.setColor(mColorCircle);
		g.setStroke(mStrokeCircle);
		DataVector girpos =  mPredictGraphics.get(mPredictGraphics.size() - 1);
		int len = (int)Math.round(girpos.minus(vec2).length());
		g.drawOval((int)Math.round(girpos.x) - len, (int)Math.round(girpos.y) - len, 
				2*len, 2*len);
	}
	
	// ############################## data handling ##############################
	
	
	/**
	 * create "robotname_gundata.dat" file if it does not exist
	 */ 
	public void writeGunDataFile(DataRobot robot){
		if(mGir.getWriteData()){
			File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_gundata.dat");
			if(!file.exists()  || (mGir.getRoundNum() == 0))
				mGir.getFileManager().writeNetLayersize(file, mNeuralNet);
		}
	}
	
	/**
	 * Writes the targeting net for this robot. Overwrites older files.
	 * @param robot robot for which the net shall be written
	 */
	private void writeNet(DataRobot robot){

		writeGunStats(robot);
		
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_gun.net");
		mGir.getFileManager().writeNetRC(file, mNeuralNet);
		System.out.println("--WRITTEN: GunNet for " + robot.getName());
		if(!mGir.isSpaceAvailable()){
			file.delete();
			System.out.println("--Not enough disk space available: Deleted GunNet for " + robot.getName());
		}
	}

	/**
	 * Reads the targeting net for this robot. If no net is available,
	 * the initial gun net is loaded.
	 * @param robot Robot for which the net shall be read
	 */
	private void readNet(DataRobot robot){

		readGunStats(robot);
		
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_gun.net");
		if((robot.getSizeGirStats() >= mGirStatsUseRounds) && robot.getAverageGirStats() < 0.1){
			// after 'mGirStatsUseRounds' round, test if the gun statistics are very low. If so,
			// start with a new randomly initialised net.
			System.out.println("!!! Low hit rate. Creating new gun net for " + robot.getName());
			int[] size = {mOverallInputSize, mOverallInputSize, 2};
			NeuralNetData data = new NeuralNetData(size, 0.7, -0.0002, 12, 0.25);
			mNeuralNet = new NeuralNet(data);
			addNewNetFlag(robot);
			mShootWait = 300;
		}else{
			if(file.exists()){
				mNeuralNet = mGir.getFileManager().readNet(file);
				System.out.println("--READ: GunNet for " + robot.getName());
				mShootWait = 0;
			}
			else{
				file = new File(mGir.getDataDirectory() + "/_initial_gun.net");
				//read initial net
				if(file.exists()){
					mNeuralNet = mGir.getFileManager().readNet(file);
					System.out.println("--READ: Initial GunNet");
					mShootWait = 90;
				}else{
					int[] size = {mOverallInputSize, mOverallInputSize, 2};
					NeuralNetData data = new NeuralNetData(size, 0.65, -0.0003, 12, 0.2);
					mNeuralNet = new NeuralNet(data);
					addNewNetFlag(robot);
					mShootWait = 150;
				}
			}
		}
	}

	/**
	 * Adds a flag to a gun statistics file that a new net has been created.
	 * @param robot the robot for which to set this flag
	 */
	@SuppressWarnings("unchecked")
	private void addNewNetFlag(DataRobot robot){
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_girstats.stt");
		
		Vector<Double> stats;
		
		if(!file.exists() || (mGir.getRoundNum() == 0))
			stats = new Vector<Double>();
		else				
			stats = mGir.getFileManager().readVector(file);
		
		stats.add(-1.0);
		mGir.getFileManager().writeVector(file, stats);
	}
	
	/**
	 * Writes the gun statistics for a robot
	 * @param robot the robot for which the statistics shall
	 * 			be written - must not be null
	 */
	@SuppressWarnings("unchecked")
	private void writeGunStats(DataRobot robot) {
			/*				
			Date date = new Date();
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy.MM.dd-HH:mm:ss");
			mFileManager.writeStats("Enemy: " + ((DataRobot)mDataBattle.mRobots.get(1)).getName() +
					" -- Battle on " + sdf.format(date));
					*/	
		
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_girstats.stt");
		
		Vector<Double> stats;
		
		if(!file.exists() || (mGir.getRoundNum() == 0))
			stats = new Vector<Double>();
		else				
			stats = mGir.getFileManager().readVector(file);
		
		if(robot.getCurrentGirStats() != -2)
			stats.add(robot.getCurrentGirStats());
		
		double temp = .0;
		int size = 0, counter = 0;
		double current;
		System.out.println("~~Gir's GunStats for " + robot.getName() + ":");	
  		
		for(int i = 0; i < stats.size(); i++){
  			current = (Double)stats.get(i);
  			if(current == -1.0){
  				// minus one means a new net has been created
  				temp = 0;
  				size = 0;
  			}else{
  	  			// don't print too much
  				if(stats.size() < 30 || i > stats.size() - 30)
					System.out.println("      Round " + counter + ": " + current);
  				counter++;
	  			if(i >= stats.size() - mGirStatsUseRounds){  					
	  				// only use the last 'mGirStatsUseRounds' rounds
	  				temp += current;
	  				size++;
  				}
  			}
		}
		if(size != 0)
			System.out.println("      Average for last " + size + " rounds: " + temp / size);
		else{
			System.out.println("     No Gunstats for this round available.");
		}
		
		if(!mGir.getWriteData() && mGir.isLastRound()){
			file.delete();
			System.out.println("~~Last round: deleted gir's gun statistics file for " + robot.getName());
		}
		else{
			mGir.getFileManager().writeVector(file, stats);	
			System.out.println("~~WRITTEN: Gir's GunStats for " + robot.getName());	
		}	
	}
	
	/**
	 * Reads the gun statistics for a robot
	 * @param robot The robot for which statistics should be read
	 */
	private void readGunStats(DataRobot robot){
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_girstats.stt");
			
		if(!file.exists() || (mGir.getRoundNum() == 0)){
			mTarget.setAverageGirStats(0.3);
			return;
		}
			
		Vector stats = mGir.getFileManager().readVector(file);
		
		double temp = .0;
		double current;
		int size = 0;
				
  		for(int i = 0; i < stats.size(); i++){
  			current = (Double)stats.get(i);
  			if(current == -1.0){
  				// minus one means a new net has been created
  				temp = 0;
  				size = 0;
  			}else if(i >= stats.size() - mGirStatsUseRounds){
  				// only use the last 'mGirStatsUseRounds' rounds
  				temp += current;
  				size++;
  			}
		}
  		System.out.println("~~READ: Gir's GunStats for " + robot.getName() + ": Average for last " +
  				size + " rounds: "+ (temp / size));
		mTarget.setAverageGirStats(temp / size);
		mTarget.setSizeGirStats(size);
	}
	
	// ############################## helpers ##############################

}
