package djc;

/**
 * Coordinate - Based on Coordinate class by Simon Parker
 */
public class Coordinate
{
    private final double EQUALITY_THRESHOLD = 2.0;
    public double x;
    public double y;

    public static double FIELD_WIDTH = 600;
    public static double FIELD_HEIGHT = 800;

    public Coordinate()
    {
	set(0, 0);
    }
    
    public Coordinate(double posX, double posY)
    {
	set(posX, posY);
    }
    
    public Coordinate(Coordinate c)
    {
	set(c.x, c.y);
    }

    public void set(double posX, double posY)
    {
	x = posX;
	y = posY;
    }

    public void set(Coordinate c)
    {
	set(c.x, c.y);
    }

    public Coordinate minus(Coordinate c)
    {
	return new Coordinate(x - c.x, y - c.y);
    }

    public Coordinate plus(Coordinate c)
    {
	return new Coordinate(x + c.x, y + c.y);
    }
    
    public boolean isEqualTo(Coordinate coordinate)
    {
	return ((Math.abs(x-coordinate.x) < EQUALITY_THRESHOLD) &&
		(Math.abs(y-coordinate.y) < EQUALITY_THRESHOLD) );
    }

    public double length()
    {
	return Math.sqrt(x*x+y*y);
    }

    public double distanceFrom(Coordinate c)
    {
	double xo = c.x - x;
	double yo = c.y - y;
	return Math.sqrt( xo*xo + yo*yo );
    }

    public double headingTo(Coordinate c)
    {
	double xo = c.x - x;
	double yo = c.y - y;
	double h=Math.sqrt(xo * xo + yo * yo);

	if( xo > 0 && yo > 0 )
	    return Math.asin( xo / h );
	if( xo > 0 && yo < 0 )
	    return Math.PI - Math.asin( xo / h );
	if( xo < 0 && yo < 0 )
	    return Math.PI + Math.asin( -xo / h );
	if( xo < 0 && yo > 0 )
	    return 2.0*Math.PI - Math.asin( -xo / h );
	if( xo == 0 && yo > 0)
	    return 0;
	if( xo == 0 && yo < 0)
	    return Math.PI;
	if( xo > 0 && yo == 0)
	    return Math.PI / 2;
	if( xo < 0 && yo == 0)
	    return 3 * Math.PI / 2;
	return 0;
    }

    public double headingToDeg(Coordinate c)
    {
	return Math.toDegrees(headingTo(c));
    }

    public Coordinate rotate(double angle)
    {
	angle = -angle; //due to the strange Coordinate system
	return new Coordinate (x * Math.cos(angle) - y * Math.sin(angle),
			       x * Math.sin(angle) + y * Math.cos(angle));
    }

    public boolean offField(double width, double height, double slop)
    {
	if(x < slop || x > width - slop || y < slop || y > height - slop)
	    return true;
	else
	    return false;
    }

    public boolean offField()
    {
	return offField(Coordinate.FIELD_WIDTH, Coordinate.FIELD_HEIGHT, 0);
    }

    static public Coordinate limitToBattleField(Coordinate position)
    {
	final double ROBOT_RADIUS = 20;
	Coordinate result = new Coordinate(position);
	
	result.x = Math.max(result.x, ROBOT_RADIUS);
	result.y = Math.max(result.y, ROBOT_RADIUS);
	result.x = Math.min(result.x, FIELD_WIDTH - ROBOT_RADIUS);
	result.y = Math.min(result.y, FIELD_HEIGHT - ROBOT_RADIUS);
	
	return result;
    }

}

