package djc;
import robocode.*;

/**
 * CornersTacticalStrategy
 * Hide in the corners during a Melee
 *
 * All code created by Dan Cieslak, 2002.
 *
 * All code that is mine is available for anyone to use for
 * any purpose, so long as I am listed as a source.
 *
 */
public class CornersTacticalStrategy extends DefaultTacticalStrategy
{
    /* ********************************************************************************** */
    /*                                   CONSTANTS                                        */
    /* ********************************************************************************** */
    public static double CLOSE_ENOUGH = 10;

    /* ********************************************************************************** */
    /*                                MEMBER VARIABLES                                    */
    /* ********************************************************************************** */
    public int arrived = 0;

    // Constructor; set up the CornersTacticalStrategy class.
    public CornersTacticalStrategy (StrategyBot theRobot, TacticalStrategyManager theStrategyManager)
    {
	super(theRobot, theStrategyManager);
	name = TacticalStrategyManager.CORNERS;
    }

    public void reset() 
    {
	arrived = 0;
    }

    /**
     * Corner to aim for.
     */
    public Coordinate findCorner(Coordinate myPos, double factor)
    {
	double width = self.getBattleFieldWidth() / 2.0;
	double height = self.getBattleFieldHeight() / 2.0;
	double mywidth = factor * self.getWidth();
	double myheight = factor * self.getHeight();
	double cornerx = 0;
	double cornery = 0;
	
	if(self.getX() > width) {  // on right half
	    cornerx = (width * 2) - mywidth;
	} else { // on left half
	    cornerx = mywidth;
	}
	if(self.getY() > height) {  // on upper half
	    cornery = (height * 2) - myheight;
	} else { // on bottom half
	    cornery = myheight;
	}
	
	return new Coordinate(cornerx, cornery);
    }

    /**
     * Find a wobble point
     *
     * TODO: Take nearest opponent into account.
     */
    public Coordinate findWobble(Coordinate myPos)
    {
	double width = self.getBattleFieldWidth() / 2.0;
	double height = self.getBattleFieldHeight() / 2.0;
	double cornerx = 0;
	double cornery = 0;
	double factorx = 0.75;
	double factory = 0.75;

	// Decide which way to wobble - true is the "mostly x direction"
	boolean bWobbleDir = self.random.nextBoolean();

	if(bWobbleDir)   // mostly x
	    factorx = 2.5;
	else 
	    factory = 2.5;

	double mywidth = factorx * self.getWidth();
	double myheight = factory * self.getHeight();

	// Where am I?
	if(self.getX() > width) {  // on right half
	    cornerx = (width * 2) - mywidth;
	} else { // on left half
	    cornerx = mywidth;
	}
	if(self.getY() > height) {  // on upper half
	    cornery = (height * 2) - myheight;
	} else { // on bottom half
	    cornery = myheight;
	}
	
	return new Coordinate(cornerx, cornery);
    }

    /**
     * Basically a state machine:
     *  1. Go to nearest corner (arrived = 1 when complete)
     *  2. Nestle into the corner (arrived = 2 when complete)
     *  3. Wobble randomly up or down (arrived = 1 when complete)
     *     a. Takes distance to nearest enemy into account
     *  4. Head back to Nestled position.
     *
     * Note: Special case if hitByBullet (arrived = 10)
     *
     * @author Dan Cieslak
     */
    public void setMovement()
    {
	if(arrived == 0) {  // Head towards corner
	    Coordinate myPos = new Coordinate(self.getX(), self.getY());
	    Coordinate myCorner = findCorner(myPos, 3);
	    self.goTo(myCorner);
	    //self.waitFor(new MoveCompleteCondition(self));
	    myPos.x = self.getX();
	    myPos.y = self.getY();
	    if(myPos.distanceFrom(myCorner) < CLOSE_ENOUGH)
		arrived = 1;
	} else if(arrived == 1) { // Nestle in
	    Coordinate myPos = new Coordinate(self.getX(), self.getY());
	    Coordinate myCorner = findCorner(myPos, 1.5);
	    self.goTo(myCorner);
	    //self.waitFor(new MoveCompleteCondition(self));
	    myPos.x = self.getX();
	    myPos.y = self.getY();
	    if(myPos.distanceFrom(myCorner) < CLOSE_ENOUGH)
		arrived = 2;
	} else if(arrived == 2) {  // Wobble
	    Coordinate myPos = new Coordinate(self.getX(), self.getY());
	    Coordinate wobblePoint = findWobble(myPos);
	    self.goTo(wobblePoint);
	    //self.waitFor(new MoveCompleteCondition(self));
	    myPos.x = self.getX();
	    myPos.y = self.getY();
	    if(myPos.distanceFrom(wobblePoint) < CLOSE_ENOUGH)
		arrived = 1;
	} else {
	    self.out.println("Unknown state");
	    arrived = 0;
	}
    }

    /**
     * onHitRobot - Drove into somebody.
     */
    public void onHitRobot(HitRobotEvent e) {
	super.onHitRobot(e);
	self.goTo(pickNewLocation(MIN_DISTANCE, MAX_DISTANCE_RANGE));
	//self.waitFor(new MoveCompleteCondition(self));
	arrived = 0;
    }
}
