package ar.horizon.util;

import static java.lang.Math.*;
import static ar.horizon.util.Util.*;

import java.awt.geom.*;

/**
 * Represents a robot's state during a given tick.
 * 
 * @author Aaron Rotenberg
 * @see RobotLog
 */
public class RobotRecording {
	// Information that is supplied when the recording is constructed.
	private final long roundTime;
	private final long totalTime;
	private final Point2D.Double location;
	private final double heading;
	private final double velocity;
	private final double energy;

	// Information that is derived from the enemy recording in
	// setEnemyRecording().
	private RobotRecording enemyRecording;
	private double enemyDistance;
	private double enemyAbsoluteBearing;
	private double lateralVelocity;

	// Information that is derived from the previous recording in
	// updateWithPreviousRecording().
	private int lateralDirection;
	private double acceleration;
	private int timeSinceAcceleration;
	private double wallDistanceForward;
	private double wallDistanceBackward;

	public RobotRecording(long roundTime, long totalTime,
			Point2D.Double location, double heading, double velocity,
			double energy) {
		this.roundTime = roundTime;
		this.totalTime = totalTime;
		this.location = location;
		this.heading = heading;
		this.velocity = velocity;
		this.energy = energy;
	}

	public long getRoundTime() {
		return roundTime;
	}

	public long getTotalTime() {
		return totalTime;
	}

	public Point2D.Double getLocation() {
		return location;
	}

	public double getHeading() {
		return heading;
	}

	public double getVelocity() {
		return velocity;
	}

	public double getEnergy() {
		return energy;
	}

	public RobotRecording getEnemyRecording() {
		return enemyRecording;
	}

	public void setEnemyRecording(RobotRecording enemyRecording) {
		this.enemyRecording = enemyRecording;
		enemyDistance = location.distance(enemyRecording.location);
		enemyAbsoluteBearing =
				absoluteBearing(location, enemyRecording.location);

		lateralVelocity = velocity * sin(enemyAbsoluteBearing - heading);
	}

	public double getEnemyDistance() {
		return enemyDistance;
	}

	public double getEnemyAbsoluteBearing() {
		return enemyAbsoluteBearing;
	}

	public double getLateralVelocity() {
		return lateralVelocity;
	}

	void updateWithPreviousRecording(RobotRecording lastEntry,
			Rectangle2D.Double fieldRectangle) {
		boolean lastEntryIsValid = (lastEntry != null);

		if (lateralVelocity != 0) {
			lateralDirection = sign(lateralVelocity);
		} else if (lastEntryIsValid) {
			lateralDirection = lastEntry.lateralDirection;
		} else {
			lateralDirection = 1;
		}

		if (lastEntryIsValid) {
			acceleration = lastEntry.getVelocity() - getVelocity();
		} else {
			acceleration = 0.0;
		}

		if (lastEntryIsValid && acceleration == 0.0) {
			timeSinceAcceleration = lastEntry.timeSinceAcceleration + 1;
		} else {
			timeSinceAcceleration = 0;
		}

		wallDistanceForward =
				getWallDistance(this, lateralDirection, fieldRectangle);
		wallDistanceBackward =
				getWallDistance(this, -lateralDirection, fieldRectangle);
	}

	public int getLateralDirection() {
		return lateralDirection;
	}

	public double getAcceleration() {
		return acceleration;
	}

	public int getTimeSinceAcceleration() {
		return timeSinceAcceleration;
	}

	public double getWallDistanceForward() {
		return wallDistanceForward;
	}

	public double getWallDistanceBackward() {
		return wallDistanceBackward;
	}
}
