package rc.yoda.utils;

import robocode.*;
import rc.yoda.utils.Laws;
import java.awt.geom.Point2D;
import java.util.ArrayList;

/**
 * Surfer - by Robert Codd (Gorded) 
 *
 * This code is here by released under the RoboWiki Public Code Licence (RWPCL),
 * datailed on: http://robowiki.net/?RWPCL
 * (Basically it means you must keep the code public if you base any bot on it)
 *
 * Surfer.java : v1.0 -- 2007/05/12
 */ 

/**
 * Surfer - a class by (Robert Codd)
 */
public class Surfer extends Movement
{	
	public static final int ROLLING_DEPTH = 10;

	/**
	 * The number of stat bins being used
	 */
	protected static final int BINS = 31;

	/**
	 * The bin that represents GF 0
	 */
	protected static final int MIDDLE_BIN = (BINS - 1) / 2;

	/**
	 * Non-segmented surf stats plus the depth of the bins
	 */	
	protected static double[] minimalSurfStats = new double[BINS + 1];

	protected static double[][][] fastSurfStats = 
		new double[Index.DISTANCE_SMALL][Index.LATERAL_VELOCITY_SMALL][BINS + 1];

	protected static double[][][][][] surfStats = 
		new double[Index.DISTANCE][Index.LATERAL_VELOCITY][Index.BULLET_POWER][Index.DELTA_HEADING][BINS + 1];

	protected static double[][][][][][][][] ultimateSurfStats = 
		new double[Index.DISTANCE_LARGE][Index.LATERAL_VELOCITY_LARGE][Index.BULLET_POWER_LARGE][Index.DELTA_HEADING]
		                                                                                         [Index.ACCELERATION][Index.TIME_SINCE_DCHANGE][Index.NEAR_WALL][BINS + 1];

	/**
	 * The location of this AdvancedRobot
	 */
	protected static Point2D.Double robotsLocation;

	/**
	 * The location of our enemy
	 */
	protected static Point2D.Double enemyLocation;

	/**
	 * A list of all waves on the battleField
	 */
	protected static ArrayList _Waves = new ArrayList(); 

	/**
	 * A list of the previous Directions this robot has gone
	 */
	protected static ArrayList _Directions = new ArrayList();

	/**
	 * A list of the previous Bearing to the 
	 * enemy this robot has being
	 */
	protected static ArrayList _Bearings = new ArrayList();

	protected static ArrayList _Distances = new ArrayList();

	protected static ArrayList _Headings = new ArrayList();

	protected static ArrayList _TimeSinceDChange = new ArrayList();

	protected static ArrayList _LateralVelocities = new ArrayList();

	protected static ArrayList _Locations = new ArrayList();

	/**
	 * The last Direction this robot was travelling
	 */
	private int previousDirection = 1;

	private int timeSinceDChange = 0;

	/**
	 * Class Constructor specfiying the AdvancedRobot Ataru is working for
	 */				
	public Surfer(AdvancedRobot robot) {
		super(robot);
		_Waves.clear();
		_Bearings.clear();
		_Directions.clear();
		_Distances.clear();
		_Locations.clear();		
		_Headings.clear();
		_TimeSinceDChange.clear();
		_LateralVelocities.clear();

		robotsLocation = new Point2D.Double();
		enemyLocation = new Point2D.Double();
	}	

	/**
	 * Event method called by this robot when it detects
	 * that the enemy fired a bullet
	 *
	 * @param double deltaEnergy the power of the bullet fired
	 */
	public void onRobotFire(double deltaEnergy) {		
		Wave wave = new Wave();

		int[] segments = { Index.distanceIndex(((Double)_Distances.get(1)).doubleValue(), 1),
				Index.lateralVelocityIndex(((Double)_LateralVelocities.get(1)).doubleValue(), 1),
				Index.distanceIndex(((Double)_Distances.get(1)).doubleValue(), 2),
				Index.lateralVelocityIndex(((Double)_LateralVelocities.get(1)).doubleValue(), 2),
				Index.bulletPowerIndex(deltaEnergy, 1),
				Index.deltaHeadingIndex(((Double)_Headings.get(1)).doubleValue() - ((Double)_Headings.get(2)).doubleValue()),
				Index.distanceIndex(((Double)_Distances.get(1)).doubleValue(), 3),
				Index.lateralVelocityIndex(((Double)_LateralVelocities.get(1)).doubleValue(), 3),
				Index.bulletPowerIndex(deltaEnergy, 2),
				Index.deltaHeadingIndex(((Double)_Headings.get(1)).doubleValue() - ((Double)_Headings.get(2)).doubleValue()),			
				Index.accelerationIndex(((Double)_LateralVelocities.get(1)).doubleValue() - ((Double)_LateralVelocities.get(2)).doubleValue()),
				Index.timeSinceDChangeIndex( ((Integer)_TimeSinceDChange.get(1)).intValue()),
				Index.nearWallIndex((Point2D.Double)_Locations.get(1)) };

		wave.fireTime = robot.getTime() - 1;
		wave.bulletVelocity = Laws.getBulletSpeed(deltaEnergy);
		wave.distanceTraveled = Laws.getBulletSpeed(deltaEnergy);

		wave.directAngle = ((Double)_Bearings.get(1)).doubleValue();
		wave.fireLocation = (Point2D.Double)enemyLocation.clone(); 
		wave.direction = ((Integer)_Directions.get(1)).intValue();
		wave.segments = segments;

		_Waves.add(wave);
	}		

	/**
	 * Event method called by Robocode when this robot's scanner
	 * passes over another robot
	 * 
	 * @param ScannedRobotEvent information about the scanned robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		robotsLocation = new Point2D.Double(robot.getX(), robot.getY());

		double enemyDistance = e.getDistance(),
		enemyBearing = e.getBearingRadians() + robot.getHeadingRadians(),
		//		relativeHeading = robot.getHeadingRadians() - (enemyBearing + Math.PI),
		lateralVelocity = robot.getVelocity() * Math.sin(e.getBearingRadians());

		int direction = lateralVelocity > 0 ? 1 : (lateralVelocity < 0 ? -1 : previousDirection);
		timeSinceDChange = direction == previousDirection ? timeSinceDChange + 1 : 0;
		previousDirection = direction;

		_Directions.add(0, new Integer(direction));
		_Bearings.add(0, new Double(enemyBearing + Math.PI));
		_Distances.add(0, new Double(enemyDistance));
		_Locations.add(0, robotsLocation.clone());
		_Headings.add(0, new Double(robot.getHeadingRadians()));
		_LateralVelocities.add(0, new Double(lateralVelocity));	
		_TimeSinceDChange.add(0, new Integer(timeSinceDChange));
		if (_Bearings.size() > 2) 
		{ 
			_Bearings.remove(2); 
			_Directions.remove(2);
			_Distances.remove(2);
			_Locations.remove(2);
			_TimeSinceDChange.remove(2);	
		}
		else if (_LateralVelocities.size() > 3) 
		{
			_LateralVelocities.remove(3); 
			_Headings.remove(3);
		}

		enemyLocation = YUtils.project(robotsLocation, enemyBearing, enemyDistance);

		updateWaves();
	}	

	/**
	 * Event method called by Robocode when this robot
	 * gets hit by a bullet
	 * 
	 * @param HitByBulletEvent information about ther bullet
	 * that hit this robot
	 */
	public void onHitByBullet(HitByBulletEvent e) {
		if (!_Waves.isEmpty()) 
		{
			Point2D.Double bulletLocation = new Point2D.Double(e.getBullet().getX(), e.getBullet().getY());
			Wave hitWave = null;

			for (int count = 0; count < _Waves.size(); count++) 
			{
				Wave wave = (Wave)_Waves.get(count);

				if (Math.abs(wave.distanceTraveled - robotsLocation.distance(wave.fireLocation)) < 18
						&& Math.round(Laws.getBulletSpeed(e.getBullet().getPower()) * 10) == Math.round(wave.bulletVelocity * 10)) 
				{
					hitWave = wave;
					break;
				}
			}
			if (hitWave != null) {
				logHit(hitWave, bulletLocation);
				_Waves.remove(_Waves.lastIndexOf(hitWave));
			}
		}
	}

	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired collides with a bullet fired by another robot
	 *
	 * @param BulletHitBulletEvent information about the bullets
	 */	
	public void onBulletHitBullet(BulletHitBulletEvent e) {	
		if (!_Waves.isEmpty()) 
		{
			Bullet bullet = e.getBullet().getName() == robot.getName() ? e.getHitBullet() : e.getBullet();
			Point2D.Double bulletLocation = new Point2D.Double(bullet.getX(), bullet.getY());
			Wave hitWave = null;

			for (int count = 0; count < _Waves.size(); count++) 
			{
				Wave wave = (Wave)_Waves.get(count);

				if (Math.abs(wave.distanceTraveled - robotsLocation.distance(wave.fireLocation)) < 18
						&& Math.round(Laws.getBulletSpeed(bullet.getPower()) * 10) == Math.round(wave.bulletVelocity * 10)) 
				{
					hitWave = wave;
					break;
				}
			}
			if (hitWave != null) {
				logHit(hitWave, bulletLocation);
				_Waves.remove(_Waves.lastIndexOf(hitWave));
			}
		}
	}

	/**
	 * Advances all waves in the air by one tick
	 */
	protected void updateWaves() {
		for (int count = 0; count < _Waves.size(); count++) 
		{
			Wave wave = (Wave)_Waves.get(count);

			wave.distanceTraveled = (robot.getTime() - wave.fireTime + 1) * wave.bulletVelocity;
			if (wave.distanceTraveled > robotsLocation.distance(wave.fireLocation) + 18) 
			{
				_Waves.remove(count--);
			}
		}
	}

	/**
	 * Logs a hit on the specified wave at the specified Location
	 *
	 * @param Wave the wave the hit 
	 * @param Point2D.Double the location at which the wave hit
	 */
	protected void logHit(Wave wave, Point2D.Double targetLocation) {
		int index = getFactorIndex(wave, targetLocation);				

		double[] firstStats = minimalSurfStats,
		secondStats = fastSurfStats[wave.segments[0]][wave.segments[1]],
		thirdStats = surfStats[wave.segments[2]][wave.segments[3]][wave.segments[4]][wave.segments[5]],
		fourthStats = ultimateSurfStats[wave.segments[6]][wave.segments[7]][wave.segments[8]][wave.segments[9]][wave.segments[10]][wave.segments[11]][wave.segments[12]];

		firstStats[BINS]++;
		secondStats[BINS]++;
		thirdStats[BINS]++;
		fourthStats[BINS]++;


		for (int count = 0; count < BINS; count++) 
		{
			double weight =  (20 - wave.bulletVelocity) / 3;
			firstStats[count] = YUtils.rollingAverage(firstStats[count], Math.pow(0.5, Math.abs(index - count) / 2), 
					Math.min(firstStats[BINS], ROLLING_DEPTH), weight);	

			secondStats[count] = YUtils.rollingAverage(secondStats[count], Math.pow(0.5, Math.abs(index - count) / 2), 
					Math.min(secondStats[BINS], ROLLING_DEPTH), weight);					

			thirdStats[count] = YUtils.rollingAverage(thirdStats[count], Math.pow(0.5, Math.abs(index - count) / 2), 
					Math.min(thirdStats[BINS], ROLLING_DEPTH), weight);	

			fourthStats[count] = YUtils.rollingAverage(fourthStats[count], Math.pow(0.5, Math.abs(index - count) / 2), 
					Math.min(fourthStats[BINS], ROLLING_DEPTH), weight);				
		}

		minimalSurfStats = firstStats;
		fastSurfStats[wave.segments[0]][wave.segments[1]] = secondStats;
		surfStats[wave.segments[2]][wave.segments[3]][wave.segments[4]][wave.segments[5]] = thirdStats;
		ultimateSurfStats[wave.segments[6]][wave.segments[7]][wave.segments[8]][wave.segments[9]][wave.segments[10]][wave.segments[11]][wave.segments[12]] = fourthStats;
	}

	/**
	 * Return the segment that has has the best stats
	 * for the specified wave
	 *
	 * @param Wave the wave for which stats are needed
	 * @return double an stat array for the Wave
	 */
	protected double[] getWaveStats(Wave wave) {
		if (ultimateSurfStats[wave.segments[6]][wave.segments[7]][wave.segments[8]][wave.segments[9]][wave.segments[10]][wave.segments[11]][wave.segments[12]][BINS] > 0) {
			return ultimateSurfStats[wave.segments[6]][wave.segments[7]][wave.segments[8]][wave.segments[9]][wave.segments[10]][wave.segments[11]][wave.segments[12]];
		}
		else if (surfStats[wave.segments[2]][wave.segments[3]][wave.segments[4]][wave.segments[5]][BINS] > 0) {
			return surfStats[wave.segments[2]][wave.segments[3]][wave.segments[4]][wave.segments[5]];
		}
		else if (fastSurfStats[wave.segments[0]][wave.segments[1]][BINS] > 0) {
			return fastSurfStats[wave.segments[0]][wave.segments[1]];
		}
		return minimalSurfStats;	
	}

	/**
	 * Calculate the position of the bullet represented by the
	 * enemy wave and the bin
	 * 
	 * @param the enemy wave the bullet is on
	 * @param the bin representation of the bullet
	 * @return the current location of the bullet
	 */
	protected Point2D.Double projectBin(Wave wave, int bin) {
		return YUtils.project(wave.fireLocation,(bin - MIDDLE_BIN) * (YUtils.maxEscapeAngle(wave.bulletVelocity) * wave.direction
				/ MIDDLE_BIN) + wave.directAngle, wave.distanceTraveled);	
	}

	/**
	 * Caculates the bin representation of the targetLocation on the specified Wave
	 * 
	 * @param Wave the wave to caculate angle from the target
	 * @param Point2D.Double the location to calculate angle from the wave source
	 * @return the bin representation of the angle from the wave source
	 */
	protected static int getFactorIndex(Wave wave, Point2D.Double targetLocation) {
		double factor = YUtils.normalRelativeAngle(YUtils.absoluteBearing(wave.fireLocation, targetLocation) - wave.directAngle)
		/ YUtils.maxEscapeAngle(wave.bulletVelocity) * wave.direction;
		return (int)YUtils.limit(0,(factor + 1) * MIDDLE_BIN, BINS - 1);
	}

	static class Index {
		public static final int DISTANCE_SMALL = 3,
		DISTANCE = 5,
		DISTANCE_LARGE = 10; 
		public static final int LATERAL_VELOCITY_SMALL = 2,
		LATERAL_VELOCITY = 5,
		LATERAL_VELOCITY_LARGE = 7;
		public static final int BULLET_POWER = 3,
		BULLET_POWER_LARGE = 3;

		public static final int DELTA_HEADING = 5;		
		public static final int ACCELERATION = 3;
		public static final int TIME_SINCE_DCHANGE = 4;
		public static final int NEAR_WALL = 3;

		public static int distanceIndex(double distance, int section) {
			switch (section) {
			case 1: return (int) (distance / 500);
			case 2: return (int) (distance / 250);
			default: return (int) (distance / 100);
			}
		}

		public static int lateralVelocityIndex(double lateralVelocity, int section) {
			switch (section) {
			case 1: return (int) Math.abs(lateralVelocity / 5);
			case 2: return (int) Math.abs(lateralVelocity / 2);
			default: return (int) Math.abs(lateralVelocity / 1.3);
			}
		}

		public static int bulletPowerIndex(double power, int section) {
			return (int) Math.min(2, power / 1);
			//		if (section == 1) { return (int) (power / 1); }
			//			else { return (int) (power / 0.5); }
		}

		public static int deltaHeadingIndex(double deltaHeading) {
			return (int) ((deltaHeading + 10) / 5); 
		}

		public static int accelerationIndex(double acceleration) {
			return (acceleration > 0 ? 2 : (acceleration < 0 ? 0 : 1));
		}

		public static int timeSinceDChangeIndex(int timeSinceDChange) {
			return (int) YUtils.limit(0, timeSinceDChange / 10, TIME_SINCE_DCHANGE - 1);
		}

		public static int nearWallIndex(Point2D.Double robotsLocation) {
			if ((robotsLocation.x >= YUtils.battleFieldWidth - 150 && robotsLocation.y >= YUtils.battleFieldHeight - 150)
					|| (robotsLocation.x <= 150 && robotsLocation.y >= YUtils.battleFieldHeight - 150)  
					|| (robotsLocation.x >= YUtils.battleFieldWidth - 150 && robotsLocation.y <= 150) 
					|| (robotsLocation.x <= 150 && robotsLocation.y <= 150)) {
				return 2;
			}
			else if (robotsLocation.x <= 150 || robotsLocation.y <= 150 || robotsLocation.x >= YUtils.battleFieldWidth - 150 
					|| robotsLocation.y >= YUtils.battleFieldHeight - 150) {
				return 1;
			}				
			return 0;
		}
	}
}