package DTF;

import DTF.modules.Pattern;
import DTF.modules.PatternMatcher;
import robocode.*;
import robocode.util.*;


public class VirtualBot {
	public static double error_decay=0.93;
	
	public double asof;
	public double heading;
	public double X;
	public double Y;
	public double velocity;
	public double rotation;
	public double accel;
	public boolean known;
	public double energy;
	
	public int bestMode=0;
	public double lastReverse;
	public double period;
	
	public int idleCount=0; //# of turns not moving 
	
	public String name;
	
	public boolean fired;
	public double fireEnergy;
	
	public boolean dead=false;
	
	public static final int num_modes=9*3*2;
	
	public double[] modeError;
	
	private VirtualBot lastFireState;
	
	private VirtualBot lastState;
	
	//private AdvancedRobot robot;
	
	private PatternMatcher matcher;
	public boolean usedMatcher; 
	
	//start with safe "hasn't fired yet" values
	int lastFiredTurns=-100;
	int lastFiredAtTurns=-100;
	int lastScanDist=500;	
	
	
	public VirtualBot() {
		
	}

	public VirtualBot(VirtualBot source) {
		copyFrom(source);
	}
	
	public void copyFrom(VirtualBot other) {
		asof=other.asof;
		heading=other.heading;
		X=other.X;
		Y=other.Y;
		velocity=other.velocity;
		//robot=other.robot;
		rotation=other.rotation;
		accel=other.accel;
		known=other.known;
		bestMode=other.bestMode; //make sure predictor matches
		lastReverse=other.lastReverse;
		period=other.period;
		energy=other.energy;
		idleCount=other.idleCount;
		matcher=other.matcher;
		
		lastFiredAtTurns=other.lastFiredAtTurns;
		lastFiredTurns=other.lastFiredTurns;
		lastScanDist=other.lastScanDist;
	}
	
	public void copyFrom(AdvancedRobot me) {
		asof=me.getTime();
		heading=me.getHeading();
		X=me.getX();
		Y=me.getY();
		velocity=me.getVelocity();
		//robot=me;
		
		checkVelocity();
		
		rotation=Math.signum(me.getTurnRemaining()) * (10-velocity*0.75);

		accel=0.0; //todo
		
		bestMode=0; //todo decide
		lastReverse=period=0;
		energy=me.getEnergy();
		idleCount=0;
	}
	
	private void checkVelocity() {
		//yes it happens a lot
		if(velocity<0) {
			velocity=-velocity;
			heading=Util.normalHeading(heading+180);
		}
	}
	
	private void saveState() {
		if(lastState==null) {
			lastState=new VirtualBot(this);
		} else {
			lastState.copyFrom(this);
		}
	}
	
	public void firedAt() {
		lastFiredAtTurns=(int)asof;
		if(lastFireState==null) {
			lastFireState=new VirtualBot(this);
		} else {
			lastFireState.copyFrom(this);
		}
	}
	
	//todo replace this function with virtualbullet based benchmarks
	//virtualbullet tester will need to be able to interpolate positions when age>1
	//will want to benchmark odd/even (fast/slow) bullets separately
	private void testPredictors() {
		if(lastFireState!=null && lastFireState.known) {
			int duration=(int)(asof-lastFireState.asof);
			if(duration>11 && duration<20) {
				
				if(modeError==null) modeError=new double[num_modes];
				
				VirtualBot predictor=new VirtualBot();
				int best=0;
				for(int i=0;i<num_modes;i++) {
					predictor.copyFrom(lastFireState);
					predictor.Drive(duration, i);
					double error = Util.distance(X-predictor.X,Y-predictor.Y);
					if(error>25.0) error=error*4.0-75.0; //penalty for being big enough to miss
					error=error/(double)duration;
					
					modeError[i] = modeError[i]*error_decay + error; //todo decide whether error or error*error is better
					if(modeError[i]<modeError[best]) {
						best=i;
					}
					
				}
				if(best!=bestMode) {
					//Util.me.out.println("Switching "+name+" prediction mode to "+best);
					bestMode=best;
				}
				lastFireState.known=false;
			}
		}
	}
	
	public void Scanned(ScannedRobotEvent e) {
		double scanHeading = Util.normalHeading(e.getBearing() + Util.getMe().getHeading());
		double dist = e.getDistance();
		
		saveState();
		
		asof = Util.getMe().getTime();
		velocity = e.getVelocity();
		if(velocity!=0 || lastState==null || !lastState.known) {
			//when their velocity is zero, heading may be briefly wrong and we shouldn't update it
			heading = e.getHeading();
		}
		energy=e.getEnergy();
		
		lastScanDist=(int)dist;
		
		checkVelocity(); //needed?

		
		X = Util.sin(scanHeading) * dist + Util.getMe().getX();
		Y = Util.cos(scanHeading) * dist + Util.getMe().getY();
		
		known=true;
		
		fired=false;
		
		//update accel and rotation
		if(lastState.known) {
			double age = asof - lastState.asof;
			if(age>0) {
				if(age<=8) { //todo optimize cutoff. larger values cause more confusion, espec
					double drop=lastState.energy-energy;
					if(drop>=1 && drop<=3) { //todo should lower bound be lower?
						fired=true;
						fireEnergy=drop;
						lastFiredTurns=(int)asof-1;
					}
					
					if(velocity>7.5 || (velocity>age && velocity==lastState.velocity)) {
						accel=0;
					} else if(velocity>lastState.velocity) {
						accel=1;
					} else {
						accel=-2;
					}
					
					if(velocity==0 && lastState.velocity==0) {
						idleCount++; //for sitting duck detection
					} else {
						idleCount=0;
					}
					if(velocity>0 /* || lastState.velocity>0*/) { //apparently velocity is briefly zero, so we can't require both
						//detect rotation and period
						
						rotation=Util.normalBearing(heading-lastState.heading);
						
						if(Math.abs(rotation)>=Math.min(Math.max(180-age*10,age*10+1),170)) {
							rotation=0; //assume they just switched directions
							if(accel<0) accel=1; //if they did switch, they may be speeding up.
							
							double revtime=asof-velocity;
							
							double thisPeriod=revtime-lastReverse;
							
							if(lastReverse>0 && thisPeriod>16.0 && (thisPeriod<60||thisPeriod>period*1.5)) { //todo min/maxPeriod constants
								if(period==0) {
									period=thisPeriod;
								} else {
									period=period*0.7+thisPeriod*0.3; //todo find good ratio
								}
								//Util.me.out.println("period: "+period);
							}
							lastReverse=revtime;
						}
						rotation=rotation/age;
					} else {
						rotation=0;
					}
				} else {
					accel=0; //todo guess accel based on wall distance if velocity!=8
					//guesses for specific velocities (odd vs even tells a bit)
					if((((int)velocity)&1)==1) accel=0.5; else if(velocity>1 && velocity<7) accel=-0.3;
					
					rotation=0;
					if((period>0.0||lastReverse>0.0) && asof>lastReverse+period) { //we don't know
						lastReverse=0.0;
					}
				}
			}
		}
		
		if(matcher==null) matcher=new PatternMatcher();
		matcher.addPattern(new Pattern((int)X, (int)Y, (int)heading, velocity, (int)asof, lastFiredAtTurns, lastScanDist));
		
		
		//todo replace this with virtual guns, in the process making PatternMatcher one of those guns rather than separate
		//when the matcher has no match, give it the score of the gun that would have been used otherwise.
		testPredictors();
	}
	
	//todo mode benchmarking and selection. Benchmarking happens X turns after firing.
	public void Drive(int turns) {
		if(turns==0) return;
		
		if(matcher!=null) {
			//robot.out.println("querying pattern matcher");
			Pattern p=matcher.predictFromCurrent((int)asof+turns);
			if(p!=null) {
				//robot.out.println("pattern matcher returned "+p.x+", "+p.y);
				asof+=turns;
				X=Util.clampX(p.x);
				Y=Util.clampY(p.y);
				//todo heading, velocity, accel
				usedMatcher=true;
				return;
			}
		}
		usedMatcher=false;
		Drive(turns,bestMode);
	}

	public void Drive(int turns, int mode) {
		//mode 0 - normal
		//mode 1 - evasive stop (stop if not accelerating)
		//mode 2 - evasive reverse (above, but reversing after stop)
		//mode 3 - no accel, but can decel
		//mode 4 - no accel, bounce if decel
		//mode 5 - no predict
		//mode 6 - constant speed
		//mode 7 - midway (average of speeding up to 8 and reversing to -8, with no rotation). This mode more hurt by repeated calls to Drive() than others
		//mode 8 - stop once max speed reached
		//todo seesaw
		// update num_modes constant
		
		
		int submode=mode/9; //todo this is ugly
		int submode2=submode/3;
		submode=submode%3;
		mode=mode%9;
		//submode 0: slide on wall
		//submode 1: stop on wall
		//submode 2: bounce on wall
		
		//submode2 0: moves in one direction
		//submode2 1: follows period
		if(mode==5 || idleCount>2) { // if nopredict or sitting duck. todo optimize threshold
			asof+=turns;
			return;
		}
		
		if(period==0.0 || lastReverse==0.0 || (mode!=6 && asof>=lastReverse+period && velocity>7.0) || (mode==6 && asof>=lastReverse+period*1.5)) {
			//either we don't know period,
			//or past due for a bounce and not slowing down, so assume they changed their mind
			submode2=0;
		}
		
		
		double vForward=velocity,vReverse=velocity; //for mode 7 midway
		if(mode==7) {
			rotation=0;
			accel=0;
		}
		
		double maxvel=(mode==6)?velocity:8.0;
		
		boolean nobounce=false;
		
		if(mode==1 || mode==2) {
			//assume they'll stop when they detect our shot
			if(accel==0 && velocity>0) accel=-2;
		}
		if(mode==6 || ((mode==3 || mode==4) && accel>0)) accel=0;
		for(int i=0;i<turns;i++) {
			//changes in order: heading, accel, velocity, then movement
			
			if(mode==8 && (velocity>7 || accel==0)) accel=-2;
			
			double maxRotation=10.0-0.75*Math.abs(velocity);
			
			heading=Util.normalHeading(heading + Math.min(Math.max(rotation,-maxRotation),maxRotation)); 
			if((mode==2 || mode==4) && velocity==0 && accel!=0 ) {
				heading=Util.normalHeading(heading+180);
				accel=1;
				lastReverse=asof;
			}
			if(submode2==1 && mode!=7) { //follows period
				if(asof>=lastReverse+period-velocity) {
					if(asof>=lastReverse+period) {
						//todo decel before bounce
						velocity=0;
						heading=Util.normalHeading(heading+180);
						accel=1;
						lastReverse=asof;
					} else {
						accel=-1; //decel
					}
				}
			}
			if(mode==7) {
				//average of both escape velocities
				vForward=Math.min(vForward+1.0,8.0);
				if(vReverse>0) {
					vReverse=Math.max(vReverse-2, 0.0);
				} else {
					vReverse=Math.max(vReverse-1, -8.0);
				}
				velocity=(vForward+vReverse)*0.5;
			} else {
				velocity=Util.clamp(velocity+accel,0,maxvel);
			}
			
			X+=Util.sin(heading)*velocity;
			Y+=Util.cos(heading)*velocity;
			asof+=1.0;
			
			if((submode==1 || submode==2) && !Util.inArena(X, Y)) { //if stop on wall hit
				X=Util.clampX(X);
				Y=Util.clampY(Y);
				if(submode==1) {
					velocity=0;
					if(!(mode==2 || mode==4 || rotation!=0)) return; //if they don't bounce or turn, we just stop here
				} else if(velocity>1 || accel<0) { //a cheap/bad way to make sure we don't double-bounce in one turn in mode 2/4
					//submode 2
					//todo reduce duplication of bounce code
					
					velocity=0;
					rotation=0;
					if(!nobounce) {
						heading=Util.normalHeading(heading+180);
						accel=1;
						lastReverse=asof;
					}

				}
			}
		}
		//I know it's wrong to clamp this late, or assume that they don't stop 100% when the hit a wall, but really we don't know what they plan to do.
		X=Util.clampX(X);
		Y=Util.clampY(Y);
	}
	

	
}
