package tzu.movement;

import java.util.*;
import tzu.intel.*;
import tzu.util.*;
import tzu.strategy.*;
import tzu.event.*;
import robocode.*;

/**
 * Manages the movement for an AdvancedRobot.  Provides some high-level
 * movement methods.
 */
public class MoveManager extends AbstractManager {

    int                     direction               = FORWARD;
    Bot                     target                  = null;
    static                  Random random           = new Random();
    public MoveTracker      moveTracker;


    /**
     * Create and initialize the move manager.
     */
    public MoveManager(AdvancedRobot ar) {
        super(ar, null);
        moveTracker = new MoveTracker(ar);
        myRobot.addCustomEvent(moveTracker);
    }


    /**
     * Reinitialize the move manager in second and subsequent rounds.
     */
    public void reinitialize() {
        target          = null;
        direction       = FORWARD;
        myRobot.addCustomEvent(moveTracker);
    }

    public void takeTurn() {
    }


    /**
     * Set the robot to move to point x,y using shortest distance.
     */
    public void moveTo(double x, double y) {

        setSpeed(MAX_SPEED);
        turnTo(BotMath.calcHeading(getX(), getY(), x, y));
        setMove(BotMath.calcDistance(getX(), getY(), x, y));
    }


    public void wiggleTo(double x, double y) {
        setSpeed(MAX_SPEED);
        moveTo(x,y);
        setTurn(getTurn() + wiggle()/2);
    }

    public void wiggleTo(Point p) {
        wiggleTo(p.x, p.y);
    }


    /**
     * Get away from a ramming bot.
     */
    public void evadeRam(Bot rammer) {

        if (rammer != null) {
            setSpeed(MAX_SPEED);
            turnTo(BotMath.zeroTo360(rammer.getBearing() + 180));
            setMove(SKIP);
        }
    }

    /**
     * Returns a angle between plus/minus WIGGLE_ANGLE:  This
     * can be added to robot turns to impart randomness to our
     * robot's movement.
     */
    public static double wiggle() {
        return (WIGGLE_ANGLE -
            (WIGGLE_ANGLE * 2 * random.nextDouble()));
    }


    /**
     * Returns the shortest turn to align this robot with the
     * specified heading without reversing current direction of movement.
     */
    public double shortestTurn(double heading) {
        return BotMath.plusMinus180(getRealHeading(), heading);
    }


    /**
     * Turns the robot towards the specified heading and changes
     * the direction of movement if required.
     */
    public void turnTo(double heading) {

        double turn = shortestTurn(BotMath.zeroTo360(heading));

        if (BotMath.abs(turn) > 90.0) {
            changeDirection();
            turn += (turn > 90.0 ? -A_180 : +A_180);
        }
        setTurn(turn);
    }


    /**
     * Change from forward to reverse or vise versa.
     */
    public void changeDirection() {

        if (direction * myRobot.getVelocity() >= 0.0) {
            direction *= -1;
            setMove(getMove());
        }
    }


    /**
     * Set the maximum speed of this robot.
     */
    public void setSpeed(double speed) {

        if (speed == MAX_SPEED && myRobot.getTime() % 2 == 0) {
            speed = ALMOST_MAX_SPEED;
        }
        myRobot.setMaxVelocity(speed);
    }


    /**
     * Get the absolute speed of this robot.
     */
    public double getSpeed() {
        return BotMath.abs(myRobot.getVelocity());
    }


    /**
     * Set the robot to move a specified distance.
     */
    public void setMove(double distance) {
        myRobot.setAhead(distance * direction);
    }


    /**
     * Get the amount of robot movement remaining.
     */
    public double getMove() {
        return BotMath.abs(myRobot.getDistanceRemaining());
    }


    /**
     * Set the robot to turn the specified number of degrees.
     */
    public void setTurn(double angle) {
        myRobot.setTurnRight(angle);
    }


    /**
     * Get the amount of robot turn remaining.
     */
    public double getTurn() {
        return myRobot.getTurnRemaining();
    }


    public int getDirection() {
        direction = (myRobot.getVelocity() < 0.0) ? REVERSE : FORWARD;
        return direction;
    }


    /**
     * Return the direction this robot is moving towards.
     */
    public double getRealHeading() {

        if (myRobot.getVelocity() < 0.0) {
            direction = REVERSE;
            return (myRobot.getHeading() + A_180) % A_360;
        }
        direction = FORWARD;
        return myRobot.getHeading();
    }

    public boolean headedForWall() {
        return BattleField.headedForWall(getX(), getY(), getRealHeading());
    }

    public double getX() { return myRobot.getX(); }
    public double getY() { return myRobot.getY(); }
}
