package dam;

import java.io.Serializable;
import robocode.*;

import dam.util.*;
/**
 * BasicTarget.java
 * This is the superclass of Target - implements the barebones stuff
 * needed for targetting, without all the refinements of Target
 * 
 * @author David Mold
 *
 * Jun 28, 2002
 */
public class BasicTarget implements Serializable {

	protected String name = "NONAME";
	protected double bearing = 0;
	protected double heading = 0;
	protected double headingChange = 0;
	protected long lastScanTime = 0;
	protected double velocity = 0;
	public double x = 0;
	public double y = 0;
	protected double distance = 100000;
	protected double energy = 0;
	/**
	* The buffer that holds an array of the target's last 200 positions, and
	* other useful information
	*/
	private PatternBuffer patternBuffer = new PatternBuffer();
	
	/**
	 * Constructor for BasicTarget.
	 */
	public BasicTarget() {
		
	}
	
	public BasicTarget createTarget(AdvancedRobot bot)
	{
		BasicTarget t = new BasicTarget();
		t.headingChange = 0;
		t.x = bot.getX();
		t.y = bot.getY();
		t.bearing = BotMath.normaliseBearing(heading - bot.getHeadingRadians());
		t.lastScanTime = bot.getTime();
		t.velocity = bot.getVelocity();
		t.distance = distance;
		t.energy = bot.getEnergy();
		t.heading = bot.getHeadingRadians();
		return t;
	}
	
	public void update(ScannedRobotEvent e, AdvancedRobot r) {
		//long diff = r.getTime() - getLastScanTime();
		if(name.equals("NONAME")) name = e.getName();
		distance = e.getDistance();
		double absbearing = (r.getHeadingRadians() + e.getBearingRadians()) % (2 * Math.PI);
		x = r.getX() + Math.sin(absbearing) * distance;
		y = r.getY() + Math.cos(absbearing) * distance;
		headingChange = BotMath.normaliseBearing((e.getHeadingRadians() - heading) / (r.getTime() - lastScanTime));
		bearing = e.getBearingRadians();
		heading = e.getHeadingRadians();
		lastScanTime = r.getTime();
		velocity = e.getVelocity();
		energy = e.getEnergy();
		//if(diff < 4) 
		patternBuffer.addReading(headingChange, velocity, lastScanTime);
	}
	
	public PatternBuffer getPatternBuffer() {
		return patternBuffer;
	}

	/**
	 * Returns the bearing.
	 * @return double
	 */
	public double getBearing() {
		return bearing;
	}

	/**
	 * Returns the distance.
	 * @return double
	 */
	public double getDistance() {
		return distance;
	}

	/**
	 * Returns the energy.
	 * @return double
	 */
	public double getEnergy() {
		return energy;
	}

	/**
	 * Returns the heading.
	 * @return double
	 */
	public double getHeading() {
		return heading;
	}

	/**
	 * Returns the headingChange.
	 * @return double
	 */
	public double getHeadingChange() {
		return headingChange;
	}

	/**
	 * Returns the lastScanTime.
	 * @return long
	 */
	public long getLastScanTime() {
		return lastScanTime;
	}

	/**
	 * Returns the name.
	 * @return String
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the velocity.
	 * @return double
	 */
	public double getVelocity() {
		return velocity;
	}

	/**
	 * Returns the x.
	 * @return double
	 */
	public double getX() {
		return x;
	}

	/**
	 * Returns the y.
	 * @return double
	 */
	public double getY() {
		return y;
	}

	
	
	public boolean equals(Object o) {
		if (o instanceof BasicTarget) {
			BasicTarget t = (BasicTarget) o;
			return t.getName().equals(getName());
		}
		return false;
	}
	
	public double getAccuracy(){
		return 0;
	}

}
