package romz.component.targeting.guessfactor;

import static romz.math.RobotMath.direction;
import static romz.math.RobotMath.limit;
import static romz.math.RobotMath.maxEscapeAngle;
import static romz.math.RobotMath.round;

import java.util.List;

import robocode.util.Utils;
import romz.component.targeting.base.AbstractTargeting;
import romz.model.wave.Wave;

public class DistanceSegmentedGuessFactorTargeting extends AbstractTargeting {
	
	/** The maximum distance at which we fire with maximum power */
	public static final double MAX_POWER_DISTANCE = 300;
	
	/** Number of fire angle sectors for hit stats */
	public static final int NB_FIRE_ANGLE_SECTORS = 9;
	
	/** Half the range of the total fire sector */
	public static final int HALF_FIRE_ANGLE_RANGE = (NB_FIRE_ANGLE_SECTORS - 1) / 2;
	
	/** Distance sectors to segment hit data */
	public static final double[] DISTANCE_SECTOR_LIMITS = {180, 250};
	
	private int[][] enemyHitProfile = new int[DISTANCE_SECTOR_LIMITS.length + 1][NB_FIRE_ANGLE_SECTORS];
	
	private int currentEnemyDistanceSector;
	private int[] currentEnemyHitProfile;
	
	@Override
	public void handleWaveHit(List<Wave> hittingWaves) {
		for (Wave wave : hittingWaves) {
			double guessFactor = limit(-1, wave.hitOffsetAngle / wave.maxEscapeAngle, 1);
			int hitGuessFactorIndex = getFireSector(guessFactor);
			int hitDistanceSector = getDistanceSector(wave.startEnemyDistance);
			enemyHitProfile[hitDistanceSector][hitGuessFactorIndex]++;
		}
	}
	
	public double selectFireAngle(double firePower) {
		double fireAngle;
		currentEnemyDistanceSector = getDistanceSector(situation.enemy.distance);
		currentEnemyHitProfile = enemyHitProfile[currentEnemyDistanceSector];
		if (useGuessFactorTargeting()) {
			fireAngle = bestGuessFactorAngle(firePower);
		} else {
			fireAngle = circularAcceleratedAngle(firePower);
		}
		return fireAngle;
	}

	private boolean useGuessFactorTargeting() {
		return hasHitData() && situation.enemy.velocity != 0;
	}

	private boolean hasHitData() {
		for (int index = 0; index < currentEnemyHitProfile.length; index++) {
			if (currentEnemyHitProfile[index] > 0) {
				return true;
			}
		}
		return false;
	}

	private double bestGuessFactorAngle(double firePower) {
		int bestIndex = getBestGuessFactorIndex(currentEnemyDistanceSector);
		double bestGuessFactor = (double) (bestIndex - HALF_FIRE_ANGLE_RANGE) / HALF_FIRE_ANGLE_RANGE;
		double angleOffset = direction(situation.enemy.heading, situation.enemy.absoluteBearing) * bestGuessFactor * maxEscapeAngle(firePower);
		return Utils.normalRelativeAngleDegrees(situation.enemy.absoluteBearing + angleOffset);
	}

	private int getBestGuessFactorIndex(int distanceIndex) {
		int bestIndex = 0;
		for (int index = 0; index < currentEnemyHitProfile.length; index++) {
			if (currentEnemyHitProfile[index] > currentEnemyHitProfile[bestIndex]) {
				bestIndex = index;
			}
		}
		return bestIndex;
	}

	private int getFireSector(double guessFactor) {
		return HALF_FIRE_ANGLE_RANGE + round(guessFactor * HALF_FIRE_ANGLE_RANGE);
	}
	
	private int getDistanceSector(double distance) {
		for (int i = 0; i < DISTANCE_SECTOR_LIMITS.length; i++) {
			if (distance < DISTANCE_SECTOR_LIMITS[i]) {
				return i;
			}
		}
		return DISTANCE_SECTOR_LIMITS.length;
	}

	@Override
	protected boolean selectFire() {
		return true;
	}
	
}
