package ags.rougedc.gun;

import ags.rougedc.base.Rules;
import ags.rougedc.gun.util.*;
import ags.rougedc.robots.*;
import ags.rougedc.waves.SelfWave;
import ags.utils.points.*;
import java.util.List;
import java.util.LinkedList;

public class PatternPredictor implements Predictor {
    public final class HistoryPoint {
        public final int accelIndex;
        public final int turnIndex;
        public final double accel;
        public final double turn;
        private HistoryPoint (double turn, double accel) {
            this.turn = turn;
            this.accel = accel;
            this.turnIndex = PatternNode.getTurnIndex(turn);
            this.accelIndex = PatternNode.getAccelIndex(accel);
        }
    }
    
    private final Rules rules;
    private static final int DEPTH = 15;    
    private static final PatternNode root = new PatternNode(0,0);
    private final LinkedList<HistoryPoint> history = new LinkedList<HistoryPoint>();
    private MovementProfile profile;
    
    public PatternPredictor(Rules rules) {
        this.rules = rules;
    }
    
    public MovementProfile getPredictedProfile() {
        return profile;
    }
    
    public void remember(EnemyRobot target) {
        final double accel = target.getAcceleration();
        final double turn = target.getAngularVelocity();
        final HistoryPoint newpoint = new HistoryPoint(turn, accel);
        //System.out.println("Turn: "+newpoint.turnIndex+"  Accel: "+newpoint.accelIndex);
        history.add(newpoint);
        if (history.size() > DEPTH+1)
            history.removeFirst();
        
        root.addHistory(new LinkedList<HistoryPoint>(history), turn, accel);
    }

    public void run(AbsolutePoint nextLocation, EnemyRobot target,
            double bulletpower) {        
        profile = new MovementProfile();
        final SelfWave wave = new SelfWave(nextLocation, target, bulletpower);
        
        simulate(wave, target, history, profile);
    }
    
    private void simulate(SelfWave wave, VirtualRobot bot, LinkedList<HistoryPoint> history, MovementProfile profile) {
        history = new LinkedList<HistoryPoint>(history);
        while (true) {
            List<PatternNode> nodes = findLongestMatch(history, bot.getVelocity().magnitude);
            if (nodes == null)
                return; // Just in case

            PatternNode bestnode = null;
            for (PatternNode node : nodes) {
                if (bestnode == null || node.getHitCount() > bestnode.getHitCount())
                    bestnode = node;
            }

            if (bestnode == null)
                return; // Just in case

            final PatternRobot sim = new PatternRobot(rules, bot, bestnode.TURN, bestnode.ACCEL);
            bot = sim;
            sim.move();
            wave.move();
            wave.checkHit(sim);
            if (wave.expired(sim)) {
                profile.add(wave.getHitRange(), 1);
                return;
            }
             
            final HistoryPoint newpoint = new HistoryPoint(bestnode.TURN, bestnode.ACCEL);
            history.add(newpoint);
            if (history.size() > DEPTH+1)
                history.removeFirst();
        }
    }
    
    private List<PatternNode> findLongestMatch(LinkedList<HistoryPoint> thishistory, double velocity) {
        LinkedList<HistoryPoint> h = new LinkedList<HistoryPoint>(thishistory);
        for (int i=h.size()-1; i>0; i--) {
            PatternNode currentnode = root;
            h.removeFirst();
            for (HistoryPoint historypoint : h) {
                currentnode = currentnode.getChild(historypoint.turnIndex, historypoint.accelIndex);
                if (currentnode == null)
                    break;
            }
            if (currentnode != null) {
                final List<PatternNode> nodelist = currentnode.getValidChildren(velocity);
                if (nodelist.size() > 0) {
                    //System.out.println("Depth found: "+i+"  Hits:"+currentnode.getHitCount()+"  Options:"+nodelist.size());
                    return nodelist;
                }
            }
        }
        //System.out.println("No match found!");
        return null;
    }

    public void waveFired(SelfWave wave) {}
    public void waveHit(SelfWave wave) {}
}
