package ag;
import java.awt.Color;

import robocode.*;
import ag.battledata.DataBattle;
import ag.battledata.DataRobot;
import ag.battledata.DataVector;
import ag.movement.Movement;
import ag.targeting.TargetingNeural;
import ag.team.*;

import java.awt.Graphics2D;


/**
 * The Robot
 * @author Andree Grosse
 */
public class Gir extends TeamRobot
{

	private final int minQuota = 30000;
	private final int minDataQuota = 100000;
	
	// ---- global variables ----
	private DataBattle mDataBattle;
	private DataRobot mData;
		
	private TargetingNeural mTargeting;
	private Movement mMovement;
	
	private FileManager mFileManager;
	private int mNumTeammates;
	
	private Color[] mColors;
	boolean mWriteData;
	/// -------------------------
		
	/**
	 * initializes Gir
	 */
	private void init(){
		
		setBodyColor(new Color(224, 0, 0));
		setBulletColor(Color.WHITE);
		setGunColor(Color.BLACK);
		setRadarColor(Color.BLACK);
		setScanColor(Color.RED);
				

		// Intialise DataBattle with Gir's name
		mDataBattle = new DataBattle(this);
		mTargeting = new TargetingNeural(this);
		mMovement = new Movement(this);
		mFileManager = new FileManager(this);
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		setAdjustRadarForRobotTurn(true);
		mNumTeammates = 0;
		
		mColors = new Color[]{Color.BLUE, Color.GREEN, Color.CYAN};
		
		mWriteData = true;
		
		Team_Message msg = new Team_Message(Team_Message.Message.NEWTEAMMEMBER, 
				null);
		try{
			broadcastMessage(msg);
		}catch(Exception e){}
			
		System.out.println("GIR, reporting for duty!");
		System.out.println("Data Quota left: " + getDataQuotaAvailable());
	}
	
	/**
	 * run: Gir's default behavior
	 */	
	public void run() {
		// initialize
		init();
		
		while(true) {
			
		//--------------------------------------------------------	
			//System.out.println("---- (Gir) Tick starts : " + getTime());
			// make a Snapshot of yourself
			mData = mDataBattle.addDataRobotSnapshot(getName(), getTime(), getX(), getY(), new DataVector(getHeadingRadians()), getVelocity(), getEnergy());
			// send my data
			
			Object[] value = new Object[]{new Team_RobotData(getName(), getTime(),
					getX(), getY(), getHeadingRadians(), getVelocity(), getEnergy())};
			
			Team_Message msg = new Team_Message(Team_Message.Message.ROBOTDATA, value);
			try{
				broadcastMessage(msg);
			}catch(Exception e){}
			
		//--------------------------------------------------------

			//System.out.println("Targeting...");
			mTargeting.run();
			//System.out.println("Movement...");
			mMovement.run();
			
			mDataBattle.removeOldData();			
		//--------------------------------------------------------	
			//System.out.println("---- Tick ends: " + getTime());
			execute();
		//--------------------------------------------------------	
		}	
		
	}
	
	// ############# getter & setter ####################
	
	/**
	 * Returns the DataRobot object for Gir
	 * @return The DataRobot object for Gir
	 */
	public DataRobot getData(){
		return mData;
	}
	
	/**
	 * Returns the FileManager object
	 * @return The FileManager
	 */
	public FileManager getFileManager(){
		return mFileManager;
	}
	
	/**
	 * Returns the DataBattle object where all robot informations
	 * are managed
	 * @return The DataBattle object
	 */
	public DataBattle getDataBattle(){
		return mDataBattle;
	}
	
	/**
	 * Returns the current target
	 * @return the current target
	 */
	public DataRobot getTarget(){
		return mTargeting.getTarget();
	}
	
	/**
	 * Returns the current number of team members
	 * @return current number of team members
	 */
	public int getNumTeammates(){
		return mNumTeammates;
	}
	
	/**
	 * Returns the current number of enemies
	 * @return current number of enemies
	 */
	public int getEnemies(){
		return getOthers() - mNumTeammates;
	}
	
	/**
	 * Returns the Movement object
	 * @return the Movement object
	 */
	public Movement getMovement(){
		return mMovement;
	}
	
	/**
	 * Returns a color
	 * @param i the number of the color in Gir's colorset to be used
	 * @return a color
	 */
	public Color getColor(int i){
		if(i < 0 || i > mColors.length)
			return mColors[mColors.length - 1];
		return mColors[i];
	}
	
	/**
	 * Returns the boolean which saves if to write training data to disk
	 * @return The mWriteData boolean
	 */
	public boolean getWriteData(){
		return mWriteData;
	}
	
	// ############################## reaction methods ##############################

	/**
	 * This method will be called by the Robocode engine when 
	 * Gir sees another robot 
	 */
	public void onScannedRobot(ScannedRobotEvent e){	
						
		//	calculate robots Position
		double alpha = e.getBearingRadians() + getHeadingRadians();
		double x = getX() + e.getDistance() * Math.sin(alpha);
		double y = getY() + e.getDistance() * Math.cos(alpha);
		
		getDataBattle().addDataRobotSnapshot(e.getName(), getTime(), x, y, new DataVector(e.getHeadingRadians()), e.getVelocity(), e.getEnergy());
		
		mTargeting.onScannedRobot(e);
		//mMovement.onScannedRobot(e);
	}
	
	/**
	 * This method will be called by the Robocode engine when a bullet 
	 * hits another robot
	 */
	public void onBulletHit(BulletHitEvent e) {
		mTargeting.onBulletHit(e);	
		mMovement.onBulletHit(e);
	}
	
	/**
	 * This method will be called by the Robocode engine when one 
	 * of Girs bullets misses (hits a wall) 
	 */
	public void onBulletMissed(BulletMissedEvent e){
		mTargeting.onBulletMissed(e);
	}
	
	/**
	 * This method will be called by the Robocode engine when Gir 
	 * is hit by a bullet
	 */
	public void onHitByBullet(HitByBulletEvent e) {
	     mMovement.onHitByBullet(e);
	}
	
	/**
	 * This method will be called by the Robocode engine when one of
	 * Gir bullets hits another bullet
	 */
	public void onBulletHitBullet(BulletHitBulletEvent e){
		mMovement.onBulletHitBullet(e);
	}
	
	/**
	 * This method will be called by the Robocode engine when Gir 
	 * wins a battle 
	 */
	public void onWin(WinEvent e) {
		System.out.println("MmmmMmhh...");	
		//((DataRobot)mDataBattle.mRobots.get(0)).printData();
	}
	
	/**
	 * This method will be called by the Robocode engine when another 
	 * robot dies
	 */
	public void onRobotDeath(RobotDeathEvent e){
		DataRobot dead = getDataBattle().getRobotByName(e.getName());
		mTargeting.onRobotDeath(dead);
		mMovement.onRobotDeath(dead);
		dead.setDead(true);
		
	}
	
	/**
	 * This method will be called by the Robocode engine when Gir dies 
	 */
	public void onDeath(DeathEvent e){
		System.out.println("----= Finally! =----");	
		//((DataRobot)mDataBattle.mRobots.get(0)).printData();
		try{
			broadcastMessage(new Team_Message(Team_Message.Message.TEAMMEMBERDIED, 
							new Boolean[]{false}));
		}catch(Exception ex){}
		
		mTargeting.onDeath(e);
		mMovement.onDeath(e);
	}	
	
	/**
	 * This method will be called by the Robocode engine when Gir collides
	 * with another robot.
	 */
	public void onHitRobot(HitRobotEvent e){
		mMovement.onHitRobot(e);
	}
	
	/**
	 * This method will be called by the Robocode engine when Gir 
	 * received a team message
	 */	
	public void onMessageReceived(MessageEvent e){
		Team_Message msg = (Team_Message)e.getMessage();
		String out = "TEAM: Message received (" + getTime() + "): ";
		Object[] value = msg.getValue();
		switch(msg.getMessage()){

			case ROBOTDATA: 	//out += "RobotData for ";	
								Team_RobotData data = (Team_RobotData)value[0];
								//out +=  data.getTick() + " for " + data.getName();
								mDataBattle.addDataRobotSnapshot(data.getName(), data.getTick(), 
										data.getX(), data.getY(), new DataVector(data.getHeading()),
										data.getVelocity(), data.getEnergy());
								//System.out.println(out);
								break;
			case TEAMTARGET:	if(value == null){
									out += "TEAMTARGET = null";
									mTargeting.setTeamTarget(null);
									mMovement.getMoveTeam().setTeamTarget(null);
								}
								else{
									String str = (String)value[0];
									out += "TEAMTARGET = "  + str;
									DataRobot target = getDataBattle().getRobotByName(str);
									
									mTargeting.setTeamTarget(target);
									mMovement.getMoveTeam().setTeamTarget(target);
								}
								System.out.println(out);
								break;
			case NEWTEAMMEMBER: 	out += "New Teammate";
									mNumTeammates++;
									String str = (String)value[0];
									if(str != null){
										DataRobot robot = getDataBattle().addNewRobot(
												str, 100.0, !isTeammate(str));
										if((Boolean)value[1])
											mMovement.getMoveTeam().setLeader(robot);
										out += " (Leader)";
									}
									System.out.println(out);
									break;
			case TEAMMEMBERDIED:	out += "Teammate died";
									mNumTeammates--;
									// If this team member chose the team target, reset the
									// team target to null
									if((Boolean)value[0]){
										mTargeting.setTeamTarget(null);
										mMovement.getMoveTeam().setTeamTarget(null);
										out += " (Leader)";
									}			
									System.out.println(out);												
									break;
			case BULLETHIT:			out += "Team bullet hit ";
									str = (String)value[0];
									Double power = (Double)value[1];
									out +=  str;
									mMovement.onMessageBulletHit(str, power);
									System.out.println(out);
									break;
			case HITBYBULLET:		out += "Teammate hit by bullet from ";
									Team_BulletData bd = (Team_BulletData)value[0];
									out += bd.getOwner();
									mMovement.onMessageHitByBullet(bd);	
									System.out.println(out);
									break;
			case FIREDBULLET:		//out += "Teammate fired a bullet";
									bd = (Team_BulletData)value[0];
									mMovement.onMessageBulletFired(bd);	
									//System.out.println(out);
									break;
			case ROBOTCOLLISION:	out += "Collision with ";
									str = (String)value[0];
									out += str;
									mMovement.onMessageRobotCollision(str);
									System.out.println(out);
									break;
		}		
	}

	
	// ####################### setter & getter #######################
		
	// ###############################################################
	
	/**
	 * This method will be called by the Robocode engine to add custom
	 * painting
	 */
	public void onPaint(Graphics2D g){
		mMovement.onPaint(g);
		mTargeting.onPaint(g);
	}
	
	// ####################################### helpers #######################################
	
	/**
	 * Returns if there is still space available on the disk.
	 * @return true if there is enough space available,, otherwise false
	 */
	public boolean isSpaceAvailable(){
		return getDataQuotaAvailable() > minQuota;
	}

	/**
	 * Returns if there is still enough space available on the disk
	 * to store training data.
	 * @return true if there is enough space available,, otherwise false
	 */
	public boolean isDataSpaceAvailable(){
		return getDataQuotaAvailable() > minDataQuota;
	}
	
	/**
	 * Returns if this is the last round
	 * @return true if this is the last round, otherwise false
	 */
	public boolean isLastRound(){
		return getRoundNum() == getNumRounds() - 1;
	}
	/*

	private double getAngleDiff(double a, double b){
		double diff = Math.abs(a - b);
		return Math.min(Math.abs(diff), Math.abs(2*Math.PI - diff));	
	}

	 */

	
}
