package fromHell.targeting;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import robocode.AdvancedRobot;
import robocode.util.Utils;
import fromHell.utils.FHUtils;
import static fromHell.utils.FHUtils.*;

public class GFTWave {
    private static final int ACCEL_INDEXES = 3;
    public static final double[] DISTANCE_SLICES = { 150, 300, 450, 600 };
    public static final double[] DISTANCE_SLICES_FASTER = { 250, 450 };
    public static final double[] VELOCITY_SLICES = { 1, 3, 5, 7 };
    public static final double[] VELOCITY_SLICES_FASTER = { 3, 6 };
    public static final double[] WALL_SLICES = { 0.15, 0.35, 0.55, 0.75 };
    public static final double[] WALL_SLICES_FASTER = { 0.35, 0.8 };
    public static final double[] WALL_SLICES_REVERSE = { 0.4, 0.8 };
    public static final double[] WALL_SLICES_REVERSE_FASTER = { 0.6 };
    public static final double[] TIMER_SLICES = { 0.2, 0.6, 1.2 };
    public static final double[] TIMER_SLICES_FASTER = { 0.6 };
    private static final int BINS = 37;
    public static final int MIDDLE_BIN = (BINS - 1) / 2;
    
    private static AdvancedRobot robot;
    private static double[][][][][][][] factorsFaster = new double[DISTANCE_SLICES_FASTER.length + 1][VELOCITY_SLICES_FASTER.length + 1][ACCEL_INDEXES][TIMER_SLICES_FASTER.length + 1][WALL_SLICES_FASTER.length + 1][WALL_SLICES_REVERSE_FASTER.length + 1][BINS];
    private static double[][][][][][][] factorsSlower = new double[DISTANCE_SLICES.length + 1][VELOCITY_SLICES.length + 1][ACCEL_INDEXES][TIMER_SLICES.length + 1][WALL_SLICES.length + 1][WALL_SLICES_REVERSE.length + 1][BINS];
    public static List<GFTWave> waves;

    private double bulletVelocity;
    private double distanceFromGun;
    private double startBearing;
    private double bearingDirection;
    private Point2D gunLocation;
    private static Point2D targetLocation;
    private static Point2D startTargetLocation = new Point2D.Double();
    private long startTime;
    public int accelIndex;
    public int distanceIndex;
    public int distanceIndexFaster;
    public int velocityIndex;
    public int velocityIndexFaster;
    public int velocityChangedIndex;
    public int velocityChangedIndexFaster;
    public int reverseWallIndex;
    public int reverseWallIndexFaster;
    public int wallIndex;
    public int wallIndexFaster;
    public double weight = 1;

    static void init() {
	waves = new ArrayList<GFTWave>();
    }

    public GFTWave(AdvancedRobot robot) {
	GFTWave.robot = robot;
    }

    static void updateWaves() {
	List<GFTWave> reap = new ArrayList<GFTWave>();
	Iterator<GFTWave> iterator = waves.iterator();
	while(iterator.hasNext()) {
	    GFTWave wave = iterator.next();
	    wave.setDistanceFromGun((robot.getTime() - wave.getStartTime()) * wave.getBulletVelocity());
	    if (wave.passed(WALL_MARGIN)) {
		if (wave.getRobot().getOthers() > 0) {
		    wave.registerVisit();
		}
		reap.add(wave);
	    }
	}
	iterator = reap.iterator();
	while(iterator.hasNext()){
	   waves.remove(iterator.next());
	}
    }

    void registerVisit() {
	int index = Math.max(1, visitingIndex());
	double[][] buffers = statBuffers();
	for (int i = 0; i < buffers.length; i++) {
	    registerVisit(buffers[i], index);
	}
    }

    void registerVisit(double[] buffer, int index) {
	buffer[0]++;
	for (int i = 1; i < BINS; i++) {
	    buffer[i] *= 1 - weight / 600;
	}
	for (int i = 1; i < BINS; i++) {
	    buffer[i] += (weight / 600) / (Math.pow(i - index, 2) + 1);
	}
    }

    int mostVisited() {
	double waveCount = waves.size();
	int mostVisitedIndex = MIDDLE_BIN + MIDDLE_BIN / 5;
	double most = 0;
	for (int i = 1; i < BINS; i++) {
	    double visits = 0;
	    double[][] buffers = statBuffers();
	    for (int j = 0; j < buffers.length; j++) {
		visits += waveCount * buffers[j][i] / Math.max(1, buffers[j][0]);
	    }
	    if (visits > most) {
		mostVisitedIndex = i;
		most = visits;
	    }
	}
	return mostVisitedIndex;
    }

    double[][] statBuffers() {
	return new double[][] { factorsFaster[distanceIndexFaster][velocityIndexFaster][accelIndex][velocityChangedIndexFaster][wallIndexFaster][reverseWallIndexFaster], factorsSlower[distanceIndex][velocityIndex][accelIndex][velocityChangedIndex][wallIndex][reverseWallIndex] };
    }

    public double maxEscapeAngle() {
	return FHUtils.maxEscapeAngle(bulletVelocity) * 1.2;
    }

    public boolean passed(double distanceOffset) {
	return distanceFromTarget() < distanceOffset;
    }

    public int visitingIndex() {
	return visitingIndex(targetLocation);
    }

    public int visitingIndex(Point2D location) {
	return visitingIndex(FHUtils.absoluteBearing(gunLocation, location));
    }

    public int visitingIndex(double bearing) {
	return (int) FHUtils.minMax(Math.round(((Utils.normalRelativeAngle(bearing - startBearing)) / bearingDirection) + MIDDLE_BIN), 1, BINS - 1);
    }

    public double distanceFromTarget(Point2D location, int timeOffset) {
	return gunLocation.distance(location) - distanceFromGun - (double) timeOffset * bulletVelocity;
    }

    public double distanceFromTarget(int timeOffset) {
	return distanceFromTarget(targetLocation, timeOffset);
    }

    public double distanceFromTarget() {
	return distanceFromTarget(0);
    }

    public void setStartTime(long time) {
	this.startTime = time - 1;
    }

    public double getStartTime() {
	return this.startTime;
    }

    public void setDistanceFromGun(double distance) {
	this.distanceFromGun = distance;
    }

    public double travelTime() {
	return distanceFromTarget(0) / bulletVelocity;
    }

    public AdvancedRobot getRobot() {
	return robot;
    }

    public void setBulletVelocity(double bulletVelocity) {
	this.bulletVelocity = bulletVelocity;
    }

    public double getBulletVelocity() {
	return this.bulletVelocity;
    }

    public void setStartBearing(double startBearing) {
	this.startBearing = startBearing;
    }

    public void setOrbitDirection(double bearingDirection) {
	this.bearingDirection = bearingDirection;
	if (bearingDirection == 0) {
	    this.bearingDirection = 0.73;
	}
    }

    public void setGunLocation(Point2D gunLocation) {
	this.gunLocation = gunLocation;
    }

    public void setTargetLocation(Point2D targetLocation) {
	GFTWave.targetLocation = targetLocation;
	GFTWave.startTargetLocation.setLocation(targetLocation);
    }

    public double wallDistance(double direction, Rectangle2D fieldRectangle) {
	double targetDistance = distanceFromTarget();
	for (double walldistance = 0; walldistance < 2; walldistance += 0.01) {
	    if (!fieldRectangle.contains(FHUtils.project(gunLocation, startBearing + MIDDLE_BIN * bearingDirection * direction * walldistance, targetDistance))) {
		return walldistance;
	    }
	}
	return 2;
    }
}
