package ags.rougedc.movement;

import ags.rougedc.robots.*;
import ags.utils.points.*;
import ags.rougedc.waves.EnemyWave;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;

/**
 * Precise prediction for surfing
 * 
 * @author Alexander Schultz
 */
public class SurfingPredictor extends VirtualRobotPredictor {
    private final List<EnemyWave> waves;
    public final List<EnemyWave> passedWaves;
    public final RobotDriver driver;
    public final List<RobotDriver> drivers;
    public final Map<EnemyWave, double[]> waveDanger;
    public final AbsolutePoint fieldcorner;
    public final long timestart;

    public SurfingPredictor(VirtualRobot robot, RobotDriver driver, List<RobotDriver> drivers, List<EnemyWave> waves, Map<EnemyWave, double[]> waveDanger, List<EnemyWave> passedWaves, AbsolutePoint fieldcorner, long timestart) {
        super(robot, fieldcorner);
        this.fieldcorner = fieldcorner;
        this.driver = driver;
        this.drivers = drivers;
        this.timestart = timestart;
        
        // Copy the waves! 
        this.waveDanger = new HashMap<EnemyWave, double[]>();
        this.waves = new ArrayList<EnemyWave>(waves.size());
        this.passedWaves = new ArrayList<EnemyWave>();
        for (EnemyWave wave : waves) {
            EnemyWave newwave = wave.clone();
            this.waves.add(newwave);
            if (passedWaves != null && passedWaves.contains(wave))
                this.passedWaves.add(newwave);
        }
    }
    
    @Override
    public long getDeltaTime() {
        return super.getDeltaTime() + timestart;
    }

    @Override
    public double getAcceleration() {
        return driver.getAcceleration();
    }

    @Override
    public double getAngularVelocity() {
        return driver.getAngularVelocity();
    }

    public double getDanger(double maxdanger) {
        double danger = 0;
        
        while(waves.size() > 0 && danger <= maxdanger) {
            driver.run(this, waves);
            move();
            
            //double tickweight = 1.0/Math.pow(1.02, getDeltaTime());
            //double tickweight = 1.0/(1+(getDeltaTime()-1));
            double tickweight = 1.0;
            
            boolean wavePassed = false;
            for (java.util.Iterator<EnemyWave> iter = waves.iterator(); iter.hasNext(); ) {
                EnemyWave wave = iter.next();
                wave.move();
                wave.checkHit(this);
                
                double[][] dangerpoints = wave.getDangerPoints();
                
                // Set up the pointdanger array
                double[] pointdanger = waveDanger.get(wave);
                if (pointdanger == null) {
                    pointdanger = new double[dangerpoints[1].length];
                    waveDanger.put(wave, pointdanger);
                }
                
                double waveweight = (robocode.Rules.getBulletDamage(wave.getPower())-wave.getPower())*tickweight;
                
                for (int x = 0; x < dangerpoints[1].length; x++) {
                    double p = dangerpoints[0][x]; double pointweight = dangerpoints[1][x];
                    if ((wave.getNewHitRange() != null)) {
                        if (wave.getLastHitRange() == null || !wave.getLastHitRange().intersects(p)) {
                            double distanceoutside = 10*wave.getNewHitRange().distanceOutside(p);
                            if (distanceoutside > 0)
                                distanceoutside++;
                            double distanceweight = 1/((1+distanceoutside*distanceoutside));
                            double thisdanger = pointweight*waveweight*distanceweight;
                            if (thisdanger > pointdanger[x]) {
                                danger += (thisdanger - pointdanger[x]);
                                pointdanger[x] = thisdanger;
                            }
                        }
                    }
                }
                
                if (wave.getRadius() >= getLocation().distance(wave.getOrigin()) && !passedWaves.contains(wave)) {
                    passedWaves.add(wave);
                    wavePassed = true;
                }
                
                if (wave.expired(this)) {
                    iter.remove();
                }
            } // Done waves
            
            if (danger > maxdanger)
                return danger;
            
            // A wave has passed us. Give the option of changing directions!
            if (wavePassed) {
                double lowestDanger = maxdanger-danger;
                for (RobotDriver newdriver : drivers) {
                    SurfingPredictor predictor = new SurfingPredictor(this, newdriver, drivers, waves, waveDanger, passedWaves, fieldcorner, getDeltaTime()); 
                    double newdanger = predictor.getDanger(lowestDanger);
                    if (newdanger < lowestDanger) {
                        lowestDanger = newdanger;
                    }
                }
                danger += lowestDanger;
                return danger;
            }
        }
        
        return danger;
    }
}
