package xander.core.math;

import xander.core.Resources;

public class RCPhysics {
	
	public static final double MAX_SPEED = 8d;			// max robot speed
	public static final double MAX_TURN_RATE = 10d;		// max robot turn rate (at 0 velocity)	
	public static final double MAX_FIRE_POWER = 3d;		// max gun fire power
	public static final double MIN_FIRE_POWER = 0.1d;   // min gun fire power
	public static final double ROBOT_WIDTH = 40d;       // width of robot in pixels
	public static final double ROBOT_HALF_WIDTH = ROBOT_WIDTH/2d;
	public static final double ACCELERATION_RATE = 1d;  // robot acceleration rate
	public static final double DECELERATION_RATE = -2d; // robot deceleration rate
	public static final double MIN_BULLET_VELOCITY 
			= getBulletVelocity(MAX_FIRE_POWER);
	public static final double MAX_BULLET_VELOCITY
			= getBulletVelocity(MIN_FIRE_POWER);
	public static final double MIN_EFFECTIVE_HIT_RATIO 
			= MAX_FIRE_POWER / (getEnergyRegained(MAX_FIRE_POWER) + getBulletDamage(MAX_FIRE_POWER)); 
	
	/**
	 * Returns the minimum effective hit ratio (or break even point) for the given bullet power.
	 * 
	 * @param power         bullet power
	 * 
	 * @return              minimum effective hit ratio (break even point)
	 */
	public static double getEffectiveHitRatio(double power) {
		return power / (getEnergyRegained(power) + getBulletDamage(power));
	}
	
	/**
	 * Get the bullet velocity for a given fire power.
	 *
	 * @param power			fire power of bullet
	 *
	 * @return				bullet velocity
	 */
	public static double getBulletVelocity(double power) {
		return (20d - 3d*power);
	}

	/**
	 * Returns a bullet power given it's velocity.
	 * 
	 * @param velocity      velocity of bullet
	 * 
	 * @return              power of bullet
	 */
	public static double getBulletPower(double velocity) {
		return (20d - velocity) / 3d;
	}
	
	/**
	 * Returns the maximum turn rate for the given velocity.
	 * 
	 * @param velocity      velocity of robot
	 * 
	 * @return              maximum turn rate for the given velocity
	 */
	public static double getMaxTurnRate(double velocity) {
		return MAX_TURN_RATE - 0.75 * Math.abs(velocity);
	}
	
	/**
	 * Get the damage done by a bullet of given fire power.
	 * 
	 * @param power			fire power of bullet
	 * 
	 * @return				damage bullet would cause
	 */
	public static double getBulletDamage(double power) {
		double damage = 4d * power;
		if (power > 1) {
			damage += (2d * (power - 1d));
		}
		return damage;
	}
	
	/**
	 * Gets the damage done when running into a wall.
	 * 
	 * @param velocity    velocity at time of hitting wall
	 * 
	 * @return            damage done by hitting wall
	 */
	public static double getWallHitDamage(double velocity) {
		return Math.max(0, Math.abs(velocity)*0.5 - 1);
	}
	
	/**
	 * Get the amount of energy regained by bullet of given power
	 * hitting an opponent.
	 * 
	 * @param power			bullet power
	 * 
	 * @return				energy bullet ower regains
	 */
	public static double getEnergyRegained(double power) {
		return 3 * power;
	}
	
	/**
	 * Get the firepower needed to kill target.  Results may exceed the 
	 * max allowed fire power.
	 * 
	 * @param energy				energy remaining in target
	 * 
	 * @return						firepower required to kill target
	 */
	public static double getFirePowerToKill(double energy) {
		double firePower = energy / 4d;
		if (firePower > 1) {
			firePower = (energy + 2) / 6d;
		}
		return firePower + 0.1;	// add 0.1 for disabled bots
	}
	
	/**
	 * Returns the gun heat generated by firing a bullet of given power.
	 * 
	 * @param power    power of bullet
	 * 
	 * @return         gun heat generated
	 */
	public static double getGunHeatGenerated(double power) {
		return 1 + power/5d;
	}
	
	/**
	 * Returns the number of ticks remaining until the gun is cool.
	 * 
	 * @return         time in ticks until gun is cool
	 */
	public static int getTimeUntilGunCool() {
		double timeToCool = Resources.getRobotProxy().getGunHeat() / Resources.getRobotProxy().getGunCoolingRate();
		return (int) Math.round(Math.ceil(timeToCool));
	}
	
	/**
	 * Returns the number of ticks for a gun to cool after firing a bullet with given power.
	 * 
	 * @param power    power of bullet fired
	 * 
	 * @return         time until gun can cool again after firing bullet of given power
	 */
	public static int getTimeUntilGunCool(double power) {
		double timeToCool = getGunHeatGenerated(power) / Resources.getRobotProxy().getGunCoolingRate();
		return (int) Math.round(Math.ceil(timeToCool));
	}
	
	/**
	 * Returns the energy change rate per tick for a robot firing with the given hit ratio
	 * and average fire power.  Positive values are net energy gains, while negative
	 * values are net energy losses.  This does not take possible bullet damage against
	 * opponent into account.
	 * 
	 * @param hitRatio        hit ratio against opponent
	 * @param averagePower    average bullet power
	 * @param gunCoolingRate  rate at which gun cools
	 * 
	 * @return    energy change rate
	 */
	public static double getEnergyRate(double hitRatio, double averagePower, double gunCoolingRate) {
		return (hitRatio*getEnergyRegained(averagePower)-averagePower)/(averagePower/gunCoolingRate);
	}
	
	public static boolean isSteadySpeed(double v0, double v1) {
		// For our purposes, a tiny variation in speed is allowed to still be considered a steady speed.
		return Math.abs(v1 - v0) < 0.05;
	}
	
	public static boolean isAcceleration(double v0, double v1) {
		// Crossover, if it can happen (pos -> neg or neg -> pos without either being 0), could be viewed as accel or decel.
		// For our purposes, this will be determined by whether v1 is a higher speed than v0.
		return (!isSteadySpeed(v0, v1) && Math.abs(v1) > Math.abs(v0));
	}
	
	public static boolean isDeceleration(double v0, double v1) {
		// Crossover, if it can happen (pos -> neg or neg -> pos without either being 0), could be viewed as accel or decel.
		// For our purposes, this will be determined by whether v1 is a lower speed than v0.
		return (!isSteadySpeed(v0, v1) && Math.abs(v1) < Math.abs(v0));
	}
}
