package rc.yoda.utils;

import robocode.AdvancedRobot;
import java.awt.geom.Rectangle2D; 
import java.awt.geom.Point2D;

/** 
 * YUtils - by Robert Codd (Gorded) 
 *
 * This code is here by released under the RoboWiki Public Code Licence (RWPCL),
 * datailed on: http://robowiki.net/?RWPCL
 * (Basically it means you must keep the code public if you base any bot on it)
 *
 * YUtils.java : v1.0 -- 2007/05/12
 */ 

/**
 * YUtils - A utilies class that contains
 * common variables and methods needed 
 * throughout this robot
 *
 * @author Robert Codd
 * @version v 1.0
 */
public final class YUtils 
{
	/**
	 * The pixel width of a robocode robot
	 */	
	public static final int BOT_WIDTH = 36;
		
	/**
	 * The distance ahead of the robot to start wall smoothing
	 */
	public static final int WALL_STICK = 185;
	
	/**
	 * The closest distance a robot can be from a wall with out hitting it
	 */	
	public static final int WALL_MARGIN = 18; 
		
	/**
	 * The pixel width of the current battle field, defaults to 800
	 */
	public static double battleFieldWidth = 800;
	
	/**
	 * The pixel height of the current battle field, defaults to 600
	 */
	public static double battleFieldHeight = 600; 
				
	/**
	 * A rectangle the size of the current battle field
	 */
	public static Rectangle2D.Double field = new Rectangle2D.Double(0, 0, battleFieldWidth, battleFieldHeight);
	
	/**
	 * A rectangle the size of the current safe battle field, the border the size of WALL_MARGIN is removed
	 */
	public static Rectangle2D.Double battleField = new 
		Rectangle2D.Double(WALL_MARGIN, WALL_MARGIN, battleFieldWidth - (WALL_MARGIN * 2), battleFieldHeight - (WALL_MARGIN * 2));
	
	/**
	 * Extracts battle field specifications from an
	 * AdvancedRobot class
	 * 
	 * @param an AdvancedRobot 
	 */ 
	public static void setFieldAttr(AdvancedRobot robot) {
		battleFieldWidth = robot.getBattleFieldWidth();
		battleFieldHeight = robot.getBattleFieldHeight();		
		field = new Rectangle2D.Double(0, 0, battleFieldWidth, battleFieldHeight);
		battleField = new Rectangle2D.Double(WALL_MARGIN, WALL_MARGIN, 
			battleFieldWidth - (WALL_MARGIN * 2), battleFieldHeight - (WALL_MARGIN * 2));
	} 
	
	/**
	 * Calculates the sign(+/-) of an number
	 *
	 * @param a number
	 * @return one if positive 
	 */
	public static int sign(double v) {
    	return v >= 0 ? 1 : -1;
	}

	/**
	 * Calculates the max angle a robot can travel 
	 * going max velocity if firing a bullet 
	 * travel a specified velocity
	 *
	 * @param bullet velocity
	 * @return max angle a robot can travel
	 */
    public static double maxEscapeAngle(double velocity) {
        return Math.asin(8.0/velocity);
    }

	/**
	 * Caps a value to within a minimum and maximum range
	 *
	 * @param minimum value of range
	 * @param value to cap
	 * @param maximum value of range
	 * @return the capped value
	 */
    public static double limit(double minimum, double value, double maximum) {
        return Math.max(minimum, Math.min(value, maximum));
    }

	/**
	 * Caculates the distance traveled while stopping
	 * from a specified velocity
	 *
	 * @param velocity at which appling breaks
	 * @return distance traveled while stopping
	 */
	public static double stopDistance(double velocity) {
		return Math.ceil(velocity / 2) * (Math.floor(velocity / 2) + 1);
	}

	/**
	 * Calculates how to move based on your oldheading and
	 * the heading you want
	 *
	 * @param wanted heading
	 * @param heading current at
	 * @return the amount to turn
	 * @credit BY : PEZ
	 */
    public static double backAsFrontTurn(double newHeading, double oldHeading) {
		return Math.tan(newHeading - oldHeading);
    }

	/**
	 * Calculates direction to move based on 
	 * your oldheading and your wanted heading
	 *
	 * @param wanted heading
	 * @param heading current at
	 * @return the amount to turn
	 * @credit BY : PEZ
	 */
    public static int backAsFrontDirection(double newHeading, double oldHeading) {
		return sign(Math.cos(newHeading - oldHeading));
    }

	/**
	 * Calculates the angle opposite side a using the cosine Law
	 *
	 * @param length of side a
	 * @param length of side b
	 * @param length of side c
	 * @return the angle opposite side a 
	 */
	public static double cosineLaw(double a, double b, double c) {
		return Math.acos((Math.pow(b, 2) + Math.pow(c, 2) - Math.pow(a, 2)) / (2 * b * c));
	}

	/**
	 * Calculates a rolling average of a current average
	 * and a new entry using the data current depth 
	 * and this entry's weighting 
	 * 
	 * @param current average of sample
	 * @param new entry to average
	 * @param depth of the sample
	 * @param weighting for this entry
	 * @credit BY : Paul Evans
	 */
	public static double rollingAverage(double currentAvg, double newEntry, double depth, double weighting) {
    	return (currentAvg * depth + newEntry * weighting)/(depth + weighting);
	} 

	/**
	 * Caculates the absolute bearing relative 
	 * from source to the target 
	 *
	 * @param source fo bearing
	 * @param target of the bearing
	 * @param absolute bearing from source to target
	 */
	public static double absoluteBearing(Point2D.Double source, Point2D.Double target) {
        return Math.atan2(target.getX() - source.getX(), target.getY() - source.getY());
    }

	/**
	 * Normalizes an angle to an absolute angle.
	 * The normalized angle will be in the range from 0 to 2*PI, where 2*PI
	 * itself is not included.
	 *
	 * @param angle the angle to normalize
	 * @return the normalized angle that will be in the range of [0,2*PI[
	 */
	public static double normalAbsoluteAngle(double angle) {
		return (angle %= 2 * Math.PI) >= 0 ? angle : (angle + 2 * Math.PI);
	}

	/**
	 * Normalizes an angle to a relative angle.
	 * The normalized angle will be in the range from -PI to PI, where PI
	 * itself is not included.
	 *
	 * @param angle the angle to normalize
	 * @return the normalized angle that will be in the range of [-PI,PI[
	 */
	public static double normalRelativeAngle(double angle) {
		return (angle %= 2 * Math.PI) >= 0 ? (angle < Math.PI) ? angle : angle - 2 * Math.PI : (angle >= -Math.PI) ? angle : angle + 2 * Math.PI;
	}

	/**
	 * Projects a location a certain distance
	 * along its current angle
	 *
	 * @param location of the point to project
	 * @param angle point is travel 
	 * @param distance to project the point
	 * @return the new projected point
	 */
	public static Point2D.Double project(Point2D.Double location, double angle, double distance) {
		return new Point2D.Double(location.getX() + Math.sin(angle) * distance, location.getY() + Math.cos(angle) * distance);
	}
	
	public static double getMinBulletPower(double damage) {
		return limit(0, damage / 4 > 1 ? Math.ceil((damage / 4 - 1) / 0.15) / 10 + 1 : damage / 4, 3);
	}	

	/**
	 * Caculates a "wallsmoothed" angle for a point traveling
	 * along a specified angle and a left or right orientation
	 * 
	 * @param point to wallsmooth fro
	 * @param angle to project point
	 * @param orientation of the robot (-1 / 1)
	 * @return wallsmoothed angle to travel at
	 */
    public static double wallSmoothing(Point2D.Double botLocation, double angle, int orientation) {
        while (!battleField.contains(project(botLocation, angle, WALL_STICK))) { angle += orientation * 0.1; }
        return angle;
    }
}
