package sgp;
import robocode.*;
import java.util.*;
import java.awt.Color;

/**
 * EvasionStrategy - a class by Simon Parker
 */
public class EvasionStrategy extends Strategy
{
	private JollyNinja robot = null;
	Target target = null;

	static private int winCount = 0;

	boolean isLocked = false;
	int unsuccessfulScanCounter = 0;

	static private int bulletHitCount = 0;
	static private int bulletMissCount = 0;

	final int MAX_SCAN_FAIL_COUNT = 2;
	public static double ROBOT_RADIUS = 45.0/2;

	final double RADAR_TURN_ANGLE = 40;

	private double previousHeading = 0;
	private Coordinate currentPosition = new Coordinate();
	
	private final int NUM_VELOCITIES = 8 + 8 + 1;
	private double[] optimumVelocityArray = new double[NUM_VELOCITIES];
	private int optimumVelocityIndex = 0;
	private double AVERAGE_VELOCITY_CHANGE_TIME = 20.0;
	private double TRANSITION_PROBABILITY = 0.067;
	
	static private int constructionCount = 0;

	public void reset()
	{
		if (target != null) target.reset();
		
		AVERAGE_VELOCITY_CHANGE_TIME = robot.parameters.evasionTransitionTime.getValue();
		TRANSITION_PROBABILITY = 1.0 - Math.exp(Math.log(0.5) / AVERAGE_VELOCITY_CHANGE_TIME);

//		optimumVelocityArray[0] = 8;
//		optimumVelocityArray[1] = 1;
//		optimumVelocityArray[2] = -8;
//		optimumVelocityArray[3] = -1;

		for (int i = 0; i < NUM_VELOCITIES; i++)
		{
			optimumVelocityArray[i] = i - 8;
		}
		
	}

	public EvasionStrategy(StrategyManager theStrategyManager)
	{
		super(theStrategyManager);
		constructionCount++;
		robot = JollyNinja.getInstance();
	}

	//any init that needs doing at the start of the turn
	public void startTurn()
	{
		double heading_deg = Environment.getHeading();
		double velocity = Environment.getVelocity();
		
		robot.angularVelocity = normalRelativeAngle(heading_deg - previousHeading);
		currentPosition.set(Environment.getX(), Environment.getY());
		robot.jiggleAnalyser.update(currentPosition, Environment.getVelocity(), heading_deg);
		robot.patternAnalyser.update(currentPosition, Environment.getVelocity(), heading_deg, (int)Environment.getTime());
		robot.antiDodgeAnalyser.update(currentPosition, Environment.getVelocity(), heading_deg, (int)Environment.getTime());

		if (unsuccessfulScanCounter < 0)
		{
			unsuccessfulScanCounter = 0;
		}
	}


	public void endTurn()
	{
		previousHeading = Environment.getHeading();
	}

	public double evaluateGoodness(Coordinate robotPosition, double robotHeading, double robotVelocity)
	{
		if (target == null) return Math.random();

		double BULLET_AVOIDANCE_WEIGHT = robot.parameters.evasionBulletAvoidanceWeight.getValue();
		double OPTIMUM_TARGET_DISTANCE = robot.parameters.evasionOptimumTargetDistance.getValue();
		double OPTIMUM_TARGET_DISTANCE_WEIGHT = robot.parameters.evasionOptimumTargetDistanceWeight.getValue();
		double BEARING_WEIGHT = robot.parameters.evasionBearingWeight.getValue();
		double CENTRE_WEIGHT = robot.parameters.evasionCentreWeight.getValue();
		double VELOCITY_WEIGHT = robot.parameters.evasionVelocityWeight.getValue();

		double bearingDifference_deg = normalRelativeAngle(robotPosition.headingTo(target.position) - target.heading_deg);
		double bearingGoodness = Math.abs(Math.cos(Math.toRadians(bearingDifference_deg)));

		double distanceGoodness = - OPTIMUM_TARGET_DISTANCE_WEIGHT * Math.abs(robotPosition.distanceFrom(target.position) - OPTIMUM_TARGET_DISTANCE);

		Coordinate centrePosition = new Coordinate(Environment.battleFieldHeight / 2, Environment.battleFieldWidth / 2);
		double centreProximityGoodness = - CENTRE_WEIGHT * robotPosition.distanceFrom(centrePosition);
		
				

		return BULLET_AVOIDANCE_WEIGHT * target.bulletList.getBulletAvoidanceGoodness(robotPosition, Environment.getTime())
			+ evaluateWallGoodness(robotPosition)
			+ BEARING_WEIGHT * bearingGoodness
			+ distanceGoodness
			+ centreProximityGoodness
			+ VELOCITY_WEIGHT * getVelocityGoodness(robotVelocity);
	}

	public void setGunRotation() //control the gun motion and fires if neccessary
	{
		if (target == null) return;
		if (target.position == null) return;
		

		Coordinate estimatedTargetPosition = target.getEstimatedPosition(Environment.getTime());
		double targetDistance = currentPosition.distanceFrom(estimatedTargetPosition);

		double bulletPower = calculateBulletPower(currentPosition, estimatedTargetPosition);

		Intercept intercept = target.gunManager.getIntercept(currentPosition, bulletPower);
		if (intercept == null) return;
		

		double turnAngle = normalRelativeAngle(intercept.bulletHeading_deg - Environment.getGunHeading());
		// move gun to target angle
		robot.setTurnGunRight(turnAngle);

		if 	(Math.abs(turnAngle) <= intercept.angleThreshold)
		{
			if 	(
					(intercept.impactPoint.x > 0) &&
					(intercept.impactPoint.x < Environment.battleFieldWidth) &&
					(intercept.impactPoint.y > 0) &&
					(intercept.impactPoint.y < Environment.battleFieldHeight)
				)
			{
				if ((Environment.getGunHeat() == 0) && (Environment.getEnergy() > bulletPower))
				{
					target.fire(bulletPower);
				}
			}
		}
	}


	public void onHitWall(HitWallEvent e)
	{
		/*
		if (Math.abs(e.getBearing()) >= 90)
		{
			robot.setAhead(MOVE_DISTANCE);
			currentVelocity = MAX_VELOCITY;
		}
		else
		{
			robot.setAhead(-MOVE_DISTANCE);
			currentVelocity = -MAX_VELOCITY;
		}
		robot.execute();
		*/
	}

	public void onHitRobot(HitRobotEvent e)
	{
		/*
		if (e.getBearing() > -90 && e.getBearing() < 90)
		{
			robot.setBack(MOVE_DISTANCE);
			currentVelocity = -MAX_VELOCITY;
		}
		else
		{
			robot.setAhead(MOVE_DISTANCE);
			currentVelocity = MAX_VELOCITY;
		}
		robot.execute();
		*/
	}

	public void onScannedRobot(ScannedRobotEvent e)
	{
		unsuccessfulScanCounter--;

		if (target == null)
		{
			target = new Target("1v1", e.getName(), robot);
		}
		
		if (target.name != e.getName())
		{
			reset();
		}

		isLocked = true;
		
		target.update(e, Environment.getTime());
	}

	public void onHitByBullet(HitByBulletEvent e)
	{
		if (target != null)
		{
			target.onHitByBullet(e);
		}
	}
	
	
	public void onDeath(DeathEvent event)
	{
		if (target != null)
		{
			target.saveMovementLog();
			target.setDead();
		}
	}

	//control the radar motion
	public void setScan()
	{
		if (isLocked)
		{
			if (unsuccessfulScanCounter > MAX_SCAN_FAIL_COUNT)
			{
				unsuccessfulScanCounter = 0;
				isLocked = false;

				//???robot.out.println("Lost Lock");
				return;
			}

			unsuccessfulScanCounter++;
			robot.scan();

			Coordinate robotPosition = Environment.getRobotPosition();

			double targetRadarAngle = robotPosition.headingTo(target.position);
			double radarTurnRightAngle = normalRelativeAngle(targetRadarAngle - Environment.getRadarHeading());

			if (radarTurnRightAngle > 0)
			{
				radarTurnRightAngle += RADAR_TURN_ANGLE/2;
			}
			else
			{
				radarTurnRightAngle -= RADAR_TURN_ANGLE/2;
			}

			robot.setTurnRadarRight(radarTurnRightAngle);

		}
		else
		{
			// look for new robots
			robot.setTurnRadarRight(720);
		}
	}

	private double calculateBulletPower
	(
		Coordinate robotPosition,
		Coordinate targetPosition
	)
	{
		//???double power = 2.0 + Math.random();
		double power = 3;

		power = Math.min(Strategy.getPowerFromDamage(target.energy) + 0.001, power);
		
		power = Math.min(Environment.getEnergy() / 2.0, power);
		
		power = Math.max(0.1, power);

		return power;

//		if (target == null) return 0.1;
//		double hitPercent = target.getBulletHitPercentage();
//		double HIT_RATE_THRESHOLD = robot.parameters.hitRateThreshold.getValue();
//
//		if (hitPercent > HIT_RATE_THRESHOLD) return 3.0;
//		else return 2.0001;

/*		double hitPercent = target.getBulletHitPercentage(robotPosition);
		double myEnergy = robot.getEnergy();
		double distance = robotPosition.distanceFrom(targetPosition);

		double HIT_RATE_THRESHOLD = robot.parameters.hitRateThreshold.getValue();
		double MAX_DISTANCE_DIFF = robot.parameters.maxDistanceDifference.getValue();
		double MIN_DISTANCE = robot.parameters.minDistance.getValue();

		final double MAX_BULLET_POWER = 3.0;
		final double MIN_BULLET_POWER = 0.5;
		final double HIT_RATE_WEIGHT = Math.min(hitPercent / HIT_RATE_THRESHOLD, 1.0);
		double DISTANCE_WEIGHT = Math.min((MAX_DISTANCE_DIFF - MIN_DISTANCE - distance) / (MAX_DISTANCE_DIFF), 1.0);
		if (DISTANCE_WEIGHT < 0) DISTANCE_WEIGHT = 0;

		double power = MAX_BULLET_POWER * HIT_RATE_WEIGHT * DISTANCE_WEIGHT;
		power = Math.max(MIN_BULLET_POWER, power);
		power = Math.min(MAX_BULLET_POWER, power);

		//return power;
		return power;
*/
	}

	public void onBulletHit(BulletHitEvent event)
	{
		if (target == null) return;
		target.onBulletHit(event);
	}

	public void onWin(WinEvent e)
	{
		winCount++;
		if (target != null) target.setDead();
		
		robot.out.println("Win Count = " + winCount);
	}
	
	private double getVelocityGoodness(double velocity)
	{
		int newVelocityIndex = 0;
		if (Math.random() < TRANSITION_PROBABILITY)
		{
			do
			{
				newVelocityIndex = (int)Math.floor(Math.random() * optimumVelocityArray.length);
			} while (newVelocityIndex == optimumVelocityIndex);
			optimumVelocityIndex = newVelocityIndex;
		}
		
		return Math.abs(optimumVelocityArray[optimumVelocityIndex] - velocity);
	}
}
