package tjk;
import robocode.*;
import java.awt.Color;
import java.util.Random;
import java.awt.Graphics2D;

/**
 * A_Flat - a robot by Mr. Kiesel
 */
public class AFlatNatural extends AdvancedRobot
{
	
	// This variable is the virtual gun array.
	private VGun gun = new VGun(this);

	// This variable holds which way the radar is spinning.
	boolean turningRadarRight = true;
	
	// This variable gets increased every time an enemy is scanned.
	static double scan = 1;
	
	// These hold the size of the battlefield
	double battlefieldHeight;
	double battlefieldWidth;
	
	// These can be set to change how the robot reacts to the wall.
	// wallAvoid is our TOO CLOSE TO WALL variable.  
	// If it's here, the robot will head for the center of the battlefield
	double wallAvoid = 45;
	// wallSmooth is the BEGIN DODGING THE WALL variable.
	double wallSmooth = 100;
	
	// This tells the robot how close it wants to circle around something.
	//  If the robot is closer than this, it will spiral away.
	//  If the robot is farther away, it will spiral in.
	double orbitDistance = 200;
	
	// These variables (lot of them, huh?)  hold onto the data about the enemy BETWEEN
	//  scans.  This way we can see what the enemy is doing (turning, speeding up, etc.)
	double oldEnemyEnergy = 0;
	double newEnemyEnergy = 0;
	double oldEnemyHeading = 0;
	double newEnemyHeading = 0;
	double oldEnemyBearing = 0;
	double newEnemyBearing = 0;
	double oldEnemyVelocity = 0;
	double newEnemyVelocity = 0;
	String oldEnemyName = "";
	String newEnemyName = "";

	// This is a random number generator.
	Random randgen;

	/**
	 * run: A_Flat's default behavior
	 */
	public void run() {
		
		// Unlock the Radar and Gun from each other and the Body.
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		
		// Set colors
		setBodyColor(new Color(255,140,0));
		setGunColor(new Color(75,0,0));
		setRadarColor(new Color(139,0,0));
		setScanColor(Color.red);
		setBulletColor(new Color(255,69,0));
		
		// Get Battlefield height and width.
		battlefieldHeight = getBattleFieldHeight();
		battlefieldWidth = getBattleFieldWidth();
		
		// Start the random number generator.
		randgen = new Random();
		
		// Spin the radar to the right forever
		turnRadarRight();
	}
	/**
	 *  End of run
	*/

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		
		
		// update scan by making it one bigger.
		scan++;
		
		// This is a NEW SCAN.  Every "new" variable we had is old now.
		oldEnemyName = newEnemyName;
		oldEnemyEnergy = newEnemyEnergy;
		oldEnemyHeading = newEnemyHeading;
		oldEnemyBearing = newEnemyBearing;
		oldEnemyVelocity = newEnemyVelocity;
		
		// Now we can get NEW data about the enemy.
		newEnemyName = e.getName();
		newEnemyEnergy = e.getEnergy();
		newEnemyHeading = e.getHeading();
		newEnemyBearing = e.getBearing();
		newEnemyVelocity = e.getVelocity();
		
		// The distance to the enemy.
		double enemyDistance = e.getDistance();

		// If we're looking at the same enemy as the last scan, and the scans show that
		//  the enemy just fired a bullet, we'd better dodge!
		if ( isOldEnemy() &&  enemyFiredBullet() ) {
			reverseCourse();
		}
		// Otherwise, if we're one-on-one or the enemy is far away, or if we're closer
		//   than 55 to the enemy, head for them!
		// This is why I have good ram damage recently.. 
		// If I'm close, I go for the ram!
		else if ( (getOthers() <= 1 && enemyDistance > 350) || enemyDistance < 55  ) {
			headFor(newEnemyBearing, 250);
		}
		// Otherwise, if I'm WAY too close to the wall, I ought 
		//   to head for the center of the battlefield.
		else if (closeToWall(wallAvoid)) {
			goToCenter();
		}
		// Otherwise, if I'm sorta close to the wall, I ought to face 90 degrees
		//   from the center of the battlefield.
		else if ( closeToWall(wallSmooth) ) {
			smoothAlongWall();
		}
		// Otherwise, I orbit the spot where the enemy WILL BE soon.
		//   (Using the Linear gun code to see where the enemy will be.)
		else {
			orbit(newEnemyBearing + linearGunTurn(e, Math.min(2,getOthers())), enemyDistance);
		}
		
		// Don't run the rest of onScannedRobot if the enemy is 
		//    more than 425 away.
		// If the enemy is too far away, the robot will
		//    head toward them for a better shot later.
		//if ( enemyDistance > 550) {
		//	return;
		//}
		// Reverse the turn of the Radar.
		setTurnRadarRight( 1.95 * fixAngle(newEnemyBearing + getHeading() - getRadarHeading()) );	
		
		//setTurnRadarRight( 1.9 * fixAngle( getHeading() + e.getBearing()  ) );
		
		gun.shoot(e);
		
	
	}
	/**
	 * End of onScannedRobot:
	 */

	public void onPaint(Graphics2D g) {
		gun.onPaint(g);
	}

	/**
	 * onHitWall:  What to do when the robot hits the wall.
	 */
	public void onHitWall(HitWallEvent e) {
		//Go to the center of the battlefield.
		goToCenter();
	}
	/**
	 * End of onHitWall:
	 */

	/**
	 * onHitRobot: What to do when we hit (or are hit by) another robot
	 */
	public void onHitRobot(HitRobotEvent e) {
		// Bearing to the enemy, fixed.
		double bearing = fixAngle(e.getBearing());
		
		// If your gun is already pointed at the enemy bot, FIRE!!
		if ( Math.abs(fixAngle(bearing + getHeading() - getGunHeading()  )) < 0.75  ) {
			setFire(3);
		}
		
		// Head for the enemy. Ram him again!
		headFor(bearing, 250);
	}
	/**
	 * End of onHitRobot:
	 */
	
	
	/**
	 * onHitByBullet: What to do when you're hit by a bullet
	 */
	public void onHitByBullet(HitByBulletEvent e) {
		// If the gun is already pointed at the enemy, FIRE!
		//if ( Math.abs(fixAngle(e.getBearing() + getHeading() - getGunHeading())) < 1  ) {
		//	setFire(3);
		//}
		
		// Reverse turn and movement.
		//setTurnRadarRight( 3.0 * fixAngle( getHeading() + e.getBearing() - getRadarHeading() ) );
		//reverseRadarTurn();
		//reverseCourse();
	}
	/**
	 * End of onHitByBullet:
	 */

	/**
	 * onWin: What to do when you WIN THE ROUND!!
	 */
	public void onWin(WinEvent event) {
		// Stop moving.
		setAhead(0);
		
		// Spin the body, gun and radar in different directions.
		setTurnGunLeft(Double.POSITIVE_INFINITY);
		setTurnRadarRight(Double.POSITIVE_INFINITY);
		setTurnRight(Double.POSITIVE_INFINITY);
		
		// Move back and forth by 10 pixels.
		//  This uses an infinite loop and a boolean.
		//      If ahead is true, go ahead 10 pixels and set ahead = false
		//      If ahead is false, go back 10 pixels and set ahead = true
		boolean ahead = true;
		while (true) {
			if (ahead) {
				ahead(10);
				ahead = false;
			}
			else {
				back(10);
				ahead = true;
			}
		}
	}
	/**
	 * End of onWin:
	 */

	/**
	 * fixAngle: fixes an angle to make turning as quick as possible
	 */
	public double fixAngle(double angle) {
		// If angle is bigger than 360 or smaller than -360, cut it down to size.
		angle = angle % 360;
		
		// Fix the angle to be between -180 and 180
		if (angle > 180) {
			return angle - 360;
		}
		else if (angle < -180) {
			return 360 + angle;
		}
		else {
			return angle;
		}
	}
	/**
	 * End of fixAngle
	 */

	/**
	 * turnRadarLeft: Makes the radar spin left forever
	 */
	public void turnRadarLeft() {
		setTurnRadarLeft( Double.POSITIVE_INFINITY );
		turningRadarRight = false;
	}
	/**
	 * End of turnRadarLeft:
	 */

	/**
	 * turnRadarRight: Makes the radar spin Right forever
	 */
	public void turnRadarRight() {
		setTurnRadarRight( Double.POSITIVE_INFINITY );
		turningRadarRight = true;
	}
	/**
	 * End of turnRadarRight:
	 */
	 
	/**
	 * reverseRadarTurn: Makes the radar spin the other way around.
	 */
	public void reverseRadarTurn() {
		// If the radar is turning right, make it turn left.
		if (turningRadarRight) {
			turnRadarLeft();
		}
		// Otherwise, make it turn right.
		else {
			turnRadarRight();
		}
	}
	/**
	 * End of reverseRadarTurn:
	 */
	
	/**
	 * myFire: fires the gun if it isn't overheated.
	 */
	public void myFire(double firepower) {
		// If the gun isn't overheated, fire a bullet.
		if (getGunHeat() == 0.0) {
			setFire(firepower);
		}
	}
	/**
	 * End of myFire:
	 */

	/**
	 * isOldEnemy:	returns true if the current enemy is the same one the 
	 * 				robot saw on the last scan.
	 */
	public boolean isOldEnemy() {
		// Return true the new enemy's name is the same as the old enemy's name.
		return newEnemyName.equals(oldEnemyName);
	}
	/**
	 * End of isOldEnemy:
	 */

	/**
	 * enemyFiredBullet:	returns true if the current enemy has lost 
	 * 				between 0.1 and 3.0 energy. This means that they 
	 * 				probably fired a bullet.
	 */
	public boolean enemyFiredBullet() {
		// The difference between the enemy's old energy scan and new energy scan.
		double diff = oldEnemyEnergy - newEnemyEnergy;
		
		// If the energy difference is bigger than 0.1 and less than 3.0
		//    then the enemy probably fired a bullet.
		if (diff >= 0.1 && diff <= 3.0) {
			return true;
		}
		else {
			return false;
		}
	}
	/**
	 * End of enemyFiredBullet:
	 */

	/**
	 * reverseCourse: Start heading the opposite direction
	 */
	public void reverseCourse() {
		// Change scan to be on the other side of zero. 
		//   Just to mess up the movement.
		scan = -scan;
		// Set the robot to move in the opposite direction from where it's going now.
		setAhead(-5 * getDistanceRemaining());
	}
	/**
	 * End of reverseCourse:
	 */

	/**
	 * headFor: Head toward a bearing as quickly as possible
	 */
	public void headFor(double bearing, double distance) {
		bearing = fixAngle( bearing );
		
		if ( Math.abs(bearing) <= 90 ) {
			setAhead(distance);
			setTurnRight(bearing);
		}
		else {
			setBack(distance);
			setTurnLeft(fixAngle(180 - bearing));			
		}
	}
	/**
	 * End of headFor:
	 */

	/**
	 * headingToCenter: Return the direction toward the center of the battlefield
	 */
	public double headingToCenter() {
		return Math.toDegrees( Math.atan2( battlefieldWidth/2 - getX() , battlefieldHeight/2 - getY() ) );
	}
	/**
	 * End of headingToCenter:
	 */
	
	/**
	 * goToCenter: go to the center of the battlefield
	 */
	public void goToCenter() {
		headFor(headingToCenter() - getHeading(), 3*wallAvoid);
	}
	/**
	 * End of goToCenter:
	 */

	/**
	 * distanceTo: return the distance between two points
	 *  Good old distance formula. Thank you, Pythagoras.
	 */
	public double distanceTo(double x, double y, double x2, double y2) {
		return Math.sqrt( Math.pow(x-x2,2) + Math.pow(y-y2,2) );
	}
	/**
	 * End of distanceTo:
	 */

	/**
	 * closeToWall: return true if the robot is closer than "clearance" to the wall
	 */
	public boolean closeToWall(double clearance) {
		// variables to hold my x and my y, since we need to use them so much.
		double X = getX();
		double Y = getY();
		
		// If we're in the "safe zone", 
		if ( X > clearance && X < battlefieldWidth - clearance && Y > clearance && Y < battlefieldHeight - clearance) {
			// Even if we're far from the walls themselves, check to see if we're 2*clearance 
			//   from the corners. Report that we're close to the wall if we're too close 
			//    to the corners.
			if (distanceTo(0,0,X,Y) < 2*clearance || distanceTo(0,battlefieldHeight,X,Y) < 2*clearance || distanceTo(battlefieldWidth,0,X,Y) < 2*clearance || distanceTo(battlefieldWidth,battlefieldHeight,X,Y) < 2*clearance) {
				return true;
			}
			// Otherwise, we're safe.
			else {
				return false;
			}
		}
		// Otherwise, report that we're too close.
		else {
			return true;			
		}
	}
	/**
	 * End of closeToWall:
	 */

	/**
	 * wiggle: move back and forth in a weird pattern.
	 *  the weird pattern is provided by running the scan variable 
	 *  through some trig functions to make a random-ish back and forth
	 *  movement.
	 */
	public void wiggle(double distance) {
		if ( scan % 3 == 0 ) {
			double trig = Math.sin(scan / 13) * Math.cos(scan / 21);
			double add = 40;
			double dist =  500 / distance;
			if (trig < 0) {
				add = -add;
			}
			setAhead( 1750 * trig * dist + add);
		}
	}
	/**
	 * End of wiggle:
	 */
	
	/**
	 * orbit: Move back and forth around some point.
	 *      If we're farther than "orbitDistance" away from it, we'll close in.
	 *      If we're closer than "orbitDistance" away from it, we'll move out.
	 *      Back and forth motion provided by "wiggle" function.
	 */
	public void orbit(double bearing, double distance) {
		// Move back and forth
		wiggle(distance);
		
		// This variable will adjust the turn to make the robot close in
		//   or back off from its target.
		double turnAdjust = 20;
		
		// If the robot's moving backward, turnAdjust needs to go the other way.
		if (getDistanceRemaining() > 0) {
			turnAdjust = -1 * turnAdjust;
		}
		
		// If we're closer than we need to be to the enemy, turnAdjust needs to go the other way.
		if ( distance < orbitDistance ) {
			turnAdjust = -1 * turnAdjust;
		}
		
		// Turn 90 degrees to the provided bearing, plus the turnAdjust.
		//  This provides the orbit.
		setTurnLeft( fixAngle( 90 - bearing + turnAdjust ));
	}
	/**
	 * End of orbit: 
	 */

	/**
	 * smoothAlongWall: orbit the center of the battlefield. 
	 * Because of the way that Orbit works, this will naturally 
	 * pull the robot closer to the center and away from the wall.
	 */
	public void smoothAlongWall() {
		orbit(headingToCenter() - getHeading(), distanceTo(getX(), getY(), battlefieldWidth/2, battlefieldHeight/2));
	}
	/**
	 * End of smoothAlongWall:
	 */

	/**
	 * linearGunTurn:  What is the turn adjustment made by the linear gun?
	 */
	public double linearGunTurn(ScannedRobotEvent e, double firepower) {
		double linearGunTurn = Math.sin(Math.toRadians(180 - getHeading() - e.getBearing() + e.getHeading()));
		linearGunTurn = (e.getVelocity() * linearGunTurn) / (20 - 3 * firepower);
		linearGunTurn = Math.toDegrees(Math.asin( linearGunTurn ));
		return linearGunTurn;
	}
	/**
	 * End of linearGunTurn:
	 */

	/**
	 * linearGun:  Use linearGunTurn to turn the gun at the enemy.
	 *  the "adjust" number lets us aim BETWEEN where the enemy is and where it is going, if we want!
	 *       adjust = 0 *Dumb Fire*
	 *       adjust = 1 Linear Gun
	 *       adjust = 0.5 Fires right between where they are and where they're going.
	 *       etc.
	 */
	public void linearGun(ScannedRobotEvent e, double firepower, double adjust) {
		double shootHere = getHeading() + e.getBearing() - getGunHeading() - (adjust * linearGunTurn(e,firepower));
		setTurnGunRight(fixAngle(shootHere));
	}
	/**
	 * End of linearGun:
	 */
	
	/**
	 * oldEnemyMovingInStraightLine: returns true if this enemy is the same enemy as the last scan 
	 *   AND it is not turning or changing its velocity much.
	 */
	public boolean oldEnemyMovingInStraightLine() {
		if (isOldEnemy() && Math.abs(newEnemyBearing - oldEnemyBearing) < 0.12 && Math.abs(oldEnemyVelocity - newEnemyVelocity) < 0.32) {
			return true;
		}
		else {
			return false;
		}
	}
	/**
	 * End of oldEnemyMovingInStraightLine:
	 */
	
	/**
	 * newEnemyStationaryOrFast: returns true if this enemy isn't the one we scanned last
	 *  AND it's moving close to a speed of 0 or 8  (including negatives)
	 */
	public boolean newEnemyStationaryOrFast() {
		if ( !isOldEnemy() && (Math.abs(newEnemyVelocity) <= 0.2 || Math.abs(newEnemyVelocity) >= 7.8) ) {
			return true;
		}
		else {
			return false;
		}
	}
	/**
	 * End of newEnemyStationaryOrFast:
	 */

}
/**
 * End of A_Flat - a robot by Mr. Kiesel
 */
