package chase.s2.stat;
 
import java.util.ArrayList;
import java.util.List;
 
/**
 * This class handles multiple stat buffers, as if they were one.
 */
public class StatBufferSet {
	private static final class StatItem {
		StatBuffer buffer;
		float weight;
	}
 
	private List<StatItem> autobin = new ArrayList<StatItem>();
	
	public final BinUpdater updater;
	
	public StatBufferSet(BinUpdater updater) {
		if (updater == null)
			updater = new BinUpdater.DefaultUpdater();
		this.updater = updater;
	}
 
	/**
	 * Add a stat buffer to this buffer set.
	 */
	public void add(StatBuffer bin, float weight) {
		StatItem si = new StatItem();
		si.buffer = bin;
		si.weight = weight;
		autobin.add(si);
	}
 
	/**
	 * Gets the single best GF from each StatBuffer.
	 */
	public double[] getBestGFList(StatData data) {
		double[] output = new double[autobin.size()];
		int i = 0;
		for(StatItem s : autobin) {
			output[i++] = s.buffer.getBestGF(data);
		}
		return output;
	}
 
	/**
	 * Gets a list of all the bins for the raw data.
	 */
	private float[][] getBinList(StatData data) {
		float[][] output = new float[autobin.size()][];
		int i = 0;
		for(StatItem s : autobin) {
			output[i++] = s.buffer.getStatBin(data);
		}
		return output;
	}
 
	/**
	 * Gets the list of weights.
	 */
	private float[] getWeightList() {
		float[] output = new float[autobin.size()];
		int i = 0;
		for(StatItem s : autobin) {
			output[i++] = s.weight;
		}
		return output;
	}
 
	public float[] getCombinedScoreData(StatData sd) {
		float[][] data = getBinList(sd);
		float[] weights = getWeightList();
		return getCombinedScoreData(data,weights);
	}
 
	/**
	 * This combines the data over all the bins in such
	 * a way that it can be read easily as if it were a
	 * single stat bin.
	 */
	private float[] getCombinedScoreData(float[][] data, float[] weights) {
		if(weights.length != data.length)
			throw new IllegalArgumentException("Data length and weights length must match.");
 
		/* Determine the longest and shortest bin set */
		int shortest = Integer.MAX_VALUE;
		int longest = Integer.MIN_VALUE;
		for(float[] d : data) {
			if(d.length > longest) longest = d.length;
			if(d.length < shortest) shortest = d.length;
		}
 
		/* Get each of their scores, and record the largest of each. */
		float[][] score = new float[data.length][];
		float[] largest = new float[data.length];
		for(int i=0; i<data.length; ++i) {
			score[i] = StatBuffer.getBinScores(data[i], StatBuffer.scanWidth);
			for(int j=0;j<score[i].length;++j)
				if(score[i][j] > largest[i])
					largest[i] = score[i][j]; 
		}
 
		/* Equalize each of the stat bins */
		for(int i=0; i<data.length; ++i)
			for(int j=0;j<score[i].length;++j)
				score[i][j] /= largest[i];
 
		/* If they are equal, take the easy route */
		if(shortest == longest) {		
			largest = new float[shortest];
			for(int i=0; i<data.length; ++i)
				for(int j=0;j<shortest;++j)
					largest[j] += data[i][j] * weights[i];
 
			return largest;
		}
 
		/* Otherwise we have to do things the 'hard' way! */
		/* We use the longest and combine the sums from the others, based on gf */
		largest = new float[longest];
		for(int i=0; i<data.length; ++i) {
			float[] tmpSum = new float[longest];
			/* Get the gf for the new length and multiply it by the old weight */
			for(int j=0; j<score[i].length; ++j) {
				double index = StatBuffer.GFToIndex(StatBuffer.indexToGF(j, score[i].length), longest);
				updater.update(tmpSum, index, score[i][j]);
			}
			/* Add to output buffer */
			for(int j=0; j<longest; ++j) {
				largest[j] += tmpSum[j] * weights[i];
			}
		}
 
		return largest;
	}
 
	/**
	 * Gets the best GF based on weights.
	 */
	public double getBestGF(StatData data) {
		float[] score = getCombinedScoreData(data);
		int bestIndex = 0; /* TODO: This could be problematic */
		for(int i=0; i<score.length; ++i)
			if(score[i] > score[bestIndex])
				bestIndex = i;
		return StatBuffer.indexToGF(bestIndex, score.length);
	}
 
	/**
	 * Updates this stat buffer handler with the given raw data
	 */
	public void update(StatData data, float weight) {
		for(StatItem s : autobin) {
			s.buffer.update(data, weight);
		}
	}
}
