package theo.simple.surf;
import theo.simple.enemy.Enemy;
import theo.simple.utils.botUtils;
import theo.simple.wave.Wave;
import theo.simple.surf.predictor.*;
import robocode.*;
import robocode.util.Utils;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;
import java.awt.Color;
/**
 * LucidSurf - a class by Damij
 */
public class LucidSurf
{

	private static final int NUM_NEIGHBORS = 5;

	AdvancedRobot bot;
	Enemy target;
	List<Enemy> _enemyList;
	List<Double> _neighborList;
	List<Wave> _waveList;
	List<Point2D.Double> _spotList;
	List<Point2D.Double> _futurePoss;
	HashMap<double[],Double> _factorMap;
	private HashMap<String,Double> _statsBuffer;
	
	public Wave currWave,lastWave;
	
	ScannedRobotEvent sre;
	
	public Point2D.Double pos;
	Rectangle2D.Double playField;
	
	public double lastBPower,heading,maxDanger,velocity;
	private double latDirection,lastVelocity,lastHeading,lastLatDirection;
	private double plannedDirection;
	
	private double[] stats;
	
	private long currTime;
	
	private boolean flatten;
	
	
	public LucidSurf(AdvancedRobot bot){
		this.bot = bot;
		
		this._enemyList = new ArrayList<Enemy>();
		this._neighborList = new ArrayList<Double>();
		this._waveList = new ArrayList<Wave>();
		this._factorMap = new HashMap<double[],Double>();
		this._spotList = new ArrayList<Point2D.Double>();
		this._futurePoss = new ArrayList<Point2D.Double>();
		this._statsBuffer = new HashMap<String,Double>();
		
		this.stats = new double[11];
		this.flatten = true;
		
		this.playField = new Rectangle2D.Double(18d,18d,this.bot.getBattleFieldWidth()-36d,this.bot.getBattleFieldHeight()-36d);
		
		this.pos = new Point2D.Double();
	}
	
	public void update(Enemy target, List<Enemy> _enemyList, ScannedRobotEvent e){
		this.target = target;
		this._enemyList = _enemyList;
		this.sre = e;
		this.pos.x = this.bot.getX();
		this.pos.y = this.bot.getY();
		this.heading = this.bot.getHeadingRadians();
		this.velocity = this.bot.getVelocity();

		if(this.velocity!=0.0){
			this.latDirection = (this.velocity*this.sre.getBearingRadians()>=0?1:-1);
			this.lastLatDirection = this.latDirection;
		}
		else{
			this.latDirection = this.lastLatDirection;
		}
		
		if(this.target.energyDelta<=3 && this.target.energyDelta>0){
			this.spawnWave(this.target.energyDelta);
		}
		
		this.advanceWaves();
		this.currWave = this.getClosestSurfableWave();
		if(currWave!=null&&!this.currWave.equals(this.lastWave)){
			this.currWave._neighborList = this.setNeighbors();
		}
		this.lastWave = this.currWave;
		
		this.setStats();
		this.setStatsBuffer();
		
		this.surf();
	}
	
	private void setStatsBuffer(){
		this._statsBuffer.put("posX",this.pos.x);
		this._statsBuffer.put("posY",this.pos.y);
		this._statsBuffer.put("velocity",this.velocity);
		this._statsBuffer.put("latDir",this.latDirection);
	}
	
	private Wave getClosestSurfableWave(){
		Wave closeWave = null;
		double distLeft, timeLeft, leastTime = Double.POSITIVE_INFINITY;
		for(Wave toProp : this._waveList){
			distLeft = Math.abs(toProp.distTraveled - this.pos.distance(toProp.startPoint));
			timeLeft = distLeft/toProp.velocity;
			if(timeLeft < leastTime && timeLeft > 2.0d){
				leastTime = timeLeft;
				closeWave = toProp;
			}
		}
		return closeWave;
	}
	
	private Wave getIncidentWave(Bullet b){
		Wave closeWave = null;
		double distDelta, leastDist = Double.POSITIVE_INFINITY;
		Point2D.Double hitPoint = new Point2D.Double(b.getX(),b.getY());
		for(Wave toProp : this._waveList){
			distDelta = Math.abs(toProp.distTraveled - hitPoint.distance(toProp.startPoint));
			if(distDelta < leastDist & Math.abs(toProp.bPower-b.getPower()) <= 0.1){
				closeWave = toProp;
				leastDist = distDelta;
			}
		}
		return closeWave;
	}
	
	private void spawnWave(double bPower){
		this.lastBPower = bPower;
		Point2D.Double ourPoint = new Point2D.Double(this._statsBuffer.get("posX"),this._statsBuffer.get("posY"));
		
		Wave wave = new Wave();
		wave.bPower = bPower;
		wave.velocity = botUtils.bSpeed(wave.bPower);
		wave.startPoint = (Point2D.Double)target.pos.clone();
		wave.aimPoint = ourPoint;
		wave.absBearing = botUtils.absBearing(wave.startPoint,wave.aimPoint);
		wave.absAngle = botUtils.absAngle(wave.startPoint,wave.aimPoint);
		wave.shotTime = this.bot.getTime()-1;
		wave.distance = target.pos.distance(ourPoint);
		wave.botWidth = 36.0/wave.distance;
		wave.latDirection = this._statsBuffer.get("latDir");
		wave.isReal = true;
		wave.maxEscapeAngle = Math.asin(8.0/wave.velocity)*wave.latDirection;
		wave.stats = this.stats.clone();
		
		this._waveList.add(wave);
	}
	
	private List<Double> setNeighbors(){
		this._neighborList = botUtils.findNearestSurfNeighbors((double[])this.currWave.stats.clone(),_factorMap,NUM_NEIGHBORS);
		return this._neighborList;
	}
	
	private void advanceWaves(){
		this.currTime = this.bot.getTime();
		for(int w = 0; w < _waveList.size(); w ++){
			Wave toProp = _waveList.get(w);
			toProp.setDistTraveled((this.currTime-toProp.shotTime)*toProp.velocity);
			if(toProp.distTraveled > toProp.startPoint.distance(this.pos)-15){
				if(flatten){
					storeWave(toProp);
				}
				_waveList.remove(toProp);
				w--;
			}
		}
	}
	
	private void setStats(){
		this.stats[0] = Utils.normalRelativeAngle((this.heading-target.absAngle))/Math.PI;
		this.stats[1] = target.distance/1000.0;
		this.stats[2] = this.lastBPower/3.0;
		this.stats[3] = 36.0d/target.distance;
		this.stats[4] = this.lastVelocity/8.0;
		this.stats[5] = (this.velocity-this.lastVelocity)/2.0;
		this.stats[6] = this.velocity*Math.sin(target.bearing)/8.0;
		
		this.lastVelocity = this.velocity;
		this.lastHeading = this.heading;
		
		for(int i = 0; i < this.stats.length; i ++){
			this.stats[i] = botUtils.limit(0d,Math.abs(this.stats[i]),1d);
		}
	}
	
	public void onHit(Bullet b){
		Wave hitWave = this.getIncidentWave(b);
		if(hitWave != null){
			this.storeWave(hitWave);
		}
		else{
			System.out.println("We dropped a wave Sir");
		}
	}
	
	private void storeWave(Wave wave){
		this._factorMap.put(wave.stats,this.getStandardFactor(wave,this.pos));
	}

	public void onWin(WinEvent e){
		this._enemyList.clear();
		this._neighborList.clear();
		this._waveList.clear();
		this._spotList.clear();
		this._futurePoss.clear();
	}
	
	public void onDeath(DeathEvent e){
		this._enemyList.clear();
		this._neighborList.clear();
		this._waveList.clear();
		this._spotList.clear();
		this._futurePoss.clear();
	}
	
	public void onPaint(Graphics2D g){
		for(Wave toPaint:_waveList){
			if(toPaint.isReal){
				double radius = toPaint.distTraveled + toPaint.velocity;
				g.setColor((toPaint.isReal?Color.white:Color.yellow));
				if(toPaint.equals(this.currWave)) g.setColor(Color.blue);
				g.drawOval(new Double(toPaint.startPoint.x - radius).intValue(),
					new Double(toPaint.startPoint.y - radius).intValue(),
						new Double(radius*2).intValue(), new Double(radius*2).intValue());
				g.setColor(Color.green);
				Point2D.Double midBin = new Point2D.Double((radius*Math.sin(toPaint.absBearing) + toPaint.startPoint.x),
											(radius*Math.cos(toPaint.absBearing) + toPaint.startPoint.y));
				g.drawOval((int)midBin.x-5,(int)midBin.y-5,10,10);
			}
		}
		for(Point2D.Double pos:this._spotList){
			g.setColor(Color.red);
			g.drawOval(((int)pos.x)-1,((int)pos.y)-1,1,1);
		}
		for(Point2D.Double pos:this._futurePoss){
			g.setColor(Color.green);
			g.drawOval(((int)pos.x)-1,((int)pos.y)-1,1,1);
		}
	}
	
	private void surf(){
		if(this.currWave!=null){
			Point2D.Double desiredSpot = this.computeFuture();
			this.driveTo(desiredSpot);
		}
		else{
			this.emptySurf();
		}
	}
	
	private void emptySurf(){
		double directAngle = botUtils.absAngle(target.pos,this.pos);
		double desiredAngle = Utils.normalAbsoluteAngle(directAngle+Math.PI/2.0d-Math.PI/10.0d*this.plannedDirection);
		double dir = this.plannedDirection;
		if(!this.playField.contains(botUtils.project(this.pos,100*this.plannedDirection,this.heading)) && botUtils.project(this.pos,150*this.plannedDirection,this.heading).distance(target.pos)<this.pos.distance(target.pos)){
			dir *= -1;
		}
		Point2D.Double nextPoint = botUtils.project(this.pos,150*dir,this.findAngle(desiredAngle,this.heading, this.pos, this.latDirection));
		this.driveTo(nextPoint);
	}
	

	////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////////
	/////THIS IS ALL PORTED IN FROM PEQUOD//////////////////////////////
	////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////////


	
	private Point2D.Double computeFuture(){
		_spotList.clear(); _futurePoss.clear();
		double desiredAngle = Utils.normalAbsoluteAngle(target.absBearing+Math.PI/2.0);
		double predVelocity = velocity;
		double timeTillHit = (currWave.distTraveled-
								currWave.startPoint.distance(this.pos))
									/currWave.velocity;
		boolean interrupted = false;
		int ticks = 0;
		double[] dirGroup = new double[]{currWave.latDirection,-1d*currWave.latDirection};
		Point2D.Double point = (Point2D.Double)this.pos;
		double maxTurn = Math.PI/45d, predHeading = this.heading;
		double turnAmount = Utils.normalRelativeAngle(desiredAngle-predHeading);
		double predLatDirection = this.latDirection, lastPredLatDirection = this.latDirection;
		_spotList.add(point);
		double maxVelocity = 8d, turnRemaining = 0d;
		int spotCounter = 0;

		for(double dir: dirGroup){
			point = (Point2D.Double)this.pos;//;
			predVelocity = this.velocity;
			predHeading = this.heading;
			predLatDirection = this.latDirection;
			lastPredLatDirection = predLatDirection;
			interrupted = false;
			ticks = 0;
			desiredAngle = Utils.normalAbsoluteAngle(target.absBearing+Math.PI/2.0);
			turnAmount = Utils.normalRelativeAngle(desiredAngle-predHeading);
			maxVelocity = botUtils.limit(0d,Math.abs(Math.PI/turnAmount),8d);
			turnRemaining = turnAmount;
			spotCounter = 0;
			while(!interrupted){
				desiredAngle = Utils.normalRelativeAngle(botUtils.absAngle(currWave.startPoint,point)+Math.PI/2.0-Math.PI/10.0*predLatDirection);
				desiredAngle = findAngle(desiredAngle,predHeading,point,predLatDirection);
				predVelocity += (predVelocity*dir<0?2*dir:dir);
				predVelocity = botUtils.limit(-1*maxVelocity,predVelocity,maxVelocity);
				predLatDirection = ((predVelocity/*<0?-1:1*/)*
									(Utils.normalRelativeAngle((botUtils.absAngle(point,currWave.startPoint)
										-predHeading)))<0?-1:1);
				if(predLatDirection!=lastPredLatDirection){
					predVelocity = 0;
				}
				lastPredLatDirection = predLatDirection;
				maxTurn = Math.PI/720d*(40d - 3d*Math.abs(predVelocity));
				turnAmount = Utils.normalRelativeAngle(desiredAngle - predHeading);
				if(turnAmount < -1d*Math.PI/2.0 || turnAmount > Math.PI/2.0){
					turnAmount = (Utils.normalRelativeAngle(Math.PI+turnAmount));
				}
				turnRemaining = turnAmount;
				turnAmount = botUtils.limit(-maxTurn,turnAmount,maxTurn);
				turnRemaining = Utils.normalRelativeAngle(turnRemaining - turnAmount);
				if(Math.abs(turnRemaining)>Math.PI/2){
					turnRemaining = (Utils.normalRelativeAngle(Math.PI+turnRemaining));
				}
				maxVelocity = botUtils.limit(0,Math.abs(Math.PI/turnRemaining),8d);
						
				predHeading = Utils.normalAbsoluteAngle(predHeading+turnAmount);
				point = botUtils.project(point,predVelocity,predHeading);
				_spotList.add((Point2D.Double)point.clone());
				ticks ++;
				timeTillHit = (currWave.startPoint.distance(point)
								- currWave.distTraveled - currWave.velocity*ticks - 18)
									/currWave.velocity;
				interrupted = timeTillHit<1.0;
				
				spotCounter++;
			}
		}
		
		double angleToSpot;
		int direction = 1;
		
		Point2D.Double desiredSpotTemp = (Point2D.Double)leastDangerous(_spotList, false);
		angleToSpot = botUtils.absAngle(this.pos,desiredSpotTemp);
		direction = (Math.abs(Utils.normalRelativeAngle(angleToSpot-this.heading))<Math.PI/2.0?1:-1);
		if(direction < 0){
			angleToSpot = Utils.normalAbsoluteAngle(angleToSpot + Math.PI);
		}
		RoboPredicter predicter = new RoboPredicter(this,desiredSpotTemp.distance(this.pos),
										angleToSpot,direction);
										
		this._futurePoss = predicter.getPosList();
		
		desiredSpotTemp = leastDangerous((List<Point2D.Double>)this._futurePoss, false);
		
		return desiredSpotTemp;
	}
	
	private Point2D.Double satisfy(Point2D.Double start, Point2D.Double test, int dir){
		double angleTo = botUtils.absBearing(start,test);
		double angleChanger = Utils.normalRelativeAngle(angleTo-this.heading);
		double distTo = start.distance(test);
		int i = 0;
		while(!this.playField.contains(test)&&i<314){
			angleChanger += 0.01*dir;
			test = botUtils.project(start,distTo,angleChanger+this.heading);
			i++;
		}
		return (Point2D.Double)test;
	}
	
	private Point2D.Double leastDangerous(List<Point2D.Double> list, boolean print){
		double minDanger = Double.POSITIVE_INFINITY;
		double dangers;
		Point2D.Double returnSpot = (Point2D.Double)this.pos;
		this.maxDanger = 0;
		int s = 0;
		for (Point2D.Double testSpot : list){
			dangers = getTotalDanger(testSpot);
			if(dangers > this.maxDanger){
				this.maxDanger = dangers;
			}
			if(dangers<=minDanger){
				minDanger = dangers;
				returnSpot = testSpot;//;
			}
			s++;
			if(print)
				System.out.println("Danger spot # : " + s + " " + dangers + " || at factor " + getStandardFactor(currWave,testSpot));
		}
		if(print)
			System.out.println("Trying for factor: " + getStandardFactor(currWave,returnSpot));
		return (Point2D.Double)returnSpot;
	}
	
	private void driveTo(Point2D.Double spot){
		double angleTo = botUtils.absAngle(this.pos,spot);
		double deltaTheta = Utils.normalRelativeAngle(angleTo - this.heading);
		double dist = this.pos.distance(spot);
		double dir;
		if(deltaTheta>Math.PI/2||deltaTheta<-Math.PI/2){
			dir = -1.0;
			deltaTheta = Utils.normalRelativeAngle(Math.PI+deltaTheta);
		}
		else { dir = 1.0; }
		this.plannedDirection = dir;
		bot.setTurnRightRadians(deltaTheta*Math.signum(dist));
		bot.setMaxVelocity(Math.abs(Math.PI/this.bot.getTurnRemainingRadians()));
		bot.setAhead(dist*dir*Math.signum(dist));
	}
	
	private double getTotalDanger(Point2D.Double pos){
		double danger = 0;
		double angle1 = botUtils.absAngle(currWave.startPoint,pos);
		double botWidths = 36.0/currWave.startPoint.distance(pos);
		for(Double factor : _neighborList){
			double tempDang = getDanger(botWidths, factor, angle1, currWave);
			danger += tempDang;
		}
		return danger;
	}
	
	private static double getDanger(double botWidths, double factor, double angle1, Wave wave){
		double danger;
		double angle2 = angleForOffset(factor,wave);
		double ratio = Utils.normalRelativeAngle((angle2)-(angle1))/botWidths;
		danger = Math.pow(Math.E,-.5d*(ratio*ratio));
		return danger;
	}
	
	private static double angleForOffset(double factor, Wave wave){
		return Utils.normalAbsoluteAngle(factor*wave.maxEscapeAngle + wave.absAngle);//wave.startAngle);
	}
	
	private static double bearingForOffset(double factor, Wave wave){
		return Utils.normalRelativeAngle(factor*wave.maxEscapeAngle + wave.absAngle);//wave.startAngle);
	}
	
	private double getStandardFactor(Wave wave, Point2D.Double poss){
		double theta = Utils.normalRelativeAngle(
						botUtils.absAngle(wave.startPoint,poss)-wave.absAngle);/*(wave.startPoint,wave.aimPoint));*///wave.startAngle);
		return theta/wave.maxEscapeAngle;
	}
	
	private double findAngle(double desiredAngle, double actual, Point2D.Double pos, double dir){
		double toTurn = Utils.normalRelativeAngle(desiredAngle - actual);

		double dist = 120*dir;
		int i = 0;
		Point2D.Double projection = (Point2D.Double)botUtils.project(pos,dist,actual+toTurn);
		while(!this.playField.contains(projection)&&i<315){
			toTurn += dir*0.01;
			projection = botUtils.project(pos,dist,actual+toTurn);
			i++;
		}
		return Utils.normalAbsoluteAngle(actual+toTurn);
	}

}
