package kawigi.spare.util;
import java.util.Vector;
import kawigi.spare.SpareParts;
/**
 * EnemyState - keeps track of one enemy
 */
public class EnemyState
{
	private double x, y, v, heading, energy, deltatheta, lastmovingv;
	private long currentTime;
	private String name;
	private boolean dead;
	private Averager velAve, deltaThetaAve;
	private transient java.text.DecimalFormat fmt = new java.text.DecimalFormat("0.##");
	//used for MogBot-style pattern matcher
	private Vector movements;
	//used for probablistic targetting
	private StatBuffer[] stats;
	
	public EnemyState(String robotName)
	{
		name = robotName;
		dead = false;
		velAve = new Averager(15);
		deltaThetaAve = new Averager(15);
		movements = new Vector();
		int maxdist = (int)Math.sqrt(SpareParts.fieldx*SpareParts.fieldx+SpareParts.fieldy*SpareParts.fieldy);
		stats = new StatBuffer[maxdist/100+1];
		for (int i=0; i< stats.length; i++)
			stats[i] = new StatBuffer("");
	}
	
	public EnemyState copy()
	{
		EnemyState ret = new EnemyState(name);
		ret.dead = dead;
		ret.currentTime = currentTime;
		ret.x = x;
		ret.y = y;
		ret.v = v;
		ret.heading = heading;
		ret.energy = energy;
		return ret;
	}
	
	public Vector getMovements()
	{
		return movements;
	}
	
	public void addMovement(EnemyStateChange change)
	{
		movements.addElement(change);
	}
	
	public void trimMovements()
	{
		while (movements.size()>1000)
			movements.removeElementAt(0);
	}
	
	public void addStat(double spot, int index)
	{
		if (index >= 0 && index < stats.length)
			stats[index].addStat(spot);
	}
	
	public double getStatsBest(double distance)
	{
		int index = (int)(distance/100);
		if (index >= 0 && index < stats.length)
			return stats[index].findBest();
		else
			return 0;
	}
	
	public void setState(double x, double y, double speed, double heading, double energy, long time)
	{
		velAve.addEntry(Math.abs(speed));
		deltatheta = heading-this.heading;
		deltatheta = (deltatheta+Math.PI)%(Math.PI*2)-Math.PI;
		deltatheta /= (time-currentTime);
		deltaThetaAve.addEntry(deltatheta);
		if (speed != 0)
			lastmovingv = speed;
		this.x = x;
		this.y = y;
		v = speed;
		this.heading = heading;
		this.energy = energy;
		currentTime = time;
		dead = false;
	}
	
	public double nextChangeInRelativeAngle(double myx, double myy)
	{
		double angle1 = Math.atan2((x-myx), (y-myy));
		double x2 = x + Math.sin(heading+deltatheta)*v;
		double y2 = y + Math.cos(heading+deltatheta)*v;
		double angle2 = Math.atan2((x2-myx), (y2-myy));
		double ret = angle2-angle1;
		ret = ((ret+Math.PI)%(Math.PI*2)-Math.PI*2)%(Math.PI*2)+Math.PI;
		return ret;
	}
	
	public double getLastHeadingChange()
	{
		return deltatheta;
	}
	
	public String getName()
	{
		return name;
	}
	
	public double getX()
	{
		return x;
	}
	
	public double getY()
	{
		return y;
	}
	
	public double getHeading()
	{
		return heading;
	}
	
	public double getEnergy()
	{
		return energy;
}
	
	public double getVelocity()
	{
		return v;
	}
	
	public long getTimeElapsed(long time2)
	{
		return time2-currentTime;
	}
	
	public boolean isDead()
	{
		return dead;
	}
	
	public void kill()
	{
		dead = true;
	}
	
	public double getRecentAverageVelocity()
	{
		if (v >= 0)
			return velAve.recentAverage();
		else
			return -velAve.recentAverage();
	}
	
	public double getRecentAverageAngularVelocity()
	{
		return deltaThetaAve.recentAverage();
	}
	
	public double getDistance(double myx, double myy)
	{
		return Math.sqrt((myx-x)*(myx-x)+(myy-y)*(myy-y));
	}
	
	public double getAbsoluteBearing(double myx, double myy)
	{
		return Math.atan2((x-myx), (y-myy));
	}
	
	public void updateState(EnemyStateChange change)
	{
		v += change.getChangeInVelocity();
		if (v > 8)
			v = 8;
		else if (v < -8)
			v = -8;
		heading += change.getChangeInHeading();
		x += v*Math.sin(heading);
		y += v*Math.cos(heading);
		currentTime++;
	}
	
	public EnemyStateChange getChange(EnemyState newstate)
	{
		double deltav = newstate.v-v;
		double deltatheta = newstate.heading-heading;
		deltatheta = ((deltatheta+Math.PI)%(Math.PI*2)-Math.PI*2)%(Math.PI*2)+Math.PI;
		long dt = newstate.currentTime-currentTime;
		deltav /= dt;
		deltatheta /= dt;
		return new EnemyStateChange(deltav, deltatheta);
	}
	
	public String toString()
	{
		return name + ":(" + fmt.format(x) + "," + fmt.format(y) + ") - v=" + fmt.format(v) + ", heading " + fmt.format(heading) + " : " + fmt.format(energy);
	}
}
