package ar.horizon.util;

import static ar.horizon.util.Util.*;

import java.awt.geom.*;

/**
 * Each bullet on the battlefield from the opponent (movement) or us (targeting)
 * has a corresponding wave representing the circle of positions it could be in.
 * 
 * @author Aaron Rotenberg
 */
public class Wave {
	/**
	 * This is the recording of the <em>target</em>, not the firer. That is, it
	 * is our recording in the movement, and the enemy's recording in the gun.
	 */
	private final FireRecording recording;
	private final long fireTime;
	private final double bulletPower;
	// TODO: Refactor the precise prediction so that the max escape angles can
	// be made immutable and included in the constructor.
	private double maxEscapeAngleClockwise;
	private double maxEscapeAngleCounterclockwise;

	private double distanceTraveled;

	public Wave(FireRecording recording, long fireTime, double bulletPower) {
		this.recording = recording;
		this.fireTime = fireTime;
		this.bulletPower = bulletPower;
		this.distanceTraveled = getBulletSpeed(bulletPower);
	}

	public FireRecording getFireRecording() {
		return recording;
	}

	public RobotRecording getTargetRecording() {
		return recording.getTargetRecording();
	}

	public RobotRecording getFirerRecording() {
		return recording.getTargetRecording().getEnemyRecording();
	}

	public long getFireTime() {
		return fireTime;
	}

	public double getBulletPower() {
		return bulletPower;
	}

	public double getMaxEscapeAngleClockwise() {
		return maxEscapeAngleClockwise;
	}

	public void setMaxEscapeAngleClockwise(double maxEscapeAngleClockwise) {
		this.maxEscapeAngleClockwise = maxEscapeAngleClockwise;
	}

	public double getMaxEscapeAngleCounterclockwise() {
		return maxEscapeAngleCounterclockwise;
	}

	public void setMaxEscapeAngleCounterclockwise(
			double maxEscapeAngleCounterclockwise) {
		this.maxEscapeAngleCounterclockwise = maxEscapeAngleCounterclockwise;
	}

	public double getDistanceTraveled() {
		return distanceTraveled;
	}

	public void advance(long currentTime) {
		distanceTraveled =
				(currentTime - fireTime) * getBulletSpeed(bulletPower);
	}

	/**
	 * We want to be exact about when a bullet hits the enemy for the purposes
	 * of a gun's GF calculations. The bullet, if it is going to hit the center
	 * point of the enemy next tick, will hit him at his *current* location,
	 * since bullets move before bots in Robocode.
	 * 
	 * @see #isHitPassive(java.awt.geom.Point2D.Double)
	 */
	public boolean isHit(Point2D.Double targetLocation) {
		return distanceTraveled + getBulletSpeed(bulletPower) > targetLocation.distance(getFirerRecording().getLocation());
	}

	/**
	 * This one is only used to discard waves in the movement, so it needn't be
	 * so exact.
	 * 
	 * @see #isHit(java.awt.geom.Point2D.Double)
	 */
	public boolean isHitPassive(Point2D.Double targetLocation) {
		return distanceTraveled > targetLocation.distance(getFirerRecording().getLocation()) + 100;
	}

	public double getGuessFactor(Point2D.Double targetLocation) {
		double bearingOffset =
				angleDifference(absoluteBearing(
						getFirerRecording().getLocation(), targetLocation),
						getFirerRecording().getEnemyAbsoluteBearing());
		return getGuessFactorFromBearingOffset(bearingOffset,
				getTargetRecording().getLateralDirection(),
				maxEscapeAngleClockwise, maxEscapeAngleCounterclockwise);
	}
}
