package ar.horizon.util.graphics;

import java.awt.*;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Point2D;
import java.util.*;

import ar.horizon.util.GuessFactorArray;

import static ar.horizon.util.Util.*;

/**
 * @author Aaron Rotenberg
 */
public final class DrawingUtil {
	private DrawingUtil() {
	}

	/**
	 * This constant determines how "hot" the colors drawn by the movement and
	 * gun appear. Higher numbers make the colors more red, lower numbers make
	 * it more blue.
	 */
	private static final double COLOR_MULTIPLIER = 11.0;

	/**
	 * @param value
	 *            A number between 0.0 and 1.0. Values outside this range are
	 *            clipped to be inside it.
	 * @return A color. Longer wavelength colors have higher values. The colors
	 *         returned range from blue to red.
	 */
	public static Color getIntensityColor(double value) {
		// Note: replacing 2/3 with 5/6 here will make the colors range from
		// purple to red.
		return Color.getHSBColor((float) limit(0.0, 1.0 - value, 1.0)
				* (2.0f / 3.0f), 1.0f, 1.0f);
	}

	/**
	 * @return A list of shapes that the component should request that the robot
	 *         draw.
	 */
	public static Collection<ColoredShape> drawGuessFactorArray(
			GuessFactorArray guessFactorArray, Point2D.Double fireLocation,
			double distance, double absoluteBearing, int lateralDirection,
			double maxEscapeAngleClockwise,
			double maxEscapeAngleCounterclockwise) {
		boolean guessFactorsAvailable = (guessFactorArray != null);

		// Calculate the total values of the GF array, so that we can scale
		// the colors accurately.
		double colorMultiplier = Double.NaN;
		if (guessFactorsAvailable) {
			double totalValue = 0.0;
			for (int i = 0; i < GuessFactorArray.BINS; i++) {
				totalValue += guessFactorArray.getAtIndex(i);
			}

			colorMultiplier = COLOR_MULTIPLIER / totalValue;
		}

		Collection<ColoredShape> shapes = new LinkedList<ColoredShape>();
		for (int i = 0; i < GuessFactorArray.BINS; i++) {
			double guessFactor = GuessFactorArray.getGuessFactorFromIndex(i);
			double angle =
					absoluteBearing
							+ getBearingOffsetFromGuessFactor(guessFactor,
									lateralDirection, maxEscapeAngleClockwise,
									maxEscapeAngleCounterclockwise);

			Point2D.Double waveIntercept =
					project(fireLocation, angle, distance);

			Ellipse2D.Double shape =
					new Ellipse2D.Double(waveIntercept.x - 2,
							waveIntercept.y - 2, 4, 4);
			Color color =
					guessFactorsAvailable ? DrawingUtil.getIntensityColor(guessFactorArray.getAtIndex(i)
							* colorMultiplier)
							: Color.GRAY;
			shapes.add(new ColoredShape(shape, color));
		}

		return shapes;
	}
}
