package tzu.gun;

import tzu.util.*;
import tzu.event.*;
import tzu.intel.*;
import tzu.strategy.*;
import robocode.*;

/**
 * Manages the gun for an AdvancedRobot.
 */

public class GunManager extends AbstractManager {

    FireSolution    fireSolution    = null;
    AimManager      aimManager      = null;
    Bot             target          = null;
    double          gunTurn         = 0.0;
    double          prevGunTurn     = 0.0;
    double          firePower       = 0.0;
    double          aimTolerance    = 0.0;

    /**
     * Creates a gun manager object.
     * @param ar        Reference to your AdvancedRobot.
     * @param tsi       Object which implements the TargetStrategyInterface.
     */
    public GunManager(AdvancedRobot ar, TargetStrategyInterface tsi) {
        super(ar, tsi);
        aimTolerance = BattleField.getRobotSize();
        reinitialize();
        aimManager = new AimManager(ar, tsi);
    }

    /** Reinitialize the gun in second and subsequent rounds. */
    public void reinitialize() {

        target              = null;
        firePower           = 0.0;
        fireSolution        = null;
        gunTurn             = 0.0;
        prevGunTurn         = 0.0;
    }

    /**
     * Pick our next target and calculate a fire solution to hit it.
     * Shoot at our current target if we are ready.
     */
    public void takeTurn() {

        target          = targets.getTarget();
        firePower       = 0.0;
        fireSolution    = null;
        prevGunTurn     = gunTurn;
        gunTurn         = 0.0;

        if (target != null) {
            firePower = FirePower.calculate(target);

            if (firePower > 0.0) {
                fireSolution = aimManager.calcFireSolution(target, firePower);

                if (fireSolution != null) {
                    gunTurn = BotMath.plusMinus180(
                              myRobot.getGunHeading(),
                              fireSolution.gunAngle);

                }
            }
        }
        aimAndShoot();
    }

    /**
     * Choose our firepower, aim at the target, and if odds of
     * hitting it are good, shoot.
     */
    public void aimAndShoot() {

        myRobot.setTurnGunRight(gunTurn);
        double gunTurnRemaining = BotMath.abs(myRobot.getGunTurnRemaining());

        if (fireSolution != null) {

            /*
             * If the target is wiggling back and forth very fast, our
             * gun will wiggle back and forth really fast,
             * gunTurnRemaining will never be within the tolerance amount,
             * and our robot will never shoot.  To counter that, make
             * the tolerance amount a little larger if the target appears
             * to be wiggling.
             */
            if (target.getWigglePct() > 30 &&
                target.getCurrEstMove() < 5 &&
                target.getSpeed() < 3) {
                aimTolerance = BattleField.getRobotSize() * 3;
            } else {
                aimTolerance = BattleField.getRobotSize();
            }

            if (gunTurnRemaining < 20.0         &&
                BotMath.tan(gunTurnRemaining)   *
                BotMath.calcDistance(
                myRobot.getX(), myRobot.getY(),
                fireSolution.getExpectedHitX(),
                fireSolution.getExpectedHitY()) <
                aimTolerance                    &&
                myRobot.getGunHeat() == 0.0)    {

                Bullet b = myRobot.fireBullet(firePower);

                if (b != null) {
                    BulletTracker bt = new BulletTracker(
                            myRobot, b, fireSolution);
                    target.shotAt(firePower);
                }
                return;
            }
        }
        myRobot.execute();
    }
}
