package romz.component.movement.oscillating;

import static romz.math.RobotMath.cos;
import static romz.math.RobotMath.limit;
import static romz.math.RobotMath.random;
import static romz.math.RobotMath.sin;
import robocode.util.Utils;
import romz.component.movement.Movement;
import romz.model.Situation;
import romz.model.advice.MovementAdvice;
import romz.model.robot.Robot;

public class RandomOscillatingMovement implements Movement {
	
	/** The distance on which we want to oscillate */
	private static final double OSCILLATION_DISTANCE = 250;
	
	/** The minimum distance to keep from walls */
	private static final double MIN_WALL_DISTANCE = 100;
	
	/** The desired distance to keep from the situation.enemy */
	private static final double DESIRED_ENEMY_DISTANCE = 200;
	
	/**  Minimum distance adjustment factor, to avoid running straight ahead to the situation.enemy. */
	private static final double MIN_DISTANCE_ADJUSTMENT_FACTOR = 0.25;
	
	/** Maximum distance adjustment factor, to avoid running straight away from the situation.enemy */
	private static final double MAX_DISTANCE_ADJUSTMENT_FACTOR = 1.75;
	
	/** The greatest, the more we open or close our default perpendicular angle to adjust distance */
	private static final int DISTANCE_SENSITIVITY = 20;
	
	private Situation situation;
	
	private int gear = 1;

	@Override
	public MovementAdvice getMovementAdvice(Situation situation) {
		this.situation = situation;
		MovementAdvice movementAdvice = new MovementAdvice();
		if (situation.enemy != null) {
			movementAdvice.heading = getHeadingForAdjustedPerpendicularHeading();
		}
		if (reverseNeeded()) {
			gear = - gear;
			movementAdvice.distance = (0.5 + random()) * OSCILLATION_DISTANCE;
		}
		movementAdvice.direction = gear;
		return movementAdvice;
	}
	
	/**
	 * Calculates the heading.
	 * We always try to keep a perpendicular heading to the situation.enemy, for maximum escape angle.
	 * We always try to keep the DESIRED_ENEMY_DISTANCE from the situation.enemy:
	 * 	 If too close, we open the angle to move away
	 *   If too far, we close the angle to come closer
	 */
	private double getHeadingForAdjustedPerpendicularHeading() {
		int side = situation.enemy.relativeBearing > 0 ? 1 : -1;
		double distanceAdjustmentFactor = getDistanceAdjustmentFactor();
		double adjustedPerpendicularAngle = distanceAdjustmentFactor * 90;
		double adjustedPerpendicularHeading = Utils.normalAbsoluteAngleDegrees(situation.enemy.absoluteBearing - side * adjustedPerpendicularAngle);
		return adjustedPerpendicularHeading;
	}

	/**
	 * Calculates the distance adjustment factor.
	 * This factor will multiply our perpendicular angle of 90,
	 * in order to open or close the angle to keep constant distance from situation.enemy.
	 */
	private double getDistanceAdjustmentFactor() {
		double rawDistanceAdjustmentFactor = 1 + (DISTANCE_SENSITIVITY * (gear * (DESIRED_ENEMY_DISTANCE - situation.enemy.distance)) / DESIRED_ENEMY_DISTANCE);
		return limit(MIN_DISTANCE_ADJUSTMENT_FACTOR, rawDistanceAdjustmentFactor, MAX_DISTANCE_ADJUSTMENT_FACTOR);
	}

	private boolean reverseNeeded() {
		return situation.hero.distanceRemaining == 0 || tooCloseFromWall();
	}

	private boolean tooCloseFromWall() {
		double trueHeading = gear > 0 ? situation.hero.heading : situation.hero.heading + 180;
		double projectedX = situation.hero.x + MIN_WALL_DISTANCE * sin(trueHeading);
		double projectedY = situation.hero.y + MIN_WALL_DISTANCE * cos(trueHeading);
		return projectedX < Robot.HALF_ROBOT_SIZE
			|| projectedX > situation.battleField.width - Robot.HALF_ROBOT_SIZE
			|| projectedY < Robot.HALF_ROBOT_SIZE 
			|| projectedY > situation.battleField.height - Robot.HALF_ROBOT_SIZE;
	}

}
