package gh.micro;
import robocode.*;
import robocode.util.Utils;
import java.awt.Color;
import java.awt.geom.*;

/**
 * Gruweltje - a microbot by Gert Heijenk (GrubbmGait)
 *
 * This is the result of the mating of one GrubbmOgre and four GrubbmGrunts
 * It has the muscles of its fathers and the broad view of its mother
 * Any child of close relatives has more chance of DNA-defects,
 * in this case it lacks communicative and contactual skills, and has trouble moving smoothly near walls.
 * It compensates its (lack of) size with its enthousiasm.
 *
 * Revision information:
 * v0.1 20050403 Initial version. Stripped mini.Gruwel.
 * v0.2 20050412 Replaced radar with nano-style meleeradar (-74 bytes).
 *				 Improved first targetselection.
 *				 Bounce when wallhit is eminent.
 *				 Two minor gun-related (codesize)tweaks.
 * v0.3 20050604 Tweak codesize (-40 bytes)
 *				 Bounce more unpredictable
 *				 Stay away from walls/corners better
 * v0.4 20050605 Keep focussed on target if someone else dies (melee only)
 *				 Squeeze some code to make it fit
 * v0.5 20050612 Make short turns (< 90) to improve melee
 *				 Squeeze again (and again and again)
 * v0.6 20051221 Keep focussed on target if someone else dies (melee only)
 * v0.7 20090904 Immediately find a new target if current one dies
 *               Less preservive of energy on longer distances (better score, lower survival ??)
 *               Keep looking around when not enough energy for bullet
 */

public class Gruweltje extends AdvancedRobot {

	// movement variables
	private static long		currTime;		// the current time (mainloop, but also used in event)
	private static double	angleOffset;	// the angleoffset to prevent to get stuck in corners
	private static double	shortturn = 1;	// shortturn -1 meaning turning 90 degrees instead of 270
	
	// some info about the current locked opponent
	private static String	eName;		// enemy name
	private static double	eAttrac = Double.POSITIVE_INFINITY;	// enemy attractivity (less is better)
	private static double	oldHeading;	// the previous heading of the selected opponent

	// some gun-related statistics
	private static Rectangle2D.Double fireField;	// the shootable battlefield

	/**
	 * run: Gruweltjes default behavior
	 */
	public void run() {
		double wallindex;		// random factor of blindmansstick
		double dist = 0;		// the distance to go 

		// Give the robot an appealing look
		setColors( Color.red, Color.white, null);

		// First set the statistics
        fireField = new Rectangle2D.Double( 17, 17, getBattleFieldWidth() - 34, getBattleFieldHeight() - 34);

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
//		setAdjustRadarForGunTurn( true);		// is this necessary ??
		setTurnRadarRight( Double.POSITIVE_INFINITY);

		// mainloop
		while( true ) {
			currTime = getTime();

			// Check if nearly at end of movement
			if (Math.abs(getDistanceRemaining()) < 1) {
				dist = (( Math.sin( currTime/11) * Math.cos( currTime/29))*270 + (Math.random()*100 - 50)) * shortturn;
				setAhead( dist);
				dist = (dist < 0 ? -20 : 20);	// very small blindmansstick
				angleOffset = 0;	// no angleoffset anymore, normal new movement
			}

			// if endposition not in field, just reverse direction
			// the variable 'dist' is used as wallcheck here
			wallindex = (1 + 2 * Math.random()) * dist;	// a bit unpredictable bouncepoint (20 < x < 60)
			if (!fireField.contains( getX() + Math.sin( getHeadingRadians()) * wallindex,
                                     getY() + Math.cos( getHeadingRadians()) * wallindex)) {
				setAhead( -getDistanceRemaining());
				angleOffset = dist / 50 * shortturn;	// near-wall bounce, so go a bit towards enemy
				dist = -dist;
			}

			// perform all actions previously set
			execute();
		}
	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {

//		double tmpBearing = getHeadingRadians() + e.getBearingRadians();
		double power;
		double dist = e.getDistance();

		// first check attractivity of scanned robot
		if (!(e.getName().equals( eName))) {
			if (dist < (eAttrac * 0.8)) {
				eName = e.getName();	// new more attractive enemy
			}
			else {
				return;		// new not very appealing enemy
			}
		}
		eAttrac = dist;	// set new attractivity

		// Stay perpendicular to the opponent when in lock-mode
		power = Utils.normalRelativeAngle(e.getBearingRadians() + Math.PI/2 + angleOffset);
		setTurnRightRadians( Math.atan( Math.tan( power)));
		shortturn = (power != Math.atan( Math.tan( power)) ? -1 : 1);
		
		// At start of round, always perform two radarsweeps without locking
		if (currTime < 16) return;

		// set the power of the bullet, including energymanagement
		power = (dist > 800 ? 0.15 : (dist > 250 ? 1.9 : 3.0));
		power = Math.min( getEnergy()/5, Math.min( (e.getEnergy()/4) + 0.1, power));

		// perform lineair and circular targetting
		long deltahittime;
		Point2D.Double point = new Point2D.Double();
		double head, chead, bspeed;
		double myX, myY;
		double tmpx, tmpy;

		// perform an iteration to find a reasonable accurate expected position
		tmpx = (myX = getX()) + Math.sin( getHeadingRadians() + e.getBearingRadians()) * dist;
		tmpy = (myY = getY()) + Math.cos( getHeadingRadians() + e.getBearingRadians()) * dist;
		head = e.getHeadingRadians();
		chead = head - oldHeading;
		oldHeading = head;			// save current heading for next tick
		point.setLocation( tmpx, tmpy);
		deltahittime = 0;
		do {
			tmpx += Math.sin( head) * e.getVelocity();
			tmpy +=	Math.cos( head) * e.getVelocity();
			head += chead;
			deltahittime++;
			// if position not in field, adapt
			if (!fireField.contains( tmpx, tmpy)) {
				bspeed = point.distance( myX, myY) / deltahittime;
				power = Math.max( Math.min( (20 - bspeed) / 3.0, 3.0), 0.1);
//				power = (bspeed < 19.7 ? (20 - bspeed) / 3.0 : 0.1);
				break;
			}
			point.setLocation( tmpx, tmpy);
		} while ( (int)Math.round( (point.distance( myX, myY) - 18) / Rules.getBulletSpeed( power)) > deltahittime);

		// If the gun is cool and aligned, and I do not disable myself: Fire !
		if ((getGunHeat() == 0.0) && (getGunTurnRemaining() == 0.0)) {
			if (getEnergy() > 0.1) {
				setFire( power);
			}
			else {
				setTurnRadarRight( Double.POSITIVE_INFINITY);
			}
		}
		else {
			// when not firing, lock on target. When firing, do not lock so perform a sweep
			setTurnRadarLeft(getRadarTurnRemaining());
		}
		// Turn gun after eventual firing
		setTurnGunRightRadians( Utils.normalRelativeAngle(((Math.PI / 2) - Math.atan2( point.y - myY, point.x - myX)) - getGunHeadingRadians()));
	}

	/**
	 * onRobotDeath: What to do when someone else dies
	 */
	public void onRobotDeath(RobotDeathEvent e) {
		// If current target dies, force to find another
		if (eName.equals( e.getName())) {
			eAttrac = Double.POSITIVE_INFINITY;		// make least attractive
			setTurnRadarRight( Double.POSITIVE_INFINITY);
		}
	}
}