package bvh.fnr;

import java.io.*;
import java.util.*;
import java.util.Random;
import java.util.zip.*;
import java.awt.geom.Point2D;
import java.awt.Color;
import java.text.*;

import robocode.*;
import robocode.util.Utils;

/*
import robocode.robocodeGL.*;
import robocode.robocodeGL.system.*;
*/

/**
 * Fenrir - a robot by Bart van Hest
 *
 *   "Fenrir is a gigantic and terrible monster in the shape of a wolf. He is the eldest child
 *    of Loki and the giantess Angrboda. The Asgard gods learned of a prophecy which stated that
 *    the wolf and his family would one day be responsible for the destruction of the world."
 *
 * Fenrir is aimed to be a melee-specialist...
 *
 * The movement can be called my own, it's a simplified potential field calculation
 * where the immediate surroundings of the bot are geven hazard values denpending
 * on wall-distance, other bots and estimated bullet positions. It has therefore
 * similarities with ag-motion and minimum risk movement.
 * The strong gun is based on the pattern analyzer originally found in nano.Laulectric
 * and later in Moebius, with movement data stored for each opponent in this melee
 * version.
 * Thanks to Albert Prez and Michael Dorgan for publishing there very strong pattern
 * analyzer.
 *
 *  TODO:
 *  - melee kanon vs 1-vs-1 kanon
 *  - max. schiethoek inbouwen: Fenrir schiet te vaak buiten slagveld/ naar te grote hoeken
 *  - constanten voor hoeken hernoemen
 *  - tegen TaoW en org.statisbot laten lopen om statistiek van tegenstander te bestuderen:
 *    piek op hoek=0 ???!!!
 *
 * Wijzigingshistorie v0.36 (25-10-2004):
 *  - schieten met bp 1.9 (of 3.0 op korte afstanden)
 *  - kanon pas richten indien kanon bijna afgekoeld: sneller, want minder PM-effort.
 *    schieten indien kanon koud
 *  - doelpositie voorspelling obv. PM-match aangepast: minder ver in toekomst.
 *
 * Wijzigingshistorie v0.35 (??-??-2004):
 *  - (mislukte) random kogelsnelheid.
 *
 * Wijzigingshistorie v0.34 (18-05-2004):
 *  - terug naar v0.33 mbt beweging
 *  - kogelkracht niet langer gebaseerd op constante overbruggingstijd, maar
 *    meer random om het de wavesurfers lastiger te maken.
 *
 * Wijzigingshistorie v0.34 (18-05-2004):
 *  - eerder een nieuwe beweging inzetten: hielp veel bij mini.Fenrir v0.35
 *
 * Wijzigingshistorie v0.33 (29-02-2004):
 *  - v0.300 ongedaangemaakt: wel navigatie()-method vereenvoudigd.
 *
 * Wijzigingshistorie v0.322 (29-02-2004):
 *  - v0.300 ongedaangemaakt: wel navigatie()-method vereenvoudigd.
 *
 * Wijzigingshistorie v0.310 (29-02-2004):
 *  - hopeloze poging om v0.3 op te lappen....
 *
 * Wijzigingshistorie v0.300 (27-02-2004):
 *  - incl. wall smoothing...
 *
 * Wijzigingshistorie v0.297 (19-02-2004):
 *  - meest vlakke bewegingsprofiel, leverd geen voordeel zie onder.
 *
 zonder data:
 1st: bvh.fnr.Fenrir 0.297			    15461	4950	990	8382	1120	19	0	99	101	0
 2nd: wiki.CoriantumrWithDMMGun 1.02 14617	5050	1010	7545	1002	9	0	102	99	0
 met data:
 1st: bvh.fnr.Fenrir 0.297			    15572	4950	990	8491	1127	13	0	99	101	0
 2nd: wiki.CoriantumrWithDMMGun 1.02 14776	5050	1010	7753	961	1	0	103	99	0

 zonder data:
 1st: bvh.fnr.Fenrir 0.292		        18230	6650	1330	8828	1417	4	0	133	67	0
 2nd: wiki.CoriantumrWithDMMGun 1.02 10900	3350	670	6264	616	0	0	68	133	0
 met data:
 1st: bvh.fnr.Fenrir 0.292			    17659	6250	1250	8797	1346	15	0	125	75	0
 2nd: wiki.CoriantumrWithDMMGun 1.02 11776	3750	750	6560	710	4	0	76	125	0

 zonder data:
 1st: pez.frankie.Frankie 0.9.6.1	16441	5900	1180	8174	1187	0	0	119	82	0
 2nd: bvh.fnr.Fenrir 0.292	         13480	4100	820	7708	848	3	0	84	118	0
 met data:
 1st: pez.frankie.Frankie 0.9.6.1	18092	6300	1260	9225	1299	7	0	128	74	0
 2nd: bvh.fnr.Fenrir 0.292		      13654	3700	740	8298	911	4	0	74	126	0

 zonder data:
 1st: pez.frankie.Frankie 0.9.6.1	19569	6900	1380	9736	1549	3	0	139	62	0
 2nd: bvh.fnr.Fenrir 0.297	         12294	3100	620	7838	712	23	0	62	138	0
 met data:
 1st: pez.frankie.Frankie 0.9.6.1	19411	7200	1440	9244	1523	3	0	144	56	0
 2nd: bvh.fnr.Fenrir 0.297	         11754	2800	560	7749	631	13	0	56	144	0

 zonder data:
 1st: pez.Marshmallow 1.9.5	14512	5250	1050	7186	944	7	74	106	95	0
 2nd: bvh.fnr.Fenrir 0.297	   12957	4750	950	6462	790	3	0	96	105	0
 1st: pez.Marshmallow 1.9.5	17467	6250	1250	8618	1224	8	116	125	75	0
 2nd: bvh.fnr.Fenrir 0.292	   11455	3750	750	6314	639	1	0	76	125	0

 *
 * Wijzigingshistorie v0.294 (18-02-2004):
 *  * nieuwe navigatiemodule gebaseerd op mini.Fenrir v0.34
 *
 *
Rank Robot Score Survival Last surv. b. Bullet dmg. Bullet dmg. b. 2*Ram dmg. Ram dmg. b. #1st #2nd #3rd
1 Musashi 2.18 19405 7217 1443 9233 1498 13 0 146 56 0
2 Fenrir 0.294 13217 2900 580 8879 840 18 0 58 142 0
3 Fenrir 0.293 12371 2900 580 8115 757 18 0 58 142 0
4 Fenrir 0.292 11917 2550 510 8166 679 12 0 51 149 0

Rank Robot Score Survival Last surv. b. Bullet dmg. Bullet dmg. b. 2*Ram dmg. Ram dmg. b. #1st #2nd #3rd
1 Lacrimas 1.33 20348 7300 1460 9960 1628 0 0 148 54 0
2 Fenrir 0.292 12662 3200 640 8062 742 17 0 64 136 0
3 Fenrir 0.294 10769 2650 530 7000 584 4 0 53 147 0
4 Fenrir 0.293 9793 2250 450 6620 468 4 0 45 155 0

Rank Robot Score Survival Last surv. b. Bullet dmg. Bullet dmg. b. 2*Ram dmg. Ram dmg. b. #1st #2nd #3rd
1 FloodMini 1.4 21399 6700 1340 11542 1773 24 18 134 66 0
2 Fenrir 0.294 14974 2850 570 10641 862 50 0 57 143 0
3 Fenrir 0.293 14389 3600 720 9155 897 16 0 73 128 0
4 Fenrir 0.292 12223 3450 690 7340 729 13 0 69 131 0

Rank Robot Score Survival Last surv. b. Bullet dmg. Bullet dmg. b. 2*Ram dmg. Ram dmg. b. #1st #2nd #3rd
1 Muffin 0.6 18446 5600 1120 10294 1426 5 0 113 88 0
2 Fenrir 0.292 15191 4550 910 8741 969 20 0 93 109 0
3 Fenrir 0.294 14373 4300 860 8226 963 23 0 87 114 0
4 Fenrir 0.293 14343 4350 870 8193 904 25 0 90 113 0

Rank Robot Score Survival Last surv. b. Bullet dmg. Bullet dmg. b. 2*Ram dmg. Ram dmg. b. #1st #2nd #3rd
1 AspidReloaded 0.6 16889 5167 1033 9428 1250 10 0 105 97 0
2 Fenrir 0.293 15342 5200 1040 8038 1051 11 0 108 96 0
3 Fenrir 0.294 14548 4500 900 8193 944 10 0 91 110 0
4 Fenrir 0.292 14249 4800 960 7550 929 9 0 99 104 0

Rank Robot Score Survival Last surv. b. Bullet dmg. Bullet dmg. b. 2*Ram dmg. Ram dmg. b. #1st #2nd #3rd
1 Aristocles 0.3.5 11258 3850 770 5678 958 2 0 77 23 0
2 Fenrir 0.294 6618 1750 350 4120 392 4 0 35 65 0
3 Fenrir 0.293 6416 1650 330 4047 375 13 0 33 67 0
4 Fenrir 0.292 6070 1500 300 3873 394 2 0 30 70 0
5 Fenrir 0.295 4017 450 90 3359 112 6 0 9 91 0
6 Fenrir 0.296 3561 400 80 2975 105 0 0 8 92 0

 *
 * Wijzigingshistorie v0.292 (12-02-2004):
 *  * naamgeving constanten conform gebruik in Freya.
 * TC (250 ronden) gemiddelde van 80.5
 * Tron:                 75.31
 * SandboxDT 1.91        62.97
 * Cigaret 1.31TC        60.07
 * HTTC 1.0              78.06
 * AspidMovement 1.0     82.96
 * Fhqwhgads 1.1TC       80.20
 * FunkyChickenTC 1.0    79.20
 * Sparrow 2.5TC         93.89
 * Yngwie 1.0            93.05
 * TaoW                  99.02
 *
 *
 * Wijzigingshistorie v0.291 (12-02-2004):
 *  * controle op beschikbaarheid ruimte obv. code uit tzu.TheArtOfWar.
 *
 * Wijzigingshistorie v0.290 (12-02-2004):
 *  * combinatie van beweging v285/286 + dataopslag van v288.
 *
 * Wijzigingshistorie v0.288 (09-02-2004):
 *  * data voor patroonherkenning wordt per doel opgeslagen en ingelezen bij
 *    een volgend gevecht tegen dezelfde tegenstander. Max. data segment is
 *    10000 samples (ipv 5000), genomen in laatste deel van array zodat data
 *    uit 'midden fase'PI van een gevecht wordt genomen.
 *
 * match zonder data:
 * 1st: apv.AspidReloaded 0.6 15668 4950  990   8594  1131  2  0  99   101 0
 * 2nd: bvh.fnr.Fenrir 0.288  14775 5050  1010  7731  979   3  0  103   99 0
 * match met data:
 * 1st: apv.AspidReloaded 0.6 19517 7200  1440  9326  1523  27 0  145   56 0
 * 2nd: bvh.fnr.Fenrir 0.288  8986  2800  560   5138  458   28 0  57   144 0
 *
 * no saved data:
 * 1st: kawigi.mini.Coriantumr 1.0  17008 5700  1140  8899  1268  1  0  115   86 0
 * 2nd: bvh.fnr.Fenrir 0.288        14244 4300  860   8148  924   12 0  86   114 0
 * with saved data:
 * 1st: kawigi.mini.Coriantumr 1.0  19876 7700  1540  9076  1560  0  0  155   46 0
 * 2nd: bvh.fnr.Fenrir 0.288        7911  2300  460   4739  400   12 0  46   154 0
 *
 * Wijzigingshistorie v0.287 (09-02-2004):
 *  * opgeslagen data voor patroonherkenning 'periodiek' gemaakt zodat er geen
 *    limiet meer is aan het maximum aantal ronden in een gevecht.
 *  - een test heeft uitgewezen dat het kanon van mini.Fenrir geschikter is in
 *    melee-gevechten: misschien dit kanon overnemen in Fenrir en gebruiken
 *    tijdens melee.
 *
 * Wijzigingshistorie v0.286 (06-02-2004):
 *  * alleen locked tegenstanders meenemen in bepaling nieuwe potentiaal.
 *  * bij dood van tegenstander deze vrijgeven en controleren of doel nog locked is.
 *  * Musashi-trick geimplementeerd: niet echt een sukses...
 *
 * Wijzigingshistorie v0.285 (05-02-2004):
 *  * vereenvoudigingen in navigatie-method:
 *  - volgens GuesFactor kanon van Freya v0.037 is heeft Fenrir v0.285 de meest vlakke kurve !!!
 *
 *  Rank Robot        Score Survival Last surv. b. Bullet dmg. Bullet dmg. b. 2*Ram dmg. Ram dmg. b. #1st #2nd #3rd
 *  1 Shadow 2.33     15637  5415  1083  7723  1408   4  3  109   17  0
 *  2 Fenrir 0.285     7047  1550   310  4818   366   2  0   31   94  0
 *  3 Fenrir 0.280     6349  1250   250  4536   312   1  0   25  100  0
 *  4 Fenrir 0.2853    6265   950   190  4857   264   3  0   19  106  0
 *  5 Fenrir 0.2854    6036   800   160  4878   193   4  0   16  109  0
 *  6 Fenrir 0.2852    5784   800   160  4614   205   3  0   16  109  0
 *  7 Fenrir 0.2851    5400   700   140  4352   202   4  0   14  111  0
 *  8 Fenrir 0.287     5172   300    60  4549   101 162  0    6  119  0
 *  9 Fenrir 0.23i     5153   950   190  3801   211   0  0   19  106  0
 *  10 TheArtOfWar 1.2 5042   750   150  3966   176   0  0   15  110  0
 *  11 Fenrir 0.286    4167   300    60  3691   80   34  0    7  119  0
 *
 * Wijzigingshistorie v0.280 (02-02-2004):
 *  * v0.28 is fusie tussen v0.193 (melee-kampioenn in de Fenrir serie) en de
 *    positieve eigenschappen van v0.25/0.266 in 1-vs-1!
 *  - voorwaarden in onScannedRobot vereenvoudigd
 *  - stoppen bij scherpe draai in meleeModus, verkorten loopafstand in 1-vs-1 in
 *    navigatie-method.
 *  - onderscheid tussen meleeModus of niet, bij bepaling potentiaal: in meleeModus
 *    minder random beweging en minder kogelposities meenemen.
 *  - onderscheid tussen meleeModus of niet, bij bepaling loopafstand in bepaalXY
 *
 * Wijzigingshistorie v0.27 (28-01-2004):
 *  - tussenversie met mislukte bewegingsroutine...
 *
 * Wijzigingshistorie v0.266 (28-01-2004):
 *  - Doorgeven correcte waarden vanuit Doel naar Kogel!
 *  - nauwkeuriger bepaling dichtsbijzijnde kogel
 *  - minimale verbetering tegen meeste goede bots.
 *  - v0.266 = 262ttt
 *
 * Wijzigingshistorie v0.262 t/m 265 (26-01-2004):
 *  - diverse testversies: --> slechter vs. TaoW !
 *
 * Wijzigingshistorie v0.261 (26-01-2004):
 *  - nieuwe beweging indien weg_te_gaan < random*5 ipv 5: --> slechter vs. TaoW !
 *
 * Wijzigingshistorie v0.26 (26-01-2004):
 *  - variatie in snelheid: max. bij uitwijken kogel, random bij gewone beweging;
 *  - stoppen of bewegingsrichting omkeren indien geraakt door kogel.

1st: bvh.fnr.Fenrir 0.26 11712               6579        0  0  69 31 0
2nd: tzu.TheArtOfWar 1.2   7177              4918        0  0  31 69 0

1st: bvh.fnr.Fenrir 0.25   9119  2350  470   5638  659   1  0  47 53 0
2nd: tzu.TheArtOfWar 1.2   8700  2650  530   4873  647   0  0  54 47 0

 * Wijzigingshistorie v0.25 (27-10-2003):
 *  - zuiniger omgaan met energie/langer schieten (methode uit nieuwe bot Wodan)
 *
 * Wijzigingshistorie v0.24 (27-10-2003):
 *  - zuiniger omgaan met energie/langer schieten (methode uit nieuwe bot Wodan)
 *
 ** NOTE: Fenrir v0.193 is far better at melee, Fenrir v0.23 is better at 1-vs-1: why ???
 *
 * Wijzigingshistorie v0.23i (01-07-2003): Rumble version
 *  - 42% in Wiki.PMC
 *  - 13% in Wiki.CFC
 *
 * Wijzigingshistorie v0.23h (01-07-2003):
 *  - 42% in Wiki.PMC
 *  - 13% in Wiki.CFC
 *
 * Wijzigingshistorie v0.23g (01-07-2003):
 *  - 21.8% in Wiki.PMC
 *  -  2.6% in Wiki.CFC
 *
 * Wijzigingshistorie v0.23f (30-06-2003):
 *  + 23f is beste uit 23 en 23e:
 *    - variabele snelheid
 *    - oude kogelafstotingswaarde
 *  TO DO:
 *  - bug oplossen bij dood tegenstander in melee: Fenrir zoekt dan niet naar nieuw doel!:
 *
 * Wijzigingshistorie v0.23 (21-06-2003):
 *  - DEBUG versie incl. GL4Java
 *  - bug voor bepalen kogel afstand verwijderd
 *  - mbv dichtsbijzijnde kogel wordt bij een hit de echte hoek bepaald waaronder
 *    de kogel is afgeschoten ipv de aangenomen hoek (direct schot).
 *
 *  TO DO:
 *  - versie e probeerd onderstaande:
 *  - juiste Sign inbouwen bij bepaling gecorrigeerde doelrichting: grootte van de
 *    de hoek is min of meer ok, de richting afhankelijk maken van de bewegingsrichting
 *    van Fenrir !!!
 *
 *
 *  resultaten: 50 battles op 800x600:
 *  1st: challenge.nano.PatternBotRMC  4931  900   180   3493  358   0  0  18 32 0
 *  2nd: bvh.fnr.Fenrir RMC0.23c       1920  1600  320   0     0     0  0  32 18 0
 *
 *  - verbeterde weging kogelpositie in potentiaal
 *  1st: challenge.nano.PatternBotRMC  4033  550   110   3155  218   0  0  11 39 0
 *  2nd: bvh.fnr.Fenrir RMC0.23c       2341  1950  390   0     0     1  0  39 11 0
 *
 * Wijzigingshistorie v0.22 (19-06-2003):
 *  - variable maximale snelheid bij eigen beweging
 *  - doelpositie schatter spiegeld geschatte positie naar positie binnen slagveld
 *
 *  resultaten: 100 battles op 1000x1000:
 *  1st: tzu.TheArtOfWar 1.2  9923  2950  590   5619  764   0  0  59 41 0
 *  2nd: bvh.fnr.Fenrir 0.22  8285  2050  410   5252  572   0  0  41 59 0
 *
 * Wijzigingshistorie v0.21 (19-06-2003):
 *  - matched op transversale snelheid en versnelling
 *  - voorspeld positie obv wijziging bewegingsrichting en snelheid
 *  - max. zoeklengte van 40 naar 50 (afname in stapjes van 2)
 *
 *  resultaten: 100 battles op 1000x1000:
 *  1st: tzu.TheArtOfWar 1.2  10511 3150  630   5873  857   0  0  64 37 0
 *  2nd: bvh.fnr.Fenrir 0.21  8234  1850  370   5463  551   0  0  38 63 0
 *  1st: tzu.TheArtOfWar 1.2  10242 3100  620   5696  826   0  0  62 38 0
 *  2nd: bvh.fnr.Fenrir 0.21  8325  1900  380   5501  542   1  0  38 62 0
 *
 * Wijzigingshistorie v0.2 (18-06-2003):
 *  - variabele pattern matcher + verbeteringen als gevolg van Wiki.PatternMatcherCompetition
 *
 * Wijzigingshistorie v0.195 (27-05-2003):
 *  - menemen energie tegenstander in berekening factor: beter nabij bot met lagere
 *    energie dan bot met hogere energie bij gelijke afstand
 *  - data persistentie toegevoegd: patroon wordt opgeslagen naar file-systeem.
 *  - verbeterde naamgeving va files: bot-naam zonder versienummer+.dat extensie.
 *  to do:
 *  - subset van data opslaan
 *
 * Wijzigingshistorie v0.194 (26-05-2003):
 *  - bewegingspatroon array periodiek gemaakt: meer dan pm. 200 matches mogelijk,
 *    maar bot verliest kennis bestaande patronen als maximale lengte is bereikt.
 *    Resulteerd in gelijke score bij 200 matches.
 *  - in melee doel selecteren indien dichter bij dan 300:--> geen verbetering!
 *
 *  resultaten: 300 battles op 1000x1000:
 *  1st: abc.Tron 1.5         46466 22500 4160  17599 2201  4  0  211  34   57
 *  2nd: bvh.fnr.Fenrir 0.194 24209 11300 1000  11042 857   9  0   50 126  124
 *  3rd: bvh.fnr.Fenrir 0.193 23762 11100 820   10920 917   4  0   42 140  119
 *
 * Wijzigingshistorie v0.193 (21-05-2003):
 *  - doel-statistiek uitgebreid met berekening gemiddelde afstand en energie
 *    verlies van de tegenstander en de eigen bot.
 *
 * Wijzigingshistorie v0.192 (21-05-2003):
 *  -
 *
 * Wijzigingshistorie v0.191 (16-05-2003):
 *  - variabele factor bij berekening potentiaal.
 *
 * Wijzigingshistorie v0.190 (16-05-2003):
 *  - doel-statistiek toegevoegd
 *
 * Wijzigingshistorie v0.189 (15-05-2003):
 *  - versie 189, maar met zoekdiepte = 5
 *
 * Wijzigingshistorie v0.189 (15-05-2003):
 *  * kruising 186,187,188
 *  - alleen indien een patroon-lock is gevonden wordt er hard geschoten
 *  - alleen zware kogels meenemen in uitwijkbeleid
 *  --> conclusie: zoekdiepte=20 vind nooit een match!
 *  1st: tzu.TheArtOfWar 1.2  21937 7900  1340  11275 1420  1  0  67 24 59
 *  2nd: bvh.fnr.Fenrir 0.184 17648 5900  260   10246 1195  46 0  13 92 45
 *  3rd: bvh.fnr.Fenrir 0.189 11232 8700  1400  1021  60 51 0  70 34 46
 *
 *
 * Wijzigingshistorie v0.188 (15-05-2003):
 *  - variabele kogelkracht, zoekdiepte = 20
 *  - alle kogels meenemen in uitwijkbeleid
 *  --> identiek 0.184: resultaat 150 ronden tegen Tzu: exact gelijke overleving, meer kogelschade!
 *
 * Wijzigingshistorie v0.187 (14-05-2003):
 *  - altijd schieten, niet alleen indien een patroon-lock is gevonden
 *      --> verliest hierdoor van 0.183/0.184 obv bulletdamage, wint ruimtschoots door overleving
 *  - kogelkracht constant 2, zoekdiepte = 20 (wint daarmee ruimschoots van 0.184)
 *  - alleen zware kogels meenemen in uitwijkbeleid
 *
 * Wijzigingshistorie v0.186 (14-05-2003):
 *  - verloren code t.g.v. code-minimalisatie tot miniFenrir proberen te reproduceren:
 *    * kogel code toegevoegd (wijziging: alleen 'zware' kogels worden meegenomen in uitwijk-beweging.
 *    * beweging van Fenrir 0.184_GL
 *    * patroonherkenning van Fenrir 0.183_GL
 *    * alleen indien een patroon-lock is gevonden wordt er geschoten (zoekdiepte = 5)
 *      --> verliest hierdoor van 0.183/0.184 obv bulletdamage, wint ruimtschoots door overleving
 *
 * Wijzigingshistorie v0.185 (02-05-2003):
 *  - bug fixes.
 *
 * Wijzigingshistorie v0.184 (27-04-2003):
 *  - bug fixes.
 *
 * Wijzigingshistorie v0.183 (21-04-2003):
 *  - incl. patroon herkenning gebaseerd op methode nano.LauLectric en nano.Moebius,
 *    aangepast aan melee gevechten.
 *
 * Wijzigingshistorie v0.182 (17-04-2003):
 *  - bug fixes.
 *
 * Wijzigingshistorie v0.181 (15-04-2003):
 *  - incl. patroon herkenning gebaseerd op methode nano.LauLectric en nano.Moebius,
 *
 * Wijzigingshistorie v0.18 (10-04-2003):
 *  - incl. patroon herkenning gebaseerd op methode nano.LauLectric en nano.Moebius,
 *    aangepast aan melee gevechten.
 *
 * Wijzigingshistorie v0.x (17-02-2003): @#*# lost this version because my laptop was stolen..
 *  - incl. patroon herkenning
 *
 * Wijzigingshistorie v0.174 (25-02-2003):
 *  - increased standard walk distance: improves bullet dodging.
 *
 * Wijzigingshistorie v0.173 (25-02-2003):
 *  - increased standard walk distance: improves bullet dodging.
 *
 * Wijzigingshistorie v0.172 (25-02-2003):
 *  - verbeterde waarden parameters voor berekening vd. bewegings potentiaal.
 *
 * Wijzigingshistorie v0.17 (24-02-2003):
 *  - neemt laatste bewegingsrichting mee in bepalen nieuwe postie.
 *
 * Wijzigingshistorie v0.16 (23-02-2003):
 *  - inclusief circulaire doel pos. schatter.
 *
 * Wijzigingshistorie v0.15 (20-02-2003):
 *  - inclusief lineaire doel pos. schatter gebaseerd op gemiddelde snelheid van doel.
 *
 * Wijzigingshistorie v0.1 (17-02-2003):
 *  - gebaseerd op Balder v0.53
 *  - beweging gebaseerd op combinatie van potentiaal-methode en SandboxMini
 *
 */

public class Fenrir extends AdvancedRobot implements Constanten {
/**
* run: Fenrir's hoofdroutine:
*/
   public void run() {
      // initieer tank kleuren:
      setColors(Color.black, Color.black, Color.yellow);
/*
      // initieer GL4Java objecten:
      renderer = GLRenderer.getInstance();
      destPoint.setSize(6);
      destPoint.setColor(Color.green);
      renderer.addRenderElement(destPoint);
      for (int i=0; i < 8 ; i++) {
         potPoint[i].setColor(Color.red);
         renderer.addRenderElement(potPoint[i]);
      }
      kogelLine.setColor(Color.red);
      kogelHoek.setColor(Color.red);
      kogelGrens.setColor(Color.red);
      renderer.addRenderElement(kogelLine);
      renderer.addRenderElement(kogelHoek);
      renderer.addRenderElement(kogelGrens);
      for (int i=0; i < 6 ; i++) {
         kGrens[i].setColor(Color.yellow);
         renderer.addRenderElement(kGrens[i]);
         kLine[i].setColor(Color.yellow);
         renderer.addRenderElement(kLine[i]);
      }
*/

// initialisatie tank:
      setAdjustGunForRobotTurn(adjustGun);
      setAdjustRadarForGunTurn(adjustRadar);

// initieer slagveld (kan alleen in run-method):
      slagveld      = new Slagveld(this);

// bepaal type gevecht( o.b.v. het aantal tegenstanders):
      if ( getOthers() > 1 ) meleeModus = true;
      else                   meleeModus = false;

// start radar:
      setTurnRadarRightRadians( Double.POSITIVE_INFINITY );

// levensloop:
      while(true) {
         stuurRadar();  // stuur radar aan:
         navigatie();   // verplaats robot:
         stuurKannon(); // stuur kanon aan:
         execute();
      } // einde levensloop
   } // einde run-method

/**********************************************************************
** Methods voor event-afhandeling:
**********************************************************************/
/**
 * onScannedRobot: What to do when you see another robot
 * let op: van alle scanned bots tijdens een radar-sweep wordt de dichtstbijzijnde bot doorgegegeven!
 */
   public void onScannedRobot(ScannedRobotEvent evt) {
      Doel   opponent;
// doel nieuw opvoeren in lijst of opzoeken en bijwerken:
      opponent = memoreerDoel(evt);

// oportunistisch handelen: alleen doel selecteren als energie laag is en kans op vernietiging groot (Loki trekje :-o )
//                          of als doel dichterbij is
//                          of als ik geen doel in het leven heb:
      if (doel.locked     == false           // selecteer gunstig doel indien nog geen ander doel gevonden
      || evt.getName()    == doel.naam       // update doel indien al eerder gevonden
      || evt.getDistance() < 200D            // selecteer gunstiger doel
      || evt.getEnergy()   < 17D             // selecteer gunstiger doel
      || (doel.afstand     > evt.getDistance()
        && doel.e          > evt.getEnergy() )
         )
         doel = opponent;  // leg informatie over doel vast:
   }

/**
* onHitByBullet: What to do when you're hit by a bullet.
* This method will be called when your robot is hit by a bullet.
*/
   public void onHitByBullet(HitByBulletEvent evt) {
/*
      if (!meleeModus) {
         if (Math.random()>0.5)
            stop();
         else
         {  setMaxVelocity(8D);
            bewegingsrichtingOmkeren();
            setAhead( (-2D*voorwaards+1D) * getDistanceRemaining() );
         }
      }
*/
      Doel   opponent;
      opponent = memoreerDoel(evt.getName());
      opponent.setInfo(evt, this);  // leg informatie over doel vast:
      opponent.setTreffers(evt); // update treffers door tegenstander op Fenrir
   }

/**
* onBulletHit: What to do when your bullet hits a target.
* This method will be called when one of your bullets hits another robot.
*/
   public void onBulletHit(BulletHitEvent evt) {
      Doel   opponent;
      opponent = memoreerDoel(evt.getName());
      opponent.setInfo(evt, this);  // leg informatie over doel vast:
      opponent.setTreffersBvh(evt);  // update mijn treffers op doel

   }

/**
* onHitRobot: What to do when you're hit by a robot.
*/
   public void onHitRobot(HitRobotEvent evt) {
      Doel   opponent;
      opponent = memoreerDoel(evt.getName());
      opponent.setInfo(evt, this);  // leg informatie over doel vast:
   }

/**
* onRobotDeath: indien doel vernietigd dan dit 'vrijgeven'.
*/
   public void onRobotDeath(RobotDeathEvent evt) {
log("onRobotDeath: dood      ="+evt.getName());
log("onRobotDeath: doel.naam ="+doel.naam);
      // indien huidige doel is vernietigd, dan mijn bot initieren:
      if  (evt.getName().equals(doel.naam) ) {
         doel.locked = false;
log("onRobotDeath: doel.locked is false");
      }

      Doel   opponent;
      opponent = memoreerDoel(evt.getName());
      opponent.locked = false;

      stop();
   }

/**
* onWin: Niet stil blijven zitten om laatste op mij afgevuurde kogels te ontwijken.
* (zie UltimateDeathToNanoBots)
*/
   public void onWin(WinEvent e) {
      afronden();
      log("GEWONNEN: " + beurtOvergeslagen + " keren beurt overgeslagen.");
   }


/**
* onDeath: toon statistiek
*/
   public void onDeath(DeathEvent e) {
      afronden();
      log("....DOOD: " + beurtOvergeslagen + " keren beurt overgeslagen.");
   }

/**
* onSkippedTurn:  telt aantal keren dat beurt wordt overgeslagen;
*/
   public void onSkippedTurn(SkippedTurnEvent e) {
      beurtOvergeslagen += 1;
   }

/**********************************************************************
** Methods voor besturing tank
**********************************************************************/
/**
* navigatie(): Beweging o.b.v. potentiaalveld waarbij minder gunstige
* posities een hogere potentiaal krijgen en bot richting laagste potentiaal
* beweegd.
*/
   private void navigatie() {
      huidigePositie.setLocation( getX(), getY() );  // werken met oude, huidige en nieuwe positie !!!
      vGem= 0.857*vGem + 0.143*getVelocity();

      if ( Math.abs(getDistanceRemaining()) < 36D ) { // mogelijk verbetering door uitwijken te verwijderen!
         nieuwePositie = potentiaalMode(huidigePositie);

         if ( nieuwePositie.distance(huidigePositie) < 15 ) {
            nieuwePositie = SandboxMiniMode(huidigePositie, doel);
         }

         nieuweRichting   = Utils.normalRelativeAngle(HulpRoutines.bepaalRichting(nieuwePositie, huidigePositie) - getHeadingRadians());
         double draaiHoek = Math.atan(Math.tan(nieuweRichting));

         // draai en verplaats robot:
         setTurnRightRadians(draaiHoek);
         setAhead(huidigePositie.distance(nieuwePositie) * (nieuweRichting == draaiHoek ? 1 : -1));

//         destPoint.setPosition(nieuwePositie.x, nieuwePositie.y);
//         destLabel.setString((int)nieuwePositie.x + ", " + (int)nieuwePositie.y);

         oudePositie.setLocation(huidigePositie.x, huidigePositie.y);
      }

// eventueel snelheid minderen om sneller te kunnen draaien:
      if ( Math.abs(getTurnRemainingRadians()) > KWARTPI ) {
         setMaxVelocity( (HALFPI-Math.abs(getTurnRemainingRadians()))/7.5D );
      }
      else
         setMaxVelocity(MAXIMUM_SNELHEID);
   }

   private Point2D.Double SandboxMiniMode(Point2D.Double positie, Doel d) {
      double   richting;
      richting = -(d.afstand-300D)/200D*KWARTPI;
      richting = (HALFPI + richting);
      return( bepaalXY(positie, richting) );
   }

   private Point2D.Double potentiaalMode(Point2D.Double positie) {
      Doel     o;
      double   richting;
      double factor;
      double[] potentiaal  = {0D, 0D, 0D, 0D, 0D, 0D, 0D, 0D};
      int      kwadrant, t, l;
      kwadrant = l = 8;

      // betrek wanden in potentiaal:
      if ( positie.x < WANDAFSTAND ) {
         potentiaal[5] += 7;
         potentiaal[6] += 9;
         potentiaal[7] += 7;
      }
      if ( positie.x > (getBattleFieldWidth()  - WANDAFSTAND) ) {
         potentiaal[1] += 7;
         potentiaal[2] += 9;
         potentiaal[3] += 7;
      }
      if ( positie.y < WANDAFSTAND ) {
         potentiaal[3] += 7;
         potentiaal[4] += 9;
         potentiaal[5] += 5;
      }
      if ( positie.y > (getBattleFieldHeight()  - WANDAFSTAND) ) {
         potentiaal[0] += 7;
         potentiaal[1] += 9;
         potentiaal[7] += 7;
      }

      // bepaal potentiaal door te sommeren over tegenstanders:
      for (Enumeration e = doelen.elements(); e.hasMoreElements(); ) {
         o               = (Doel)e.nextElement();
         if (o.locked) {
            factor          = Math.max(Math.min(Math.round(600D/o.afstand), 2D), 0.5);
            if (!meleeModus)
               factor         *= Math.max(Math.min(Math.round(o.e/50D), 2D), 0.5);
            kwadrant        = (int)(Math.round(l+l*o.richting/TWEEPI))%l;
            if (meleeModus) {
               potentiaal[ kwadrant]            += factor*3.0;
               potentiaal[(kwadrant+ l/2)%l]    += factor*2.0;
            }
            else {
               potentiaal[ kwadrant]            += factor*(2D+random.nextInt(3));    // niet naar tegenstander toe bewegen
               potentiaal[(kwadrant+ l/2)%l]    += factor*(1D+random.nextInt(3));         // niet van tegenstander af bewegen
            }
            potentiaal[(kwadrant + 1)%l]     += factor*1.0;                       // en kans op beweging schuin vanaf/naar tegenstander ook verkleinen
            potentiaal[(kwadrant - 1 + l)%l] += factor*1.0;
            potentiaal[(kwadrant + 3)%l]     += factor*0.5;
            potentiaal[(kwadrant - 3 + l)%l] += factor*0.5;
         } // controle op ge'locked' zijn van tegenstander
      }

// uitbreiden potentiaal door laatste eigen bewegingsrichting te betrekken (bewegingsrichting
// omkeren wordt ontmoedigd)
// proefondervindelijk aangetoond dat dit verbetering geeft in 1-vs-1,
// verslechtering bij melee-gevechten:

      if (getOthers() == 1) {
         // afstoting van vorige positie:
         kwadrant        = (int)(Math.round(l+l*HulpRoutines.bepaalRichting(oudePositie, positie)/TWEEPI))%l;
         potentiaal[kwadrant]  += 0.5; // bewegingsrichting niet omkeren

         // aantrekking naar middelpunt slaveld:
         kwadrant        = (int)(Math.round(l+l*HulpRoutines.bepaalRichting(slagveld.middelpunt, positie)/TWEEPI))%l;
         potentiaal[kwadrant]  -= 0.5;   // naar middelpunt slagveld
      }


// uitbreiden potentiaal door nabije kogels te betrekken;
// indien kogel is gepasseerd, dan kogel verwijderen uit tabel:
      Kogel            kogel;
      double           kogelStraal, testAfstand;
      long             nu = getTime();

      for (Enumeration k = kogels.elements(); k.hasMoreElements(); ) {
// bepaal positie kogel t.o.v. eigen bot
         kogel           = (Kogel)k.nextElement();
         kogelStraal     = kogel.getAfgelegdeWeg(nu);
         testAfstand     = positie.distance( kogel.getScanPositie() ) - kogelStraal;

// nabije kogels met energie >= 1 meenemen in uitwijk-beweging:
         if ( (testAfstand <= 76D && testAfstand >= 0D) && kogel.v <= 17D ) {
            // kwadrant bepalen door huidige kogelpositie te schatten obv de
            // aanname dat deze recht op mijn bot is afgeschoten
            kwadrant        = (int)(Math.round(l+l*HulpRoutines.bepaalRichting( kogel.getPositie(nu), positie )/TWEEPI))%l;
/*
            if (meleeModus) {
               potentiaal[kwadrant]             += 1.00;
               potentiaal[(kwadrant+ l/2)%l]    += 0.50;
               potentiaal[(kwadrant + 1)%l]     += 0.25;
               potentiaal[(kwadrant - 1 + l)%l] += 0.25;
            }
            else {
               potentiaal[kwadrant]             += 1.00;
               potentiaal[(kwadrant + 1)%l]     += 0.50;
               potentiaal[(kwadrant - 1 + l)%l] += 0.50;
               potentiaal[(kwadrant + 2)%l]     += 0.25;
               potentiaal[(kwadrant - 2 + l)%l] += 0.25;
            }
*/
            if ( kogel.v <= 14D ) {
               uitwijken  = true;
            }
         }

         if ( testAfstand < 0D ) {
            kogels.remove(kogel.naam);
         }
      }

      // bepaal gunstigste richting:
      double minimum = 1D/0D;
      for (int i=0; i < l ; i++) {

//         potPoint[i].setPosition( positie.x+36*Math.sin(i*KWARTPI), positie.y+36*Math.cos(i*KWARTPI) );
//         potPoint[i].setSize( Math.round(potentiaal[i]+0.5) );

         if (    potentiaal[i] < minimum
            || ( potentiaal[i]== minimum && Math.random() > 0.25 )
            ) {
            minimum  = potentiaal[kwadrant = i];
         }
      }

      richting = kwadrant*KWARTPI+ random.nextGaussian()*EENACHTSTEPI;
      return( bepaalXY(positie, richting) );
   }

   private Point2D.Double bepaalXY(Point2D.Double positie, double richting) {
      double r;
      // inbouwen willekeur in bewegingsafstand en tijdstip waarop volgende nieuwe
      // positie wordt bepaald.
      r = 1D + 3D*random.nextGaussian();

      // bepaal nieuwe x,y positie o.b.v. huidige positie, nieuwe hoek t.o.v. doel
      // en loopafstand die afhankelijk is van huidige afstand tot doel en hierboven
      // bepaalde random factor.
      nieuwePositie.x = getX() + r * STANDAARDLOOPAFSTAND * Math.sin(richting);
      nieuwePositie.y = getY() + r * STANDAARDLOOPAFSTAND * Math.cos(richting);
      // ontwijk wanden door min./max. waarden te bepalen voor nieuwe x,y positie.
      // overleving verbeterd door positie te spiegelen t.o.v. minimum wandafstand!
      nieuwePositie.x = Math.min(getBattleFieldWidth()  - WANDAFSTAND, Math.max(WANDAFSTAND, nieuwePositie.x));// * 2D - nieuwePositie.x;
      nieuwePositie.y = Math.min(getBattleFieldHeight() - WANDAFSTAND, Math.max(WANDAFSTAND, nieuwePositie.y));// * 2D - nieuwePositie.y;

      return( nieuwePositie );
   }

   private void bewegingsrichtingOmkeren() {
      if      (nieuweRichting < 0 ) nieuweRichting += PI;  // 180 graden bij nieuwe richting optellen
      else if (nieuweRichting > 0 ) nieuweRichting -= PI;  // 180 graden bij nieuwe richting aftrekken

      if (voorwaards == 1) voorwaards = 0;  // en bewegingsrichting omkeren
      else                 voorwaards = 1;
   }

/**********************************************************************
** Methods voor besturing kanon
**********************************************************************/
/**
* Vuurkracht afhankelijk van de doelafstand en de (voorspelde) hoek tussen kanon en doel
*
* v0.2 (28-08-2002)
*/
   private void stuurKannon() {

      double kogelSnelheid, kanonVuurkracht, doelHoek, maxRichting;

      if ( doel.locked ) { // alleen kanon sturen als er op iets te schieten valt

         if ( doel.e <= 16D ) {
// energie tegenstander zodanig dat deze met n schot kan worden afgemaakt.
// bepaal hiervoor benodigde minimum vuurkracht:
            kanonVuurkracht = berekenMinimumVuurkracht( doel.e );
         }
         else 
            kanonVuurkracht = (doel.afstand < 120D) ? 3.0D : 2.0D;

         // TC-settings: kanonVuurkracht = 3D;
         // PMC-settings: kanonVuurkracht = 0.5D;
         //kanonVuurkracht = 0.5D;
         kogelSnelheid   = berekenKogelSnelheid( kanonVuurkracht );

         maxRichting   = Math.asin(MAXIMUM_SNELHEID/kogelSnelheid);

         if (getGunHeat() > getGunCoolingRate()) {
            kanonWordtGericht = false;
            doelHoek = doel.richting;
         }
         else {
            if (kanonWordtGericht && getGunTurnRemaining() == 0 && getEnergy() > kanonVuurkracht) {
               kanonWordtGericht = false;
               setFire( kanonVuurkracht );
               doel.setSchotenBvh();
            }
            else {
               kanonWordtGericht = true;
               doelHoek = doel.getVoorspeldeDoelhoek(kogelSnelheid, this);
               doelHoek = HulpRoutines.normaliseerRichting( doelHoek - getGunHeadingRadians() );

               if ( doelHoek<0 )
                  doelHoek = Math.max(doelHoek,-maxRichting);
               else
                  doelHoek = Math.min(doelHoek, maxRichting);

               setTurnGunRightRadians( doelHoek );
            }
         }
            
      } // einde controle op aanwezigheid doel

   } // einde method stuurKannon()

/**
* berekenKogelSnelheid: berekend de snelheid van de kogel op basis
*    van de kracht van de kogel.
*
* v0.1 (27-12-2002)
*/
   private double berekenKogelSnelheid(double kanonVuurkracht) {
      return(20 - 3 * kanonVuurkracht);
   }
/**
* berekenMinimumKogelVuurkracht: berekend de benodigde vuurkracht van de kogel
*    om, op  basis van de energie van het doel, het doel met n kogel
*    uit te schakelen.
*
* v0.1 (27-12-2002)
*/
   private double berekenMinimumVuurkracht(double energie) {
      double vuurkracht;
      if (energie <= 12D ) vuurkracht = energie/4D;
      else                 vuurkracht = (energie+2D)/6D;

      vuurkracht    = vuurkracht + 0.1; // en beetje meer energie om doel te vernietigen...

      return( vuurkracht );
   }

/**
* berekenVuurkracht: berekend de vuurkracht van de kogel obv gegeven snelheid.
*
* v0.1 (22-10-2003)
*/
   private double berekenVuurkracht(double kogelSnelheid) {
      return( (20D - kogelSnelheid)/3D );
   }

/**
* berekenSnelheidBijConstanteOverbruggingstijd: berekend de sneleid van de kogel
*    obv gegeven afstand tot doel, zodanig dat de overbruggingstijd vrijwel
*    constant blijft voor iedere afgevuurde kogel. Het idee is dat dit resulteerd
*    in een nauwkeuriger patroonherkenning van de beweging van de tegenstander.
*
* v0.1 (22-10-2003)
*/
   private double berekenSnelheidBijConstanteOverbruggingstijd( double afstand ) {
      return( Math.min( Math.max( (afstand / 29.5D), maxKogelsnelheid), minKogelsnelheid ) );
   }
/**********************************************************************
** Methods voor besturing radar
**********************************************************************/
/**
* stuurRadar(): Radarsturing in melee gevechten waarbij zo groot mogelijk gebied wordt bestreken:
*
* v0.4 (10-04-2003): 1-vs-1 mode toegevoegd zodat een constanter scan-tussentijd wordt bereikt.
* v0.3 (25-12-2002): radar draait door tot alle bots in melee zijn gescanned waarna de
*                    draairichting omkeerd. Op deze manier hoeft niet perse 360 graden
*                    te worden gescanned en worden toch alle bots bijgehouden.
* v0.2 (09-11-2002): bestreken gebied wordt gebaseerd op de max. en min. hoek waarbinnen zich
*                    alle tegenstanders bevinden.
* v0.1 (10-09-2002): volledige 360 graden wordt bestreken.
*/
   private void stuurRadar() {
      if (getOthers() == 1 && doel.locked) {
         setTurnRadarRightRadians(3D*HulpRoutines.normaliseerRichting( doel.richting - getRadarHeadingRadians() ));
      }
      else
      // else (melee-mode) laat de radar draaien in dezelfde richting om een
      // gelijkmatige sample-rate te verkrijgen voor de patroon herkenning.
         setTurnRadarRightRadians( Double.POSITIVE_INFINITY );
   }

/**********************************************************************
** Methods voor doel beheersing
**********************************************************************/
/**
* methoden om doelen te bewerken:
*/
   private Doel memoreerDoel(ScannedRobotEvent evt) {
      String naam = evt.getName();
      Doel   opponent;

      if( doelen.containsKey(naam) )
         opponent = (Doel)doelen.get(naam);
      else {
         opponent = new Doel();
         doelen.put(naam, opponent);
         inlezenDoelData(naam, opponent);
      }
      opponent.setInfo(evt, this);  // leg informatie over doel vast:

      return opponent;
   }

   private Doel memoreerDoel(String naam) {
      Doel opponent;
// doel nieuw opvoeren in lijst of opzoeken en bijwerken:
      if( doelen.containsKey(naam) ) {
         opponent = (Doel)doelen.get(naam);
      } else {
         opponent = new Doel();
         doelen.put(naam, opponent);
         inlezenDoelData(naam, opponent);
      }
      return opponent;
   }

   private void verwijderDoel(String naam) {
      doelen.remove(naam);
   }

/**
* geefDichtstbijzijndeKogel(String naam): haalt de meest nabije kogel uit
*    de lijst met kogels.
*
* v0.2 (28-01-2004)
*  - verbetering bepaling afstand tot eigen bot door kogelstraal te nemen en
*    geschatte kogelrichting te negeren.
*
* v0.1 (23-06-2003)
*/
   private Kogel getDichtstbijzijndeKogel()
   {
      long       nu = getTime();
      double      a,
                  kortsteAfstand = 1D/0D;
      String      n = null;
      Kogel       k;

      for (Enumeration ko = kogels.elements(); ko.hasMoreElements(); )
      {
         k = (Kogel)ko.nextElement();
         a = Math.abs( huidigePositie.distance(k.getScanPositie()) - k.getAfgelegdeWeg(nu) );
         if (a < kortsteAfstand)
         {
            kortsteAfstand = a;
            n              = k.naam;
         }
      }

      if (n != null) k = (Kogel)kogels.get(n);
      else           k = null;

      return (k);
   }

/**********************************************************************
** methods voor afronden gevecht:
**********************************************************************/
   private void afronden()
   {
      saveSatistiek(); // per ronde statistiek tonen en naar file schrijven voor studie

      // na laatste ronde aan einde van gevecht de opgebouwde data-arrays opslaan naar file
      if (getRoundNum() == getNumRounds()-1)
      {
         Doel o;
         for (Enumeration en = doelen.elements(); en.hasMoreElements(); )
         {
            o = (Doel)en.nextElement();
            opslaanDoelData( o );
         }
      }
   }

/**********************************************************************
** methods voor weergeven/opslaan statistiek m.b.t. hits/missers/matches
** en gemiddelde afstand voor elk doel.
**********************************************************************/
   private void saveSatistiek(){
      Doel d;
      for (Enumeration en = doelen.elements(); en.hasMoreElements(); ) {
         d = (Doel)en.nextElement();
         try {
            log(">>**************************************************<<");
            log("Statistiek bot "+ d.naam);
            log(">>**************************************************<<");
            log("I fire bullet amount       : "+ d.getSchotenBvh() );
            log("My hit bullet amount       : "+ d.getTreffersBvh() );
            log("                 Hit ratio : "+100.0d*d.getTreffersBvh()/d.getSchotenBvh()+"%");
            log("+-------------------------------------+");
            log("Enemy fire bullet amount   : "+ d.getSchoten() );
            log("Enemy  hit bullet amount   : "+ d.getTreffers() );
            log("                 Hit ratio : "+100.0d*d.getTreffers()/d.getSchoten()+"%");
            log("+-------------------------------------+");
            log("Pogingen patroonherkenning : "+d.probeerTeller);
            log("Succes patroonherkenning   : "+d.succesTeller);
            log("                     Ratio : "+100.0d*d.succesTeller/d.probeerTeller+"%");
            log("Gemiddelde patroonlengte   : "+d.gemiddeldePatroonLengte/(d.succesTeller==0 ? 9999999 : d.succesTeller));
            log("+-------------------------------------+");
            log("Gemiddelde afstand         : "+d.afstandGem);

            PrintStream w = new PrintStream(new RobocodeFileOutputStream(getDataFile(d.naam+".txt")));
            w.println(">>**************************************************<<");
            w.println("Statistiek bot "+ d.naam);
            w.println(">>**************************************************<<");
            w.println("I fire bullet amount       : "+ d.getSchotenBvh() );
            w.println("My hit bullet amount       : "+ d.getTreffersBvh() );
            w.println("                 Hit ratio : "+100.0d*d.getTreffersBvh()/d.getSchotenBvh()+"%");
            w.println("+-------------------------------------+");
            w.println("Enemy fire bullet amount   : "+ d.getSchoten() );
            w.println("Enemy  hit bullet amount   : "+ d.getTreffers() );
            w.println("                 Hit ratio : "+100.0d*d.getTreffers()/d.getSchoten()+"%");
            w.println("+-------------------------------------+");
            w.println("Pogingen patroonherkenning : "+d.probeerTeller);
            w.println("Succes patroonherkenning   : "+d.succesTeller);
            w.println("                     Ratio : "+100.0d*d.succesTeller/d.probeerTeller+"%");
            w.println("Gemiddelde patroonlengte   : "+d.gemiddeldePatroonLengte/(d.succesTeller==0 ? 9999999 : d.succesTeller));
            w.println("+-------------------------------------+");
            w.println("Gemiddelde afstand         : "+d.afstandGem);

            if (w.checkError()) {
               log("Probleem bij wegschrijven statistiek!");
            }
            w.close();
            log(d.naam+"'s data opgeslagen");
         } catch (Exception e) {
            log("saveSatistiek(): IOException bij opslaan data " + d.naam + " : " + e);
         }
      }
   }

/**********************************************************************
** methods voor lezen/schrijven data:
**********************************************************************/
   private void inlezenDoelData(String naam, Doel d) {
      d.rotatie  = new double[PATROONLENGTE];
      d.snelheid = new double[PATROONLENGTE];

      try {
         String   filenaam;
         double   temp1[]  = new double[FILE_LENGTE];
         double   temp2[]  = new double[FILE_LENGTE];

         filenaam = HulpRoutines.getFilenaam(naam);
         ObjectInputStream in = new ObjectInputStream(
                                   new GZIPInputStream(
                                      new FileInputStream(
                                         getDataFile( filenaam ))));

log("readInfo() open file: " + filenaam);

         temp1 = (double[])in.readObject();
         temp2 = (double[])in.readObject();

// is er een slimmere manier om array te vullen en originele lengte te behouden???
         for (int i=0; i < temp1.length ; i++) {
            d.rotatie[i]  = temp1[i];
            d.snelheid[i] = temp2[i];
         }
log("readInfo() snelheid ingelezen");

         d.bewegingsPatroon = new StringBuffer( (String)in.readObject() );
         d.scanTeller = in.readInt();
         in.close();

log("readInfo() voor doel:"+naam+", lengte rotatie ="+d.rotatie.length);
log("readInfo() voor doel:"+naam+", lengte snelheid="+d.snelheid.length);
log("readInfo(): bewegingsPatroon ingelezen, lengte =" + d.bewegingsPatroon.length());
log("readInfo() scanTeller ingelezen: scanTeller="+d.scanTeller);
log("readInfo() file " + filenaam + " gesloten");

// stopper inbouwen om grens oude/nieuwe data aan te geven.
         ++d.scanTeller;
         d.rotatie[d.scanTeller%PATROONLENGTE]  = 0;
         d.snelheid[d.scanTeller%PATROONLENGTE] = 0;
         d.bewegingsPatroon.append((char)-99);
      }
      catch (Exception e) {
log("exception inlezenDoelData() voor doel:"+naam+", e="+e);
         d.bewegingsPatroon = new StringBuffer();
         d.scanTeller = 0;
      }
   }


   private void opslaanDoelData(Doel d) {
      boolean dataruimte_beschikbaar = true;
      File    directory       = null;
      File[]  robotFiles      = null;
      long    totalBytes      = 0;

      /* the code to check if there is enough room to save my data is based on
       * code from tzu.TheArtOfWar, by Ray Vermette, including the comments.
       *
       * If the data directory does not exist, yet, then
       * it can't possibly be full.
       */
      directory  = this.getDataDirectory();
      if (directory != null) {
          /*
           * If there are no files in the directory, then
           * we certainly have room.
           */
          robotFiles = directory.listFiles();
          if (robotFiles != null) {
             for (int i = 0; i < robotFiles.length; i++) {
                totalBytes += robotFiles[i].length();
             }
             if (totalBytes + (12500) <= MAX_DATA_DIRECTORY)
                dataruimte_beschikbaar = true;
             else
                dataruimte_beschikbaar = false;
          } // einde controle op aantal datafiles
      } // einde controle of directory bestaat

      if (dataruimte_beschikbaar) {
// wegschrijven data (pm 12.5kb per bot)
      try {
         String   filenaam;
         int      startpos;
         int      lengte;

         lengte   = Math.min(FILE_LENGTE, d.scanTeller);
         double   uitvoer[] = new double[lengte];

         filenaam = HulpRoutines.getFilenaam(d.naam);
         ObjectOutputStream out = new ObjectOutputStream(
                                     new GZIPOutputStream(
                                        new RobocodeFileOutputStream(
                                           getDataFile( filenaam ))));
log("writeInfo() voor doel:"+d.naam+", file lengte   ="+lengte);
         startpos = Math.min(PATROONLENGTE, d.scanTeller)-lengte;
         System.arraycopy(d.rotatie, startpos, uitvoer, 0, lengte);
         out.writeObject(uitvoer);
log("writeInfo() rotatie  array met lengte ="+uitvoer.length + " weggeschreven");
         System.arraycopy(d.snelheid, startpos, uitvoer, 0, lengte);
         out.writeObject(uitvoer);
log("writeInfo() snelheid array met lengte ="+uitvoer.length + " weggeschreven");

         out.writeObject((String)d.bewegingsPatroon.substring(startpos, startpos+lengte)); // neemt string vanaf opgegeven positie
log("writeInfo() bewegingsPatroon met lengte ="+ lengte + " weggeschreven");
         out.writeInt(lengte);
log("writeInfo() lengte weggeschreven");
         out.flush();
         out.close();
log("writeInfo() file gesloten");

log("writeInfo() voor doel:"+d.naam+", scanTeller="+d.scanTeller);
log("writeInfo() voor doel:"+d.naam+", lengte rotatie="+d.rotatie.length);
log("writeInfo() voor doel:"+d.naam+", lengte bewegingsPatroon="+d.bewegingsPatroon.length());

      }
      catch (IOException e) {
         out.close();
log("exception opslaanDoelData() voor doel:"+d.naam+", e="+e);
      }
      } // einde controle of voldoende ruimte aanwezig is.
   }

// log method: voor debugging ...
   public void log(String tekst) {
      if (DEBUGMODE > 0) out.println(tekst);
   }
/**********************************************************************
** Constructor:
**********************************************************************/
   public Fenrir() {
      oudePositie   = new Point2D.Double(0,0);
      huidigePositie= new Point2D.Double(0,0);
      nieuwePositie = new Point2D.Double(0,0);
      nieuweRichting= 0;
      voorwaards    = 1;

      minKogelsnelheid = 20D - 3D * MAXVUURKRACHT;
      maxKogelsnelheid = 20D - 3D * MINVUURKRACHT;

// initieer doel & kogel hashtabel:
      doel          = new Doel();
      kogels        = new Hashtable();

   }
/**********************************************************************
** Definitie van variabelen
**********************************************************************/
// overige objecten:
   private boolean adjustGun  = true; // kanon draait in tegengestelde richting als bot: blijft daardoor oorspronkelijke richting behouden
   private boolean adjustRadar= true; // radar draait in tegengestelde richting als bot: blijft daardoor oorspronkelijke richting behouden
   private int     beurtOvergeslagen;

// variabelen voor sturing van beweging en radar:
   private Slagveld  slagveld;

// variabelen voor beweging:
   private int              voorwaards;
   private double           nieuweRichting;
   public  Point2D.Double   nieuwePositie, huidigePositie, oudePositie;
   private boolean          uitwijken;
   public  static double    vGem;
           Random           random = new Random();

// variabelen voor kanon:
   private double  minKogelsnelheid;
   private double  maxKogelsnelheid;
   private boolean kanonWordtGericht;

// variabelen voor uitwijkingsbeleid:
   public  static  Hashtable doelen = new Hashtable(); // hier initieren omdat in constructor hashtable iedere keer opnieuw wordt geinitieerd en opgedane data verloren gaat.

// variabelen voor overleving ;-)
   public          Hashtable kogels;

// variabelen die huidig doel of directe bedreiging bevatten:
   public  boolean   meleeModus;
   private Doel      doel;


/*
// GL4Java variabelen...
   GLRenderer        renderer;
   PointGL           destPoint = new PointGL();
   LabelGL           destLabel = new LabelGL("");
   PointGL[]         potPoint  = {new PointGL(), new PointGL(), new PointGL(), new PointGL()
                                 ,new PointGL(), new PointGL(), new PointGL(), new PointGL()};
   LineGL            kogelLine = new LineGL(),
                     kogelHoek = new LineGL();
   EllipseGL         kogelGrens= new EllipseGL();
   EllipseGL[]       kGrens    = {new EllipseGL(), new EllipseGL(), new EllipseGL()
                                 ,new EllipseGL(), new EllipseGL(), new EllipseGL()};
   LineGL[]          kLine     = {new LineGL(), new LineGL(), new LineGL()
                                 ,new LineGL(), new LineGL(), new LineGL()};
*/
}
/**********************************************************************
** EINDE CLASS Fenrir
**********************************************************************/