package gh;
import robocode.*;
import robocode.util.Utils;
import java.awt.Color;
import java.awt.geom.*;
import java.util.*;

/**
 * Griezel - the first specific meleebot of Gert Heijenk (GrubbmGait)
 *
 * Inspired by the ideas on the MeleeStrategy page and on Coriantumr
 * Created because Gruwel was threatened by
 * Loki's (mini)Freya
 * kc's   Logic
 * MAP's  Ugluk
 * deewiant's Anomaly
 *
 * Revision information:
 * v0.1   20051211 First version, Gruwel with a purely distance-driven minimumrisk movement
 * v0.2   20051221 Immediately select new target if current dies
 *                 Slowdown when turning
 *                 Fire only when aligned
 *                 Even if not firing (energymanagement), turn radar every 20 ticks (riskmanagement)
 * v0.3   20060109 Tuning riskfunction to match my own ideas better
 *                 Small bugfix radar when all enemies out of reach
 *                 Now ready for TeamRumble
 * v0.4   20060526 Slightly improved gun when enemy is near wall
 *                 only change destination when new one is significantly better
 * v0.5   20090929 Take damage given by enemy and damage taken by enemy into account in dangercalculation
 * v0.5.1 20090930 Don't be so affraid of the fieldcenter
 * v0.5.2 20091002 Don't be affraid at all for fieldcenter and corners
 */
public class Griezel extends TeamRobot
{
	// movement variables
    static final double WALL_MARGIN	= 17;
	static final double WALLAVOID	= 35;	// the area to avoid near walls
	static final long RADARTIMEOUT	= 20;	// turn radar at least once in this time
	private Point2D	myPosition;				// my current position
	private	Point2D	myLastPosition;			// my previous position when changing Goal position
	private	Point2D	myGoalPosition;			// my desired position, continuously changing
	private Point2D	centerPoint;			// the centre of the battlefield
	private long	currTime;				// the current time (mainloop)

	// scanning and locking variables
	private boolean	radarScan;				// if true, busy with a radar scan
	private long	lastscan;				// time of last radarscan

	// some info about the targets
	private double	oldHeading;				// the previous heading of the selected target
	private EnemyInfo currentTarget;		// the current target
	private HashMap	enemies;				// all available targets
	private static HashMap enemyStat;				// all enemies during battle

	// some statistics
	private double	fieldWidth;		// the width of the battlefield
	private double	fieldHeight;	// the heigth of the battlefield
    private RoundRectangle2D.Double playField; // the playable battlefield
	private Rectangle2D.Double fireField;	// the shootable battlefield

	//global declarations:
	static int[] finishes;

	/**
	 * run: Griezel's default behavior
	 */
	public void run() {
		// Give the robot an appealing look
		setColors( Color.red.brighter(), Color.orange, Color.yellow);

		// First set the statistics
		fieldWidth = getBattleFieldWidth();
		fieldHeight = getBattleFieldHeight();
        playField = new RoundRectangle2D.Double( WALLAVOID, WALLAVOID,
	    fieldWidth - 2 * WALLAVOID, fieldHeight - 2 * WALLAVOID, 50, 50);
        fireField = new Rectangle2D.Double( WALL_MARGIN, WALL_MARGIN, fieldWidth - 2 * WALL_MARGIN, fieldHeight - 2 * WALL_MARGIN);
		centerPoint = new Point2D.Double( fieldWidth / 2, fieldHeight / 2 );

		if (finishes == null)
			finishes = new int[getOthers()+1];

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
		setAdjustRadarForGunTurn( true);
		setTurnRadarRight( Double.POSITIVE_INFINITY);

		// Do not keep anything between rounds
		enemies = new HashMap();
		myPosition = new Point2D.Double();
		myLastPosition = new Point2D.Double();
		myGoalPosition = new Point2D.Double( getX(), getY());

		// Do keep enemystats during whole battle
		if (enemyStat == null)
			enemyStat = new HashMap();

		// mainloop
		while( true ) {
			currTime = getTime();
			// radar plus gunnery is done in onScannedRobot()
			// movement is done in a routine to keep the mainloop clean
			doMeleeMovement();
			if (getRadarTurnRemaining() == 0.0) {
				setTurnRadarRight( Double.POSITIVE_INFINITY);
				radarScan = true;
				out.println("Enemy out of reach, start radar again");
			}
			execute();
		}
	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {

		double absBearing;
		double dist;
		double power;
		boolean fired = false;
		EnemyInfo enemy = (EnemyInfo)enemies.get( e.getName());
		if (enemy == null)
			enemies.put( e.getName(), enemy = new EnemyInfo());

		EnemyStats enemyS = (EnemyStats)enemyStat.get( e.getName());
		if (enemyS == null) {
			enemyStat.put( e.getName(), enemyS = new EnemyStats());
			enemyS.name = e.getName();
		}

		myPosition.setLocation( getX(), getY());
		// fill in the data of this enemy
		absBearing = getHeadingRadians() + e.getBearingRadians();
		dist = e.getDistance();
		enemy.setLocation( myPosition.getX() + Math.sin( absBearing) * dist, myPosition.getY() + Math.cos( absBearing) * dist);
		enemy.energy   = e.getEnergy();
		enemy.heading  = e.getHeadingRadians();
		enemy.velocity = e.getVelocity();
		enemy.teammate = isTeammate( e.getName());

		// calculate 'attractivity' of this enemy and decide whether to attack or not
		if (enemy.teammate == true) return;
		enemy.attrac = (e.getEnergy() < 20 ? dist * 0.75 : dist);
		if (enemy != currentTarget) {
			if ((currentTarget == null) || ((enemy.attrac < (currentTarget.attrac * 0.8)) && (currentTarget.attrac > 80) && (dist < 1200))) {
				currentTarget = enemy;	// new more attractive current target
			}
		}
		if (enemy != currentTarget) return;		// forget about those distant loosers

		// At start of round, always perform 2.5 radarsweeps without locking, but do turn the gun already!!
		if (currTime < 10) {
			setTurnGunRightRadians( Utils.normalRelativeAngle( absBearing - getRadarHeadingRadians()));
			return;
		}

		// set the power of the bullet, including energymanagement
		if (getOthers() > 4)
			power = ( dist > 850 ? 0.1 : (dist > 700 ? 0.49 : (dist > 400 ? 1.9 : (dist > 240 ? 2.49 : 3.0))));
		else
			power = ( dist > 850 ? 0.1 : (dist > 700 ? 0.49 : (dist > 250 ? 1.9 : 3.0)));
		power = Math.min( getEnergy()/5, Math.min( (e.getEnergy()/4) + 0.2, power));

		// perform lineair and circular targeting
		long deltahittime;
		Point2D.Double point = new Point2D.Double();
		double head, chead, bspeed;
		double tmpx, tmpy;

		// perform an iteration to find a reasonable accurate expected position
		tmpx = enemy.getX();
		tmpy = enemy.getY();
		head = e.getHeadingRadians();
		chead = head - oldHeading;
		oldHeading = head;
		point.setLocation( tmpx, tmpy);
		deltahittime = 0;
		do {
			tmpx += Math.sin( head) * e.getVelocity();
			tmpy +=	Math.cos( head) * e.getVelocity();
			head += chead;
			deltahittime++;
			// if position not in field, adapt
			if (!fireField.contains( tmpx, tmpy)) {
				point.setLocation( warpPoint( tmpx, tmpy));
				bspeed = point.distance( myPosition) / deltahittime;
				power = Math.max( Math.min( (20 - bspeed) / 3.0, 3.0), 0.1);
				break;
			}
			point.setLocation( tmpx, tmpy);
		} while ( (int)Math.round( (point.distance( myPosition) - 18) / Rules.getBulletSpeed( power)) > deltahittime);
		point.setLocation( warpPoint( tmpx, tmpy));

		// If the gun is too hot or not aligned, don't bother trying to fire.
		if ((getGunHeat() == 0.0) && (getGunTurnRemaining() == 0.0) && (power > 0.0) && (getEnergy() > 0.1) && (radarScan == false)) {
			// Only fire the gun when it is locked and loaded, do a radarsweep afterwards.
			setFire( power);
			fired = true;
		}
		if ((fired == true) || (currTime >= (lastscan + RADARTIMEOUT))) {
			setTurnRadarRightRadians( Double.NEGATIVE_INFINITY * Utils.normalRelativeAngle( absBearing - getRadarHeadingRadians()));
			lastscan = currTime;
			radarScan = true;
		}
		else {
			// when not firing, lock on target.
			setTurnRadarRightRadians( 2.2 * Utils.normalRelativeAngle( absBearing - getRadarHeadingRadians()));
			radarScan = false;
		}
		// Turn gun after eventual firing (robocode does it also this way)
		setTurnGunRightRadians( Utils.normalRelativeAngle(((Math.PI / 2) - Math.atan2( point.y - myPosition.getY(), point.x - myPosition.getX())) - getGunHeadingRadians()));
	}

	/**
	 * warpPoint: Put point into field with a margin of 2 * WALL_MARGIN (34)
	 */
	public Point2D.Double warpPoint( double x, double y)
	{
		Point2D.Double pos = new Point2D.Double();

		pos.x = Math.min( fieldWidth  - 2 * WALL_MARGIN, Math.max( 2 * WALL_MARGIN, x));
		pos.y = Math.min( fieldHeight - 2 * WALL_MARGIN, Math.max( 2 * WALL_MARGIN, y));
		return pos;
	}

	public void onHitByBullet(HitByBulletEvent e)
	{
		EnemyInfo enemy = (EnemyInfo)enemies.get(e.getName());
		if (enemy != null) {
			enemy.lastHit = getTime();
		}
		EnemyStats enemyS = (EnemyStats)enemyStat.get(e.getName());
		if (enemyS != null) {
		    enemyS.damage_given += (4 * e.getBullet().getPower() + 2 * Math.max( e.getBullet().getPower() - 1, 0));
		}
	}

	public void onBulletHit(BulletHitEvent e)
	{
		EnemyStats enemyS = (EnemyStats)enemyStat.get(e.getName());
		if (enemyS != null)
		    enemyS.damage_taken += (4 * e.getBullet().getPower() + 2 * Math.max( e.getBullet().getPower() - 1, 0));
	}

	public void onRobotDeath(RobotDeathEvent e)
	{
		// remove dead enemy and check if it was my currentTarget
		if (enemies.remove(e.getName()) == currentTarget) {
			currentTarget = findNewTarget();
			setTurnRadarRight( Double.POSITIVE_INFINITY);
			radarScan = true;
		}
	}

	public void doMeleeMovement()
	{
		double travel;
		double goalrisk;
		double testrisk;
		double temprisk = 100;
		double angle;
		Point2D testPoint = new Point2D.Double();
		Point2D tempPoint = new Point2D.Double();

		myPosition.setLocation( getX(), getY());
		if (currentTarget != null)
		{
			// if enemy is close, keep moving
			travel = Math.max( 60, myPosition.distance(currentTarget) * 0.35);
			goalrisk = calcRisk( myGoalPosition, getHeadingRadians());			// Is my Goal-path still good enough??

			for (angle = 0; angle < Math.PI*2; angle += Math.PI/36) {
				testPoint.setLocation( myPosition.getX() + Math.sin( angle) * travel, myPosition.getY() + Math.cos( angle) * travel);
				if ((playField.contains( testPoint)) && ((testrisk = calcRisk( testPoint, angle)) < temprisk )) {
					tempPoint.setLocation( testPoint);
					temprisk = testrisk;
				}
			}
			// only if best point is significantly better than old goalposition, change
			if (temprisk < (goalrisk * 0.9)) {
				myGoalPosition.setLocation( tempPoint);
				goalrisk = temprisk;
			}
			// set the turn and the distance
			angle = Utils.normalRelativeAngle( Math.atan2( myPosition.getX() - myGoalPosition.getX(), myPosition.getY() - myGoalPosition.getY()) - getHeadingRadians());
			setTurnRightRadians( Math.atan( Math.tan( angle)));
			setMaxVelocity( 10 - (4 * Math.abs(getTurnRemainingRadians())));
			setAhead( (angle == Math.atan( Math.tan( angle)) ? -1.0 : 1.0) * travel);
//			out.println(angle+"  "+myGoalPosition);
		}
	}

	private double calcRisk(Point2D point, double movang)
	{
		// copied the framework from Kawigi's Coriantumr

		double totrisk = 0;
		double botrisk;
		double botangle;
		Collection enemySet;
		Iterator it = (enemySet = enemyStat.values()).iterator();

		// stay away from centre and corners
//		totrisk = (4 * (getOthers() - 1)) / point.distanceSq( centerPoint);
//		totrisk += 3 / point.distanceSq( 0, 0);
//		totrisk += 3 / point.distanceSq( fieldWidth, 0);
//		totrisk += 3 / point.distanceSq( 0, fieldHeight);
//		totrisk += 3 / point.distanceSq( fieldWidth, fieldHeight);

		do
		{
			EnemyStats es = (EnemyStats)it.next();
			if (enemies.containsKey( es.name)) {
				EnemyInfo e = (EnemyInfo)enemies.get( es.name);
 				if (e.teammate == false) {
					// calculate the perpendicularity (0 = head-on)
				    botangle = Utils.normalRelativeAngle( Math.atan2( e.getX() - point.getX(), e.getY() - point.getY()) - movang);
					// standard risk not energy-related
					botrisk = 100;
					// current target is probably more dangerous
					if (e == currentTarget) botrisk += 30;
					// if they have hit me recently, the threat is bigger
//					if (currTime - e.lastHit < 100) botrisk += 30;
					// hard-hitting enemies are more dangerous
					botrisk += ((es.damage_given * 2) / (getRoundNum() + 1));
					// enemies that I like to hit seem less dangerous
					botrisk -= (es.damage_taken / (getRoundNum() + 1));
					// try to never be the closest to anyone
					// TODO, seems important
					// add perpendicularity to the threat (a bit linear on angle seems better!)
					botrisk *= (1.0 + ((1 - (Math.abs(Math.sin(botangle)))) + Math.abs(Math.cos(botangle))) / 2);
				}
				else {
					// to avoid collisions
					botrisk = 10;
				}

				// all risks from any bot is distance related
				totrisk += (botrisk / point.distanceSq( e));
			}
		}
		while (it.hasNext());
		// and finally make a random threat on my current location
		totrisk += (Math.random() * 0.5) / point.distanceSq( myPosition);
		return totrisk;
	}

	private EnemyInfo findNewTarget( )
	{
		// my current target died, find a new one

		double totrisk = 0;
		EnemyInfo bestTarget= new EnemyInfo();
		Collection enemySet;
		Iterator it = (enemySet = enemies.values()).iterator();

		bestTarget.attrac = Double.POSITIVE_INFINITY;
//		bestTarget = (EnemyInfo)it.next();
		while (it.hasNext()) {
			EnemyInfo e = (EnemyInfo)it.next();
			// check the attractivity
			if ((e.teammate == false) && (e.attrac < bestTarget.attrac)) {
				bestTarget = e;
			}
		}
		return bestTarget;
	}

	/**
	 * onWin: Show my private victory dance
	 */
	public void onWin(WinEvent e)
	{
		printStats();
		//Victory dance	
		setTurnGunRight( Double.POSITIVE_INFINITY);
		setTurnRadarLeft( Double.POSITIVE_INFINITY);
		setTurnLeft( Double.POSITIVE_INFINITY);
		ahead(20);
		waitFor(new RadarTurnCompleteCondition(this));
	}

	/**
	 * onDeath: Show some statistics
	 */
	public void onDeath(DeathEvent e)
	{
		printStats();
	}

	/**
	 * printStats: Print statistics on end of round
	 */
	public void printStats( )
	{
		finishes[getOthers()]++;
		for (int i=0; i<finishes.length; i++)
			out.print(finishes[i] + " ");
		out.println();

//		Collection enemySet;
//		Iterator it = (enemySet = enemyStat.values()).iterator();
//		do
//		{
//			EnemyStats e = (EnemyStats)it.next();
//			out.println(e.name+" given:"+e.damage_given+"  taken:"+e.damage_taken);
//		}
//		while (it.hasNext());
	}
}

/**
 * EnemyInfo - used for keeping information on all enemies in a round
 * 		Based upon Coriantumr (smart move of Kawigi)
 */
class EnemyInfo extends Point2D.Double
{
	long lastHit;
	double energy;
	double velocity;
	double heading;
	double attrac;
	double least_enemy_distance;
	boolean teammate;
}

/**
 * EnemyStats - used for keeping information on all enemies during a battle
 */
class EnemyStats
{
	String name;
	double damage_taken;
	double damage_given;
}
