package tzu.util;

/** Provides common math methods for Robocode robots. */
public final class BotMath implements Constants {

    /** Return an angle within 0 to 360 degrees.  */
    public static double zeroTo360 (double angle) {
        return (angle < 0 ? angle % A_360 + A_360 : angle % A_360);
    }

    /**
     * Given two directions (a current one and the
     * one you wish to turn your radar/gun/robot to,
     * returns the shortest angle between the two.
     */
    public static double plusMinus180(double currDirection,
                                       double newDirection) {


        double a = (newDirection - currDirection) % A_360;

        if (a < -180) return a + A_360;
        else if (a > +180) return a - A_360;
        else return a;
    }

    /** Returns heading from one point to another. */
    public static double calcHeading(double fromX,
                                     double fromY,
                                     double toX,
                                     double toY) {

        return arcTan(toY - fromY, toX - fromX);
    }

    /** Calculates the distance between two given points. */
    public static double calcDistance(double fromX,
                                      double fromY,
                                      double toX,
                                      double toY) {
        return Math.sqrt(
               Math.pow(toY - fromY, 2) +
               Math.pow(toX - fromX, 2));
    }

    /**
     * Given a bullet's power, calculates how much damage it will do.
     */
    public static double calcBulletDamage(double bulletPower) {

        double damage = bulletPower * HIT_DAMAGE;

        if (bulletPower > 1) {
            damage += (bulletPower - 1) * ADDITIONAL_HIT_DAMAGE;
        }
        return damage;
    }


    /**
     * Calculate damage caused by running into a wall.
     */
    public static double calcWallDamage(double speed) {
        return Math.max(Math.abs(speed) * .5 - 1, 0);
    }

    public static double calcBulletSpeed(double firePower) {
        return MAX_BULLET_SPEED - (3 * firePower);
    }

    public static double calcTurnRate(double speed) {
        return TURN_CONSTANT_1 - TURN_CONSTANT_2 * speed;
    }

    /**
     * Calculate a x coordinate, given an existing x coordinate,
     * heading, and distance to new x coordinate.
     */
    public static double calcX(double x, double heading, double distance) {
        return x + sin(heading) * distance;
    }

    /**
     * Calculate a y coordinate, given an existing y coordinate,
     * heading, and distance to new y coordinate.
     */
    public static double calcY(double y, double heading, double distance) {
        return y + cos(heading) * distance;
    }

    /**
     * Given relative X and Y coordinates, returns the angle in
     * degrees to those coordinates (compass orientation).
     */
    public static double arcTan(double relY, double relX) {

        if (relX == 0.0) {
            return (relY > 0.0 ? A_360 : A_180);
        } else {
            return (relX > 0.0 ? A_90 : A_270) - atan(relY / relX);
        }
    }


    /** Returns sin of angle in degrees. */
    public static double sin(double angle) {
        return Math.sin(Math.toRadians(angle));
    }

    /** Returns cos of angle in degrees. */
    public static double cos(double angle) {
        return Math.cos(Math.toRadians(angle));
    }

    /** Returns tan of angle in degrees. */
    public static double tan(double angle) {
        return Math.tan(Math.toRadians(angle));
    }

    /** Returns absolute value of "d". */
    public static double abs(double d) {
        return Math.abs(d);
    }

    /** Returns arc tangent of angle. */
    public static double atan(double angle) {
        return Math.toDegrees(Math.atan(angle));
    }

    /** Returns arc sin of angle. */
    public static double asin(double angle) {
        return Math.toDegrees(Math.asin(angle));
    }

    /** Returns minimum of two double values. */
    public static double min(double a, double b) {
        return Math.min(a, b);
    }

    /** Retuns minimum of two long values. */
    public static long min(long a, long b) {
        return Math.min(a, b);
    }

    /** Returns maximum of two double values. */
    public static double max(double a, double b) {
        return Math.max(a, b);
    }

    /**
     * If given number below lower limit, will return lower limit;
     * if given number above upper limit, will return upper limit;
     * else returns number.
     */
    public static double limit(
            double number, double lowerLimit, double upperLimit) {

        return Math.max(Math.min(number, upperLimit), lowerLimit);
    }

    /** Returns "a" raised to the power of "b". */
    public static double pow(double a, double b) {
        return Math.pow(a, b);
    }

    /** Returns the square root of "a". */
    public static double sqrt(double a) {
        return Math.sqrt(a);
    }

    /** Rounds a double to 2 decimal points. */
    public static double round2(double a) {
        return (double)Math.round(a * 100) / 100;
    }

    /** Converts an angle from radians to degrees. */
    public static double toDegrees(double angle) {
        return Math.toDegrees(angle);
    }

    /** Converts an angle from degrees to radians. */
    public static double toRadians(double angle) {
        return Math.toRadians(angle);
    }
}
