package nz.jdc.nano;

import robocode.AdvancedRobot;
import robocode.DeathEvent;
import robocode.HitByBulletEvent;
import robocode.HitWallEvent;
import robocode.ScannedRobotEvent;
import robocode.util.Utils;

/**
 * AralT (formerly Aral) : Adaptive rolling average linear. 
 * A cousin of the NeophytePRAL/SRAL family.
 * 
 * Firing: rolling average of absolute linear velocity feeding into linear targeting,
 *   which hits orbit and stop/go movement well.  
 *   Some additional time-since-direction-change logic to detect and adapt to 
 *   oscillating movement (generally a hard counter to linear targetting) so it gets 
 *   decent hits against bullet dodge oscillation.
 *   Wave surfing, random movement or non-bullet-reacting oscillation is a problem for
 *   the oscillate detection, so it is slightly fudged using some variation
 *   (T_GF_OSCx, T_GF_0 etc) in the table lookup, which would otherwise just be T_GF_OSC 
 *   for the first 20 ticks then T_GF_1.
 * Movement: Bullet dodge. Toggles between uni-directional and reversing moves.
 * 
 * Credits: Pretty much everybody, especially those who contributed helpful articles to the
 * wiki, particularly the stuff on flat movement, rolling averages and nano-linear targetting.
 * Movement is based on the very elegant movement strategy of dft.Freddie 1.32 by Alcatraz271.
 * Updates and code size improvements from perusing rectent top bots like little black book,
 * Yatagan etc, so credit to Skilgannon, Sheldor, Michael Dorgan any anybody else whose code 
 * I have perused and forgotten to credit.
 *  
 * Version 1.0: Initial version. 
 *   The PRAL/SRAL family of bots is the closest I have come to an original thought in 
 *   robocode. This one is a distant cousin and also has some new ideas which might even be
 *   original, so I persist even though it is weaker than the current SOTA patten guns 
 *   (yatagan), preloaded GF (LBB) or rammers (Sabreur, Caligula).
 *   Codesize 248.
 *
 * Version 1.1: Changed to new "adept" style movement. Renamed to AralT.
 *   Codesize 249.
 *   
 * @author John Cleland
 */
public class AralT extends AdvancedRobot {
  // tunable constants
  // for movement
  private final static double MOVE_DISTANCE       = 80.0;
  private final static double PREFERRED_RANGE     = 220.0;
  private final static double CLOSE_FCT           = 437 * MOVE_DISTANCE;
  
  // for gun
  private final static double BULLET_POWER        = 2.45;
  private final static double BULLET_SPEED        = 20 - 3 * BULLET_POWER;
  private final static double ROLL_FACTOR         = 14;
  private final static double GF_1                = 1;
  private final static double GF_OSC              = -0.32;
  
  private final static double T_GF_1              = BULLET_SPEED / GF_1;
  private final static double T_GF_p5             = T_GF_1 * 2;
  private final static double T_GF_0              = T_GF_1 * 10;
  
  private final static double T_GF_OSC            = BULLET_SPEED / GF_OSC;
  private final static double T_GF_OSCx           = T_GF_OSC * 2;
  private final static double T_GF_OSC_b9         = T_GF_OSC / 1.7 ;
  private final static double T_GF_OSC_b8         = T_GF_OSC / 1.5 ;
  private final static double T_GF_OSC_b7         = T_GF_OSC / 1.3 ;
  
  // attributes
  // for movement
  private static double       enemyEnergy;
  private static double       moveDir;
  private static int          moveMode;
  
  // for gun  
  private static double       rAvgAbsLatV;
  private static double       eoDir = 1;
  private static int          time;

  /**
   * Bot run method.
   */
  public void run() {
    // testparam = Double.parseDouble(System.getProperty("RUMBLETESTER_PARAM"));
   
    // start our infinite radar loop
    turnRadarRightRadians(moveDir = MOVE_DISTANCE);
  }
  
  /**
   * Scan handler.
   * This is where it all happens.
   */
  public void onScannedRobot(ScannedRobotEvent e) {
    // register vars, should be the most commonly used
    // unused int    ri;
    double rd;

    // other vars
    double latVel;
    
    // radar scan
    setTurnRadarLeftRadians(getRadarTurnRemaining());
    
    // bullet dodge a random distance
   if ( (char) ((enemyEnergy - 1  - (enemyEnergy = e.getEnergy()))) <= 2) {
      moveDir *= (short) MOVE_TABLE.charAt(moveMode);
    }
    setAhead(moveDir * Math.random());
    
    // turn perpendicular with range control
    setTurnRightRadians(Math.cos((rd = e.getBearingRadians()) - (e.getDistance() - PREFERRED_RANGE) * moveDir / CLOSE_FCT));
    // rd has enemy relative bearing
    
    // targeting
    rAvgAbsLatV = (rAvgAbsLatV * ROLL_FACTOR + Math.abs(latVel = e.getVelocity() * Math.sin(e.getHeadingRadians() - (rd +=  getHeadingRadians())))) / (1 + ROLL_FACTOR);
    // rd now has enemy absolute bearing
    if (eoDir != (eoDir = Math.signum(latVel + eoDir))) {
      time = 0;
    }
    
    // nano linear targeting fn
    setTurnGunRightRadians(Utils.normalRelativeAngle(rd  - getGunHeadingRadians() 
      + (rAvgAbsLatV * eoDir / (short) OSC_LOOKUP.charAt(time = (time + 1) & 0x7F) ) ));

    // fire
    setFire(BULLET_POWER);
  }
  
  /**
   * Hit a wall, reverse direction and start moving immediately.
   */
  public void onHitWall(HitWallEvent e) {
    moveDir = -moveDir;
  }

  public void onDeath(DeathEvent event) {
    moveMode += 2;
  }

  /**
   * Hit by bullet, change movement strategy.
   */
  public void onHitByBullet(HitByBulletEvent e) {
    moveMode ^= 1;
  }
  
  // table constants
  private final static char  ORBIT = (char) 1;
  private final static char  OSC   = (char) -1;
  private final static String MOVE_TABLE = "" 
    + ORBIT + ORBIT + OSC + OSC + ORBIT + ORBIT + OSC + OSC + ORBIT + ORBIT + OSC + OSC + ORBIT + ORBIT
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC + ORBIT + OSC
    ;

  //---------------------------------------------------------------------------
  // osc threshold is implicit in the table position
  private final static String OSC_LOOKUP = ""
    /* reverse on fire     */ + (char) T_GF_OSC_b9 + (char) T_GF_OSC_b8 + (char) T_GF_OSC_b7 + (char) T_GF_OSC 
    /* osc pattern         */ + (char) T_GF_OSC    + (char) T_GF_OSCx   + (char) T_GF_OSCx   + (char) T_GF_OSC 
    /*                     */ + (char) T_GF_OSC    + (char) T_GF_OSCx   + (char) T_GF_OSCx   + (char) T_GF_OSC
    /*                     */ + (char) T_GF_OSC    + (char) T_GF_OSCx   + (char) T_GF_0  
    /*                     */ + (char) T_GF_OSC    + (char) T_GF_OSC_b7 + (char) T_GF_OSC_b8 + (char) T_GF_OSC_b9 
    /* slightly random     */ + (char) T_GF_p5     + (char) T_GF_p5     + (char) T_GF_1      + (char) T_GF_1
    /*                     */ + (char) T_GF_1      + (char) T_GF_1      + (char) T_GF_p5     + (char) T_GF_1  
    /*                     */ + (char) T_GF_0      + (char) T_GF_1      + (char) T_GF_OSC    + (char) T_GF_1  
    /*                     */ + (char) T_GF_1      + (char) T_GF_1      + (char) T_GF_1      + (char) T_GF_OSC  
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1 + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1   + (char) T_GF_1
    ;
}
