package tjk.universe;

import java.util.ArrayList;

/**
 * BracketHist - a class by tkiesel
 * Copyright (c) 2012 Tom Kiesel (Tkiesel @ Robowiki)
 * 
 * This software is provided 'as-is', without any express or implied warranty. In no event will the authors be held liable for any damages arising from the use of this software.
 * 
 * Permission is granted to anyone to use this software for any purpose, including commercial applications, and to alter it and redistribute it freely, subject to the following restrictions:
 * 
 *     1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. 
 *        If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
 * 
 *     2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
 * 
 *     3. This notice may not be removed or altered from any source distribution.
 * 
 *  In addition:
 *
 *     4. Use of this software in any programming competition, including the RoboRumble, without the written permission of the author is forbidden.
 *
 */

public class BracketHist
{
	
	// How big are the hit width slices that we chop the data set into?
	//  Note, for a bot that's stationary when the wave passes, a value of 36.0 would give
	//  single-pixel resolution.
	private final double TARGET_SLICES = 47.0;
	
	// What proportion of the precise hit width should serve as the bandwidth? (Movement)
	//  i.e.  0.5 is half hit-width from center.
	private final double MOVEMENT_BANDWIDTH = 0.5;
	
	// What proportion of the precise hit width should serve as the Gaussian bandwidth? (Targeting)
	//  i.e.  0.5 is half hit-width from center.
	private final double TARGET_BANDWIDTH = 0.75; 
	
	// First angle and step size for areas array.
	private double START_AREA_ANGLE = -1.0;
	private double STOP_AREA_ANGLE = 1.0;
	
	// Data read in from knn points.
	int POINTS;
	double[] centerAngle;
	double[] width;
	double[] height;
	private double AVERAGE_WIDTH = 0.0;
	
	// null array. Should remain unused.
	public static final double[] nullB = {1.0};
	
	// Max angle, for targeting.
	private double MAX_ANGLE = 0.0;
		
	// BracketHist Types.
	public enum HistType
	{
		TARGETING, MOVEMENT
	}
	
	public BracketHist(HistType type, ArrayList<GFBracket> brackets, double realWeight )
	{
		this(type, brackets, realWeight, 1.0);
	}
	
	public BracketHist(HistType type, ArrayList<GFBracket> brackets, double realWeight, double virtWeight )
	{
		this(type, brackets, realWeight, virtWeight, false, nullB);
	}
	
	public BracketHist(HistType type, ArrayList<GFBracket> brackets, double realWeight, double virtWeight, boolean statRoll, double[] halfLives )
	{
		// Movement bracket, or targeting bracket?
		boolean movement = ( type == HistType.MOVEMENT );
		
		// This round number. Used for stat roll.
		int thisRound = Universe.getRoundNum();
		
		// centers, widths, heights of all data points.
		POINTS = brackets.size();
		centerAngle = new double[POINTS];
		width = new double[POINTS];
		height = new double[POINTS];
		
		// Begin avg width calc and area calc.
		AVERAGE_WIDTH = 0.0;
		
		// Read all data in.
		for ( int i = 0; i < POINTS; i++ )
		{
			GFBracket gfb = brackets.get(i);
			
			// Load Width and center angle
			width[i] = gfb.b[1] - gfb.b[0];
			centerAngle[i] = (gfb.b[1] + gfb.b[0])/2.0;
			
			// Load height, modify based on real/virtual and decay.
			height[i] = gfb.real ? realWeight : virtWeight;
			if ( statRoll )
			{
				height[i] *= halfLives[thisRound - gfb.round];
			}
			
			// Continue avg width calc.
			AVERAGE_WIDTH += width[i];
			
			// Max/min angles.
			if ( centerAngle[i] < START_AREA_ANGLE )
			{
				START_AREA_ANGLE = centerAngle[i];
			}
			else if ( centerAngle[i] > STOP_AREA_ANGLE )
			{
				STOP_AREA_ANGLE = centerAngle[i];
			}
			
		}
		
		// Complete avg width calc.
		AVERAGE_WIDTH = AVERAGE_WIDTH/POINTS;
		
		// normalize danger (Movement) or locate max probability angle (targeting)
		if ( movement )
		{
			//Normalize.. (rough)
			// Curve is normalized to allow bot movement to control
			// amount of flattener applied in a predictable way.
			double total_height = 0.0;
			for ( int i = 0; i < POINTS; i++  )
			{
				total_height += height[i];
			}
			for ( int i = 0; i < POINTS; i++  )
			{
				height[i] = height[i] / total_height;
			}
			return;
		}
		
		
		double angle_step = AVERAGE_WIDTH/TARGET_SLICES;
		double max_area = Double.NEGATIVE_INFINITY;
		double BANDWIDTH_ADJUST = 1.0/TARGET_BANDWIDTH;
		
		double test_angle = START_AREA_ANGLE;
		while ( test_angle < STOP_AREA_ANGLE )
		{
			double test_area = 0.0;
			for (int j = 0; j < POINTS; j++)
			{
				double a = Math.abs(BANDWIDTH_ADJUST * (centerAngle[j]-test_angle) / width[j] );
				//test_area += height[j] * Math.exp(-0.5 * a * a);
				if ( Double.compare(a,1.0) <= 0.0 )
				{
					//TriCube
					double tc0 = 1 - a*a*a;
					test_area += height[j] * tc0*tc0*tc0;
				}
			}
			if ( Double.compare(test_area,max_area) > 0.0  )
			{
				max_area = test_area;
				MAX_ANGLE = test_angle;
			}
			test_angle += angle_step;
		}
		
	}
	
	// return max angle, targeting style.
	public double maxAngle()
	{
		return this.MAX_ANGLE;
	}
	
	// return danger area at a certain guess factor, movement style.
	public double GFtoArea(double GFangle) 
	{
		double BANDWIDTH_ADJUST = 1.0/MOVEMENT_BANDWIDTH;
		
		double danger = 0.0;
		for (int j = 0; j < POINTS; j++)
		{
			double a = BANDWIDTH_ADJUST * (centerAngle[j]-GFangle) / width[j];
			danger += height[j] * Math.pow(2, -Math.abs(a));
		}
		
		return danger;
	}
		
	// Inspired by Neil Coffey.
	// http://www.javamex.com/tutorials/math/exp.shtml
	//
	// e^x = lim(n->infinity) (1 + x/n)^n
	//
	//  Approximation using variable n, a cutoff, and assumes negative x
	public double exp(double x) 
	{
		
		if ( x < -10.0 ) //-11.5
		{
			return 0.0;
		}
		x = 1d + x / 16d;
		x *= x; x *= x; x *= x; x *= x;
		return x;
		
		/*
		// Further Approximation...
		// If you'll ever have a positive x, 
		// use: Math.abs(x) appropriately 
		if ( x < -11.5 ) //-11.5
		{
			return 0.0;
		}
		else if ( x < -5.0 )
		{
			x = 1d + x / 512d;
			x *= x; x *= x; x *= x; x *= x;
			x *= x; x *= x; x *= x; x *= x; x *= x;
			return x;
		}
		else if ( x < -3.5 )
		{
			x = 1d + x / 256d;
			x *= x; x *= x; x *= x; x *= x;
			x *= x; x *= x; x *= x; x *= x;
			return x;
		}
		else if ( x < -2.5 )
		{
			x = 1d + x / 128d;
			x *= x; x *= x; x *= x; x *= x;
			x *= x; x *= x; x *= x;
			return x;
		}
		else if ( x < -0.5 )
		{
			x = 1d + x / 64d;
			x *= x; x *= x; x *= x; x *= x;
			x *= x; x *= x;
			return x;
		}
		x = 1d + x / 32d;
		x *= x; x *= x; x *= x; x *= x;
		x *= x;
		return x;
		*/
	}
	
}