/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;

/* this class manages all movements strategies that a robot
 * can use and switches them if needed;
 */
public class MovementManager
{
   // movement strategies are switched continously in random intervals
   /* minimum and maximum length of interval[frames] in 
    * which the movement stategy doesn't change
    */
   static final long minMovementInterval = 60;
   static final long maxMovementInterval = 210;
   
   // robot for which this manager is used
   AdvancedRobot robot;
   // robot's body
   Body body;
   // enemy to avoid in one-on-one battle
   Enemy enemy;

   // movement strategies used for the robot
   Movement[] movements;
   int movementIndex = 0;  //< current movement strategy index
   Movement movement;      //< current movement
   // frame in which the next movement strategy will be used
   long nextMovementChange = 0;
      
   /* at construction provide robot, its body
    * and enemy in one-on-one battle
    */
   public MovementManager( AdvancedRobot robot, Body body, Enemy enemy )
   {
      this.robot = robot;
      this.body = body;
      this.enemy = enemy;
      initialize();
   }
   
   /* manages the strategies and runs an active strategy
    */   
   public void run()
   {
      // change movement strategy if needed
      switchStrategy();
      // prepare orders
      movement.run();
   }

   /* prepare movement strategies
    */
   void initialize()
   {
      // create movement logics
      movements = new Movement[] {
         new CenterOrbitalMovement( robot, body, enemy ),
         new SectorMovement( robot, body, enemy )
      };
      // select current movement strategy
      movementIndex = 0;
      movement = movements[movementIndex];
   }
   
   /* takes care to change movement strategy in random intervals
    */
   void switchStrategy()
   {
      if ( robot.getTime() >= nextMovementChange ) {
         // select new movement change time
         nextMovementChange = robot.getTime() 
            + Util.randomLong( minMovementInterval, maxMovementInterval );
         movementIndex++;
         if ( movementIndex >= movements.length )
            movementIndex = 0;
         // select new movement  
         movement = movements[movementIndex];
      }
   }
   
   /*
    * event handlers redirecting events to active movement strategy
    */
    
   public void onHitRobot( HitRobotEvent e )
   {
      movement.onHitRobot( e );
   }
   
   public void onHitWall( HitWallEvent e )
   {
      movement.onHitWall( e );
   }
   
   public void onHitByBullet( HitByBulletEvent e )
   {
      movement.onHitByBullet( e );
   }
   
}