package gh.ghmove;

import java.awt.Color;
import java.awt.geom.*;
import java.util.*;

import robocode.*;
import robocode.util.Utils;
import gh.ghutils.*;

/**
 * WeefSuffurd - a movement by Gert Heijenk
 *
 * Movement system including:
 * - WaveSurfing
 * (- Flattener )
 * (- Passive BulletShadows )
 *
 * Although the code itself is mine, the refactoring is inspired by Dookious by Voidious
 *
 * Code is open source, released under the RoboWiki Public Code License:
 * http://robowiki.net/cgi-bin/robowiki?RWPCL
 */
public class WeefSuffurd {
	// some constants
	static final String VERSION			= "0.4.9";
	static final double DEFDISTANCE		= 495;  // preferred distance
	static final double DEFDISTSURV		= 270;	// preferred distance for Survivalists

	private static double	prefDistance;
    private static SurfStat   surfStats = new SurfStat();
	private static EnemyWave  virtWave  = new EnemyWave();	// re-use one virtual wave

	private ArrayList<EnemyWave> eWaveList;

	int dirToGoBck;
	long currTime;					// the current scantime

// for the Paint routine
//	Point2D.Double posCW  = new Point2D.Double();
//	Point2D.Double posCCW = new Point2D.Double();
//	Point2D.Double posHLT = new Point2D.Double();

	// for (not so) precise prediction
	boolean wallhitCW, wallhitCCW, wallhitHLT, wallHitEminent;

	private AdvancedRobot robot;

	/**
	 * WeefSuffurd
	 */
	public WeefSuffurd( AdvancedRobot robot) {
		this.robot = robot;

		eWaveList = new ArrayList<EnemyWave>();

		if (robot.getRoundNum() == 0) {
			System.out.println("WeefSuffurd movement version: " + VERSION);
			GHUtils.setBattleFieldSize( robot.getBattleFieldWidth(), robot.getBattleFieldHeight());
			GHUtils.setGunCoolingRate( robot.getGunCoolingRate());
			surfStats.recordHit( GHUtils.MMIDBIN, GHUtils.MMIDBIN, 0, 0, 4, 0);	// only to avoid HOT !
			EnFireEval.putFirePower( 0.6);			// checking for pea-shooters ??
			virtWave.setVirtDefault();
		}
		GHUtils.setRoundNumber(robot.getRoundNum());
	}

	// onScannedRobot: Event from scanning another robot
    public void onScannedRobot( ScannedRobotEvent e) {
		currTime = robot.getTime();
		
		// Check if enemy fired, and handle accordingly
		double enemyFirePower = TickState.doCheckEnemyFired();
		if (enemyFirePower != 0.0) {
			EnemyWave eWave = new EnemyWave();
			eWave.bulletVelocity = Rules.getBulletSpeed( enemyFirePower);
			addNewWave( eWave, true);
		}
		else if (getWaveToSurf( true) == null) {
			addNewWave( virtWave, true);
		}

		doCheckEnemyWaves();
		if (GunBullets.checkActiveBullets() == true) {
//System.out.println(currTime+" New bullet ");
 			for (int i = eWaveList.size() - 1; i >= 0; i--) {
				EnemyWave currWave = (EnemyWave)eWaveList.get( i);
				if (currWave.trueWave == true) {
					GunBullets.addShadows(currWave);
				}
			}
		}
		doCheckMovement();
	}

	// simulatedScannedRobot: Simulated event to keep surfing airborne enemybullets
	//                        when enemy is already killed
	public boolean simulatedScannedRobot( ) {
		currTime = robot.getTime();
		TickState.simulatedScannedRobot(robot);
		doCheckEnemyWaves();
		doCheckMovement();
		return (eWaveList.isEmpty());
	}

	// addNewWave: Enemy fired (also virtual)
	public void addNewWave( EnemyWave ew, boolean normal) {

		ew.fireTime = robot.getTime() - 1;
		ew.direction = TickState.getDirection(2);
		ew.HOTAngle = TickState.getEneBearing(2);
		if (normal) { ew.fireLocation.setLocation( TickState.getEnemyPos(1)); }
		else { ew.fireLocation.setLocation( TickState.getEnemyPos(0)); }
		ew.myVelocityBin = TickState.getMyVelBin(2);
		ew.myNearWallBin = TickState.getMyWallBin(2);
		ew.myAccelerationBin = TickState.getMyAccBin(2);
		ew.myDistLast14Bin = TickState.getMyDL14Bin(2);
		// calculate (one time) risks for each bin
		if (ew.trueWave == true) {
			// for real wave, get real wallbin
			ew.myNearWallBin = TickState.getMyWallBin();
			ew.setRisks( surfStats.binDanger( ew));
			GunBullets.addNewWave( ew);
		}
		if (ew.trueWave == true || getVirtualWaves() == 0) {
			eWaveList.add( ew);
		}
	}

	// onHitByBullet: Log hit and update surfing statistics
	public void onHitByBullet( HitByBulletEvent e) {
		if (eWaveList.isEmpty()) {
			System.out.println( robot.getTime()+" Hit by ghost bullet(" + e.getPower() + ") I did not detect an EnemyFire");
			// why do I not log the hit ??
			Point2D.Double hitLocation = new Point2D.Double( e.getBullet().getX(), e.getBullet().getY());
			double dist = TickState.getEnemyPos().distance(hitLocation);
			long firetime = robot.getTime() - (long)(dist / e.getVelocity());
			System.out.println("Estimated firetime is : "+firetime);
			return;
		}

		if (e.getPower() != e.getBullet().getPower()) System.out.println(" !!?? E" + e.getPower() + " B " + e.getBullet().getPower());
		logEnemyHit( true, e.getBullet());
		TickState.addEnFired(e.getPower(), true);
		recalcRisks();
	}

	// onBulletHit: My bullet hit the opponent, check for rare 'bullet fired while killed' situation
	public void onBulletHit( BulletHitEvent e) {
		if (robot.getOthers() == 0) {
            double fpow = TickState.getAssFirePower();
		    if (fpow > 0.01) {
			    System.out.println( robot.getTime() + " Enemy able to fire while killed, assume bulletpower = " + fpow);
	    		EnemyWave eWave = new EnemyWave();
		    	eWave.bulletVelocity = Rules.getBulletSpeed( fpow);
			    addNewWave( eWave, false);
		    }
			else System.out.println( robot.getTime() + " Naah, enemy did not fire while killed . . .");
		}
	}

	// onBulletHitBullet: No rare occasion, against other good bots it happens ~3 times/round
	public void onBulletHitBullet( BulletHitBulletEvent e) {
		if (eWaveList.isEmpty()) {
			System.out.println( robot.getTime()+" Bullet hit by ghost bullet, I did not detect an EnemyFire");
			// why do I not log the hit ??
			Point2D.Double hitLocation = new Point2D.Double( e.getHitBullet().getX(), e.getHitBullet().getY());
			double dist = TickState.getEnemyPos().distance(hitLocation);
			long firetime = robot.getTime() - (long)(dist / e.getHitBullet().getVelocity());
			System.out.println("Estimated firetime is : "+firetime);
		
			return;
		}

		logEnemyHit( false, e.getHitBullet());
		recalcRisks();
	}

	// printStats
	public void printStats() {
		surfStats.printSegStats();
	}

	//CheckEnemyWaves: Check the status of the enemy waves
	void doCheckEnemyWaves() {
		int i;
		boolean scndwave = false;
		
		// for every still available enemywave
		for (i = eWaveList.size() - 1; i >= 0; i--) {
			EnemyWave currWave = (EnemyWave)eWaveList.get( i);
			currWave.updateStatus( currTime, TickState.getMyPos());

			// remove real waves when really passed me
			if (currWave.trueWave == true) {
				if (currWave.checkPassed()) {
					TickState.addEnFired((20 - currWave.bulletVelocity)/3, false);
					eWaveList.remove(i);
				}
				else
					scndwave = true;
			}
			else { // remove simulated wave if more than 2, or when real wave exists
				if ((scndwave == true) || (robot.getOthers() == 0))
					eWaveList.remove(i);
				scndwave = true;
			}
		}
	}

	// doCheckMovement: Check if something has to be done about the movement
	void doCheckMovement() {

		int dirToGo;
		double goAngle, adjustAngle;
		double dangerLeft, dangerRight, dangerStop;
		EnemyWave secondWave;

		if (robot.getOthers() > 0) {	// don't switch distance if opponent is already killed
			if (TickState.useShortDistance()) {
				if (prefDistance != DEFDISTSURV) System.out.println(currTime+" Switching to SHORT distance");
				prefDistance = DEFDISTSURV;
			} else {
				if (prefDistance != DEFDISTANCE) System.out.println(currTime+" Switching to LONG distance");
				prefDistance = DEFDISTANCE;
			}
		}
		goAngle = TickState.getEneBearing();
		double enemyDist = TickState.getDistance();
		adjustAngle = (enemyDist / prefDistance) * Math.PI/2;
		adjustAngle = Math.max( 0.0, Math.min( adjustAngle, Math.PI));
//		adjustAngle = Math.max( 0.2, Math.min( adjustAngle, Math.PI));
//		System.out.print(currTime+" "+enemyDistance+" "+mySpeed_t);

		EnemyWave surfWave = getWaveToSurf( false);
		secondWave = getSecondWaveToSurf( surfWave);
		dangerLeft = dangerRight = dangerStop = 0.0;

		boolean useShadow = TickState.checkUseBulletShadows();
		// calculate secondWave first! the danger must be added to the surfWave
		if (secondWave != null) {
			dangerLeft = checkDanger( secondWave, -1, useShadow);
			dangerRight = checkDanger( secondWave, 1, useShadow);
			dangerStop = checkDanger( secondWave, 0, useShadow);
		}
		wallhitCW = wallhitCCW = wallhitHLT = false;

		if (surfWave != null) {
			if ((surfWave.trueWave == true) || 
			    ((enemyDist < prefDistance/1.5) || (enemyDist > prefDistance * 1.1))) {
//System.out.println("DLno "+checkDanger(surfWave, -1, false) + " DLsh "+checkDanger(surfWave, -1, true));
//System.out.println("DRno "+checkDanger(surfWave,  1, false) + " DRsh "+checkDanger(surfWave,  1, true));
//System.out.println("DSno "+checkDanger(surfWave,  0, false) + " DSsh "+checkDanger(surfWave,  0, true));
				dangerLeft += checkDanger( surfWave, -1, useShadow);
				dangerRight += checkDanger( surfWave, 1, useShadow);
				if (enemyDist > prefDistance/2.0)
					dangerStop += checkDanger( surfWave, 0, useShadow);
				else {
					dangerStop = Double.POSITIVE_INFINITY;
//					System.out.println("Do not stop!");
				}
//			System.out.println(currTime+" "+dangerLeft+" "+dangerStop+" "+dangerRight);
				// prefer slightly current direction
				if (dirToGoBck == 1) dangerRight *= 0.95;
				if (dirToGoBck == -1) dangerLeft *= 0.95;
				if (dirToGoBck == 0) dangerStop *= 0.95;

				if (Utils.isNear(dangerLeft, dangerRight) && (dangerRight == 0.0) && (dirToGoBck != 0)) {
					dirToGo = dirToGoBck;
				}
				else if (dangerLeft < dangerRight) {
					if (dangerLeft <= dangerStop) dirToGo = -1;
					else dirToGo = 0;
				}
//				else if ((dangerStop < dangerRight) || (dangerStop == 0.0)) dirToGo = 0;
				else if (dangerStop < dangerRight) dirToGo = 0;
				else dirToGo = 1;
//			System.out.println("F dR: "+dangerRight+" dL: "+dangerLeft+" dS: "+dangerStop);
			}
			else {
				// virtual wave and in the 'goldilock zone'
				dirToGo = 0;
			}
		}
		else {
			// no wave at all, happens only after kill is made and still bullet somewhere behind me
			dirToGo = 0;
//			System.out.println(" no wave !! ");
		}

		// from here I need to know the angles
		if (dirToGo == 0) {
			goAngle += adjustAngle;
			wallHitEminent = wallhitHLT;
		}
		else { wallHitEminent = (dirToGo == 1 ? wallhitCW : wallhitCCW); }
		goAngle = GHUtils.wallSmoothing( TickState.getMyPos(), goAngle + (adjustAngle * dirToGo));
//		System.out.println(currTime+" "+dirToGo+"  "+goAngle);
		GHUtils.setBackAsFront( robot, goAngle, dirToGo);
		robot.setMaxVelocity( wallHitEminent == true ? 0 : Rules.MAX_VELOCITY);
		dirToGoBck = dirToGo;
	}

	// getWaveToSurf: Get the first (virtual) wave I have to surf (first intercepting, wettest . . .)
	EnemyWave getWaveToSurf( boolean real) {
		double interceptTime = Double.POSITIVE_INFINITY;
		EnemyWave waveToSurf = null;

		for (int i = eWaveList.size() - 1; i >= 0; i--) {
			EnemyWave currWave = (EnemyWave)eWaveList.get( i);
			// skip virtual wave if real wave requested
			if ((currWave.checkActive() == true) && (real == false) || (currWave.trueWave == true)) {
				double waveHitTime = currWave.getHitTime();
				if ((waveHitTime < interceptTime) && (waveHitTime > currTime)) {
					waveToSurf = currWave;
					interceptTime = waveHitTime;
				}
			}
		}
		return waveToSurf;
	}

	// getSecondWaveToSurf: Get the second real wave I have to surf (second intercepting)
	EnemyWave getSecondWaveToSurf( EnemyWave skipwave) {
		double interceptTime = Double.POSITIVE_INFINITY;
		EnemyWave waveToSurf = null;

		for (int i = eWaveList.size() - 1; i >= 0; i--) {
			EnemyWave currWave = (EnemyWave)eWaveList.get( i);
			if ((currWave != skipwave) && (currWave.trueWave == true) && (currWave.checkActive() == true)) {
				double waveHitTime = currWave.getHitTime();
				if ((waveHitTime < interceptTime) && (waveHitTime > currTime)) {
					waveToSurf = currWave;
					interceptTime = waveHitTime;
				}
			}
		}
		return waveToSurf;
	}

	// checkRealWave: Check if a real wave is present
	boolean checkRealWave() {

		for (int i = 0; i < eWaveList.size(); i++) {
			EnemyWave currWave = (EnemyWave)eWaveList.get( i);
			if ((currWave.trueWave == true) && (currWave.checkActive() == true)) {
				return (true);
			}
		}
		return (false);
	}

	// getVirtualWaves: Get number of virtual waves
	int getVirtualWaves() {
		int nrwaves = 0;
		for (int i = 0; i < eWaveList.size(); i++) {
			EnemyWave currWave = (EnemyWave)eWaveList.get( i);
			if (currWave.trueWave == false) {
				nrwaves++;
			}
		}
		return (nrwaves);
	}

	// logEnemyHit: find hitting wave, calculate bin and update stats
	public void logEnemyHit( boolean trueHit, Bullet enemyBullet) {
		int i, hitbin, minbin, maxbin;
		EnemyWave hitWave = null;

		Point2D.Double hitLocation = new Point2D.Double( enemyBullet.getX(), enemyBullet.getY());
		// look through the EnemyWaves, and find the one that hit us (even if different bulletpower).
		for (i = eWaveList.size() - 1; i >= 0; i--) {
			EnemyWave currWave = (EnemyWave)eWaveList.get( i);
			double traveldist = (robot.getTime() - currWave.fireTime) * enemyBullet.getVelocity();
//System.out.println("j "+j+" traveldist "+traveldist+"  wavedist "+currWave.fireLocation.distance(hitLocation));
			// normally, bullethits are exact on correct distance
			// but bullethitbullet hits are 1 off or 2 off (depending which bullet moves first ??)
			if ((currWave.trueWave == true) &&
				(Utils.isNear(currWave.fireLocation.distance(hitLocation), traveldist) ||
				 Utils.isNear(currWave.fireLocation.distance(hitLocation), traveldist - enemyBullet.getVelocity()) ||
				 Utils.isNear(currWave.fireLocation.distance(hitLocation), traveldist - 2*enemyBullet.getVelocity()) )) {
					hitWave = currWave;
					if (!Utils.isNear(currWave.bulletVelocity, enemyBullet.getVelocity())) {
						System.out.println(robot.getTime()+" Real bullet fired was "+enemyBullet.getPower());
					}
					break;
			}
		}

		if (hitWave != null) {
			hitbin = minbin = maxbin = Math.max(0, Math.min(hitWave.getHittedBin( enemyBullet.getHeadingRadians()), GHUtils.MBINS-1));

//			// if I want to calculate every position where I would have been hit
//			double hitangle = doGetAngle( hitWave.fireLocation, hitLocation);
////			System.out.println( robot.getTime()+" hitloc: "+hitLocation);
//			double chkdist  = myPos.distance( hitWave.fireLocation);
//			bullet.setLine( hitWave.fireLocation, doProjectPos( hitWave.fireLocation, hitangle, chkdist+100));
////			System.out.println( robot.getTime()+" Line: "+bullet);
//			for (int tbin = hitbin; tbin >= 0; tbin--) {
//				double tguessfactor = (double)(tbin-MIDBIN) / MIDBIN;
//				double tangleOffset = hitWave.direction * tguessfactor * hitWave.maxEscAngle;
//				double thead = Utils.normalRelativeAngle( hitWave.HOTAngle + tangleOffset);
//				Point2D.Double chkpoint = doProjectPos( hitWave.fireLocation, thead, -chkdist);
////				System.out.println( "bin: "+tbin+" chkpoint: "+chkpoint);
//				botspace.setRect( chkpoint.x - HALFBOTWIDTH, chkpoint.y - HALFBOTWIDTH, BOTWIDTH, BOTWIDTH);
//				if (bullet.intersects( botspace))
//					minbin = tbin;
//				else
//					break;
//			}
//			for (int tbin = hitbin; tbin < BINS; tbin++) {
//				double tguessfactor = (double)(tbin-MIDBIN) / MIDBIN;
//				double tangleOffset = hitWave.direction * tguessfactor * hitWave.maxEscAngle;
//				double thead = Utils.normalRelativeAngle( hitWave.HOTAngle + tangleOffset);
//				Point2D.Double chkpoint = doProjectPos( hitWave.fireLocation, thead, -chkdist);
////				System.out.println( "bin: "+tbin+" chkpoint: "+chkpoint);
//				botspace.setRect( chkpoint.x - HALFBOTWIDTH, chkpoint.y - HALFBOTWIDTH, BOTWIDTH, BOTWIDTH);
//				if (bullet.intersects( botspace))
//					maxbin = tbin;
//				else
//					break;
//			}
//			System.out.println("Mark dangerous: "+minbin+"-"+maxbin+" hitted="+hitbin);

			surfStats.recordHit( minbin, maxbin, hitWave.myVelocityBin, hitWave.myNearWallBin, hitWave.myAccelerationBin, hitWave.myDistLast14Bin);
			eWaveList.remove( i);	// We can remove this wave now, of course.
		}
		else {
			System.out.println( robot.getTime()+" Hit by unmatched "+enemyBullet.getPower()+" bullet, try even harder to match it");
//			double dist = TickState.getEnemyPos().distance(hitLocation);
//			long firetime = robot.getTime() - (long)(dist / enemyBullet.getVelocity());
//			System.out.println("Estimated firetime is : "+firetime);
//// ====
//		for (i = eWaveList.size() - 1; i >= 0; i--) {
//			EnemyWave currWave = (EnemyWave)eWaveList.get( i);
//			double traveldist = (robot.getTime() - currWave.fireTime) * enemyBullet.getVelocity();
//System.out.println("j "+j+" traveldist "+traveldist+"  wavedist "+currWave.fireLocation.distance(hitLocation));
//if (!Utils.isNear(currWave.bulletVelocity, enemyBullet.getVelocity())) {
//	System.out.println(robot.getTime()+" Matching veldet "+currWave.bulletVelocity+"  with bulletspeed "+enemyBullet.getVelocity());
//}
//		}
//// ====
		}
	}

	// recalcRisks: recalculate risks for every real wave
	public void recalcRisks( ) {
		for (int i = eWaveList.size() - 1; i >= 0; i--) {
			EnemyWave ew = (EnemyWave)eWaveList.get( i);
			if ((ew.trueWave == true) && (ew.checkActive() == true)) {
				ew.setRisks( surfStats.binDanger( ew));
			}
		}
	}

	// Copy from BasicSurfer
	// CREDIT: mini sized predictor from Apollon, by rozu
	// http://robowiki.net?Apollon
	// adapted with own distancing and evaluation of 'stop'
	// also adapted with 'constant distance' when facing a rammer
	public Point2D.Double predictPosition( EnemyWave surfWave, int direction) {
		Point2D.Double predictedPosition = (Point2D.Double)TickState.getMyPos().clone();
		double predictedVelocity = robot.getVelocity();
		double predictedHeading = robot.getHeadingRadians();
		double maxTurning, moveAngle, moveDir;
		double angle, adjustAngle;
		double enemyDist = TickState.getDistance();
		int counter = 0; // number of ticks in the future
		boolean intercepted = false;

		do {
 			angle = GHUtils.doGetAngle( surfWave.fireLocation, predictedPosition);
//			if (enemyAdvVelocity > 5.6) {
			if (TickState.getCollisions() > (GHUtils.roundNumber)) {
				adjustAngle = ( enemyDist / prefDistance) * Math.PI/2;
			} else {
				adjustAngle = ( predictedPosition.distance( surfWave.fireLocation) / prefDistance) * Math.PI/2;
			}
			adjustAngle = Math.max( 0.2, Math.min( adjustAngle, Math.PI));
			if (direction == 0) angle += adjustAngle;
			moveAngle =  GHUtils.wallSmoothing( predictedPosition, angle + (direction * adjustAngle)) - predictedHeading;
			moveDir = 1;

			if(Math.cos(moveAngle) < 0) {
				moveAngle += Math.PI;
				moveDir = -1;
			}

			moveAngle = Utils.normalRelativeAngle(moveAngle);

			// maxTurning is built in like this, you can't turn more then this in one tick
			maxTurning = Rules.getTurnRateRadians( Math.abs(predictedVelocity));
//			maxTurning = Math.PI/720d*(40d - 3d*Math.abs(predictedVelocity));
			predictedHeading = Utils.normalRelativeAngle(predictedHeading
				+ Math.max(-maxTurning, Math.min(moveAngle, maxTurning)));

			// this one is nice ;). if predictedVelocity and moveDir have
			// different signs you want to brake down
			// otherwise you want to accelerate (look at the factor "2")
			if (direction == 0) {
				predictedVelocity += (predictedVelocity < 0 ? 2 : -2);
				if (Math.abs(predictedVelocity) < 2.0) {
					predictedVelocity = 0.0;
					intercepted = true;
				}
			}
			else
				predictedVelocity += (predictedVelocity * moveDir < 0 ? 2*moveDir : moveDir);
			predictedVelocity = Math.max(-Rules.MAX_VELOCITY, Math.min(predictedVelocity, Rules.MAX_VELOCITY));

			// calculate the new predicted position
			predictedPosition = GHUtils.doProjectPos( predictedPosition, predictedHeading, predictedVelocity);

			if (!GHUtils.moveField.contains( predictedPosition)) {
				if (direction == -1) wallhitCCW = true;
				else if (direction == 1) wallhitCW = true;
				else wallhitHLT = true;
			}

			counter++;

			if (enemyDist > DEFDISTSURV) {
				// calculate position one tick earlier, because bullet moves first
				if (predictedPosition.distance( surfWave.fireLocation) <
					surfWave.getWaveDistance( currTime + counter + 1)) {
					intercepted = true;
				}
			} else {
				// do on normal tick if to close
				if (predictedPosition.distance( surfWave.fireLocation) <
					surfWave.getWaveDistance( currTime + counter)) {
					intercepted = true;
				}
			}
		} while(!intercepted && counter < 200);

//		if (direction == -1) posCCW.setLocation( predictedPosition);
//		else if (direction == 1) posCW.setLocation( predictedPosition);
//		else posHLT.setLocation( predictedPosition);

		return predictedPosition;
	}	
	//	end copy of BasicSurfer ( mini-sized precise predictor of Rozu)


	// checkDanger: get the danger for the wave when going in 'direction'.
    public double checkDanger(EnemyWave surfWave, int direction, boolean useShadows) {
		int index = surfWave.getHittedBin( predictPosition( surfWave, direction));
		index = Math.max(0, Math.min(index, GHUtils.MBINS-1));
		double danger = surfWave.checkDanger( index, useShadows);
		// danger adaptions
		danger *= (10.0 / (4.0 + (surfWave.getHitTime() - (double)currTime)));	// bullet flighttime dependent
		danger *= (1.0 + (20.0 - surfWave.bulletVelocity)/3.0);		// bulletpower dependent
		return danger;
	}


	/**
	 * onPaint: Show some stuff, maybe I am calculating something wrong
	 */
//	public void onPaint(java.awt.Graphics2D g) {
//		int i, j;
//		double dist, tothits, binangle, bindanger;
//		Point2D.Double botpos = new Point2D.Double();
//		Point2D.Double HOTpos = new Point2D.Double();
//		Point2D.Double binpos = new Point2D.Double();

////		g.setColor( Color.yellow);
////		g.drawRect( (int)(myPos.getX() - HALFBOTWIDTH), (int)(myPos.getY() - HALFBOTWIDTH), (int)BOTWIDTH, (int)BOTWIDTH);
//		// print enemy waves to check if I am aligned
//		for (i = eWaveList.size() - 1; i >= 0; i--) {
//			enemyWave currWave = (enemyWave)eWaveList.get( i);
//			dist = (robot.getTime() - currWave.fireTime) * currWave.bulletVelocity;
//			botpos.setLocation( currWave.fireLocation);
//			HOTpos.setLocation( GHUitls.doProjectPos( botpos, currWave.HOTAngle, -dist));
//			g.setColor( Color.gray);
//			if (currWave.trueWave == true) {
////				g.drawOval( (int)(botpos.getX() - dist), (int)(botpos.getY() - dist), (int)(dist*2),(int)(dist*2));
//				g.drawLine( (int)botpos.getX(), (int)(botpos.getY()), (int)HOTpos.getX(), (int)(HOTpos.getY()));
////				tothits = velStat[currWave.myVelocityBin][BINS] + wallStat[currWave.myNearWallBin][BINS];
//				for (j = 0; j < BINS; j++) {
//					binangle = (((double)j - MIDBIN) / MIDBIN) * currWave.maxEscAngle * currWave.direction;
//					binpos.setLocation( doProjectPos( botpos, currWave.HOTAngle + Math.PI + binangle, dist - 4));
////					bindanger = (int)(((velStat[currWave.myVelocityBin][j] * 2 + wallStat[currWave.myNearWallBin][j]) * 255) / tothits);
//					bindanger = currWave.binrisks[j] * 127 / currWave.avgrisk;
////					System.out.println("bin " + j + " dang " + (int)bindanger);
//					g.setColor( new Color((int)Math.max( 0, Math.min( bindanger, 255)), 0, (int)Math.max(0, Math.min(255 - bindanger, 255))));
//					g.drawOval( (int)(binpos.x - 1), (int)(binpos.y - 1), 2, 2);
//				}
//			} else {	// virtual wave(s)
//				g.drawOval( (int)(botpos.getX() - dist), (int)(botpos.getY() - dist), (int)(dist*2),(int)(dist*2));
//			}
//		}
//		if (checkRealWave() == true) {
//			g.setColor( Color.white);
//			g.drawRect( (int)(posHLT.getX() - 4), (int)(posHLT.getY() - 4), 8, 8);
//			g.setColor( Color.red.brighter());
//			g.drawRect( (int)(posCCW.getX() - 4), (int)(posCCW.getY() - 4), 8, 8);
//			g.setColor( Color.red.darker());
//			g.drawRect( (int)(posCW.getX() - 4), (int)(posCW.getY() - 4), 8, 8);
//		}
//	}


}
