package pl.Patton.Strategies;

import robocode.*;

import java.util.ArrayList;

import pl.Abstract.Strategy;
import pl.Enemy.AdvancedEnemy;
import pl.Patton.Guns.PTGun;
import pl.Utilities.AngleUtils;

/*******************************************************************************
 * A strategy for melee battles.
 * 
 * It utilizes a strafing movement, a predictive gun, and a non-stop radar.
 ******************************************************************************/
public class Melee extends Strategy {
	// Track my enemies
	public ArrayList<AdvancedEnemy> enemies = new ArrayList<AdvancedEnemy>(robot.getOthers());
	// Keep track of the one I want to track
	public int myEnemy = 0;
	// Remember which direction I'm moving in
	public byte moveDir = 1;
	// My gun
	public PTGun gun = new PTGun("Predictive Gun", robot);

	public Melee(AdvancedRobot robot) {
		super(robot);
	}

	public void run() {
		// Make myself flexible - very flexible
		robot.setAdjustRadarForRobotTurn(true);
		robot.setAdjustRadarForGunTurn(true);
		robot.setAdjustGunForRobotTurn(true);
		// While it's a melee
		while (robot.getOthers() > 1) {
			doRadar();
			// See, it takes a certain amount of time to find an enemy, and
			// I want to avoid NPEs
			if (robot.getTime() > 8) {
				doGun();
				doTank();
			}
			robot.execute();
		}
	}

	public void onScannedRobot(ScannedRobotEvent e) {
		boolean updated = false;
		// See if I already have a record of him
		for (int i = 0; i < enemies.size(); i++) {
			// If I do, then update my record
			if (enemies.get(i).name.equals(e.getName())) {
				enemies.get(i).update(e, robot);
				updated = true;
				break;
			}
		}
		// If I don't have him already, then add him
		if (!updated)
			enemies.add(new AdvancedEnemy(e, robot));
		getClosestEnemy();
	}

	public void onBulletHit(BulletHitEvent e) {
		// The enemy that hit me has recharged
		for (int i = 0; i < enemies.size(); i++)
			if (enemies.get(i).name.equals(e.getName())) {
				enemies.get(i).energy = e.getEnergy();
				break;
			}
	}

	public void onHitByBullet(HitByBulletEvent e) {
		// I want to track the bastard that killed me
		if (robot.getEnergy() < 0.01) {
			for (int i = 0; i < enemies.size(); i++)
				if (enemies.get(i).name.equals(e.getName())) {
					myEnemy = i;
					break;
				}
		}
	}

	public void onDeath(DeathEvent e) {
		// Make sure I get the onHitByBullet event so I know killed me
		for (Object obj : robot.getAllEvents()) {
			if (obj instanceof HitByBulletEvent)
				onHitByBullet((HitByBulletEvent) obj);
		}
	}

	public void onRobotDeath(RobotDeathEvent e) {
		for (int i = 0; i < enemies.size(); i++) {
			// When I've found the one that died
			if (enemies.get(i).name.equals(e.getName())) {
				// Mark him dead
				enemies.get(i).isNowDead();
				// If I was tracking it, find a new one
				if (enemies.get(myEnemy).name.equals(e.getName()))
					getClosestEnemy();
				break;
			}
		}
	}

	/**
	 * Finds the closest enemy from enemies and sets myEnemy as its index.
	 */
	public void getClosestEnemy() {
		// Simple loop
		for (int i = 0; i < enemies.size(); i++) {
			if (enemies.get(i).distance < enemies.get(myEnemy).distance - 30
					|| (enemies.get(myEnemy).name.equals(enemies.get(i).name) && !enemies.get(i).isDead)) {
				myEnemy = i;
			}
		}
	}

	/**
	 * Controls the radar for this strategy.
	 * 
	 * It basically whips the radar around forever.
	 */
	public void doRadar() {
		robot.setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
	}

	/**
	 * Controls the gun for this strategy.
	 * 
	 * It utilizes predictive targeting.
	 */
	public void doGun() {
		// Call my gun with my current enemy
		gun.run(enemies.get(myEnemy));
	}

	/**
	 * Controls the tank for this strategy.
	 * 
	 * It strafes from the closest enemy
	 */
	public void doTank() {
		// Square off against the enemy
		robot.setTurnRightRadians(AngleUtils.normalizeBearing(enemies.get(myEnemy).bearing
				+ AngleUtils.HALF_PI));
		// Switch directions every 15 ticks
		if (robot.getTime() % 15 == 0) {
			moveDir *= -1;
			robot.setAhead(100 * moveDir);
		}
	}
}