package tzu.gun;

import tzu.intel.*;
import tzu.util.*;
import robocode.*;

/**
 * Provides one static method, calculate, which calculates the best
 * firepower to use based on how much energy the target has left,
 * how much energy we have left, and how good are chances are of
 * hitting the given target.
 */
public final class FirePower implements Constants {

    /**
     * Calculate the best firepower to use to shoot at this target.
     * @param target    the enemy robot to shoot at.
     * @return          bullet firepower
     */
    public static double calculate(Bot target) {

        double power;
        double angleOfApproach;
        double speedRatio;
        double distanceRatio;
        double angleRatio;
        AdvancedRobot myRobot   = target.getMyRobot();
        double speed            = target.getSpeed();
        double heading          = target.getHeading();
        double bearing          = target.getBearing();
        double energy           = target.getEnergy();
        double distance         = target.getDistance();
        double avgSpeed         = target.getAvgSpeed();
        double myEnergy         = myRobot.getEnergy();

        if (myEnergy <= 0.1) return 0;

        /* If we are almost dead, don't waste our firepower. */
        if (myEnergy < 1) return MIN_BULLET_POWER;

        /*
         * If the target is almost dead, don't use more
         * energy than is necessary.  Also, if our robot
         * is almost dead, conserve what little energy
         * we have left.
         */
        power = BotMath.min((energy / HIT_DAMAGE),MAX_BULLET_POWER);
        power = BotMath.min((myRobot.getEnergy() / 3), power);

        /*
         * Determine the angle the target is approaching us at.  A target
         * coming closer, especially one coming straight at us, is easier
         * to hit.
         */
        angleOfApproach = BotMath.abs(BotMath.plusMinus180(bearing, heading));

        /*
         * If target is currently stationary, it might move when fired at.
         * Since there is no telling which direction it might move in (forward
         * or reverse), use a low power shot unless the angle of attack is small
         * and target is close.
         */
        if (speed           == 0.0          &&
            avgSpeed        > 2             &&
            angleOfApproach > 15            &&
            distance        > SKIP          &&
            (target.getPreviousReaction() == null  ||
             target.getPredictHitPct()      < 20)) {
            return ALMOST_MIN_BULLET_POWER;
        }

        /*
         * If we have a "wiggler" -- a robot that wiggles back and forth in
         * very short movements to confuse pattern matchers -- use a low
         * power shot to get it moving.
         */
//        if (speed < 2                           &&
//            target.getWigglePct() > 60          &&
//            target.getAvgAngleOfApproach() > 60 &&
//            target.getCurrEstMove() < 5         &&
//            target.getPreviousReaction() == null) {
//
//            return 0.2;
//        }



        /*
         * If target is not nearby, calculate how much energy
         * to use based on it's speed, distance, and angle of attack.
         */
        if (distance > JUMP) {

            speedRatio      = 1 - (speed / MAX_SPEED);
            distanceRatio   = 1 - (distance / BattleField.getDiagonal());
            angleRatio      = 1 - (angleOfApproach / 180);

            power *= BotMath.max(speedRatio,
                     BotMath.max(distanceRatio, angleRatio));

            /*
             * If our target is using on average more energy to fire at us
             * than we are at them, then increase our firepower.
             */

            if (myRobot.getOthers() < 2) {

                double powerDifference  =
                        target.getAvgShotMePower() -
                        target.getAvgShotAtPower();

                if (powerDifference > 0.0) {
                    if (powerDifference < 1) powerDifference += 1;
                    power *= powerDifference;
                }
            }
        }

        return BotMath.limit(power, MIN_BULLET_POWER, MAX_BULLET_POWER);
    }
}
