/*
 * PatternEnemyBullet - utility class
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */

package qohnil.blot;

import robocode.AdvancedRobot;

import java.io.Serializable;

import qohnil.util.Coords;
import qohnil.util.BotMath;
import qohnil.util.Polar;

public class PatternEnemyBullet extends VirtualBullet implements Serializable {

    static double reliability = 1.0;

    double directAngle = 0.0;

    public PatternEnemyBullet(double angle, double firePower, Coords coords,
                              long time, Coords targetCoords) {

        super(angle, firePower, coords, time);

        Polar p = BotMath.coordsToPolar(targetCoords, coords);
        directAngle = p.getBearing();
        hitTime = time + Math.round(BotMath.getBulletFlightTime(firePower, p.getDistance()));
    }

    /**
     * Should be called when a bullet is fired at us...
     */
    public PatternEnemyBullet(PatternAnalyzer patternAnalyzer, double firePower, Coords coords, long time,
                              Coords targetCoords,
                              double heading, double velocity) {

        this(getPatternPredictionAngle(patternAnalyzer, targetCoords, coords, heading, velocity, firePower),
                firePower, coords, time, targetCoords);

    }

    public static double getPatternPredictionAngle(PatternAnalyzer patternAnalyzer, Coords target, Coords src,
                                                  double heading, double velocity,
                                                  double firePower) {

        double bulletFlightTime = Math.round(
                BotMath.getBulletFlightTime(firePower, target.distance(src) + 500.0));
        if (bulletFlightTime < 0.0 || (int)bulletFlightTime < 0) {
            System.out.println("FATAL: BULLET_FLIGHT_TIME == " + bulletFlightTime);
        }
        patternAnalyzer.findSimilarIdx((int)bulletFlightTime);
        Polar reaction = new Polar(0.0, target.distance(src));
        for (int i = 0; i < 3; i++) {
            int predictedTime = (int)Math.round(BotMath.getBulletFlightTime(firePower, reaction.getDistance()));
            if (predictedTime < 0) {
                System.out.println("FATAL: predictedTime < 0: " + predictedTime + ", " + reaction.getDistance());
            }
            reaction = patternAnalyzer.getPrediction(predictedTime);
            if (reaction == null) {
                System.out.println("FATAL: pattern matcher returns NULL - " + predictedTime
                        + ", (max bulletFlightTime was calc. with " + bulletFlightTime + ")");
                reaction = new Polar(0.0, target.distance(src));
            }
        }
        // calculate updated bot position
        Coords predictedCoords = BotMath.polarToCoords(
                BotMath.normalizeAbsoluteAngle(heading + reaction.getBearing()),
                reaction.getDistance(), target);

        return BotMath.coordsToPolar(predictedCoords, src).getBearing();
    }

    public void setReliability(double reliability) {
        PatternEnemyBullet.reliability = reliability;
    }

    public double getReliability() {
        return PatternEnemyBullet.reliability;
    }

}
