/*
 * ObjectBuffer - utility class for Robocode robots.
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */

package qohnil.util;

/**
 * Stores data up to a defined upper limit (totalSize). If an entry is added
 * when the buffer is full, it "wraps around" and overwrites the oldest
 * entries. Retrieving the entries works from most recent (get(0) or get())
 * to oldest (get(size() - 1)).
 */
public class ObjectBuffer {
    protected Object[] buffer = null;
    protected int size = 0;
    protected int totalSize = 0;
    protected int idxCurrent = 0;

    public ObjectBuffer(int totalSize) {
        this.totalSize = totalSize;
        buffer = new Object[totalSize];
    }

    public ObjectBuffer(ObjectBuffer src) {
        this.totalSize = src.totalSize;
        buffer = new Object[totalSize];

        for (int i = src.size() - 1; i >= 0; i--) {
            this.add(src.get(i));
        }
    }

    public void clear() {
        size = 0;
        idxCurrent = 0;
    }

    public void add(Object o) {
        if (size < totalSize) {
            size++;
        }
        idxCurrent++;
        idxCurrent %= totalSize;
        buffer[idxCurrent] = o;
        if (o == null) {
            Logger.warn("WARNING: ObjectBuffer.add(): added null");
        }
    }

    public Object get(int idx) {
        if (idx >= size) {
            Logger.error("ERROR: ObjectBuffer.get(" + idx
                    + "): index out of bounds!");
            return null;
        }
        Object obj = buffer[ (idxCurrent - idx + totalSize) % totalSize ];
        if (obj == null) {
            Logger.warn("WARNING: ObjectBuffer.get(): returning null");
        }
        return obj;
    }

    public Object get() {
        return get(0);
    }

    public int size() {
        return size;
    }
}
