package gh;
import robocode.*;
import robocode.util.Utils;
import java.awt.Color;
import java.awt.Graphics.*;
import java.awt.geom.*;
import java.util.Vector;
import java.io.*;

/**
 * GrubbmGrb - a robot by Gert Heijenk
 *
 * Multiple movements and multiple guns
 *
 * Revision information:
 * v0.1    20040910 Just putting sample.Spinbot and sample.TrackFire together
 * v0.2    20040915 Introducing 'scantimeout', better 'onHitRobot' handling
 * v0.3    20040916 Introducing 'wallmove'-statemachine, better firepower on 'easy' targets
 * v0.4    20040919 Introducing 'scan'-statemachine, some tweaks in targetselection and 'onHitRobot'
 * v0.5    20040926 Solved some bugs in wallmove (e.g. not moving), rescanning from time to time
 * v0.6    20041026 Improved wall-avoidance, linear/circular targetting, start of 'real' design
 * v0.7    20041105 Added waves for virtual guns, added HOT-gun, some bulletpower tweaks
 * v0.8    20041110 Improved wavecheck and survival, solved small bulletpower-bug, added 'CTHOT'-gun
 * v0.8.1  20041112 v0.7 plus improved wavecheck and solved small bulletpower bug.
 * v0.8.2  20041113 Do fire before turning gun, wavecheck will now check correct angle.
 * v0.8.3  20041119 Ram disabled opponents, start moving when rammed, set startposition prepared against ramming.
 * v0.8.4  20041121 Solved 'rammove'-bug, improved anti-ram
 * v0.8.5  20041123 Included movement of GrubbmTwo v0.5
 * v0.9    20041128 Choose best movement (circling or oscillator) against opponent
 * v0.9.1  20041202 Re-evaluate the movement criteria
 * v0.9.2  20050202 Refactored CT-gun, added some dive-in protection
 * v0.9.3  20050204 Improved wall-handling CT-gun, removed buggy dive-in protection
 * v0.9.4  20050310 Minor tweaks in ram-, antiram- and onhitrobot-behaviour
 * v0.9.5  20050316 Removed complicated melee-stuff, removed specific anti-ram behaviour, removed startup-positioning
 * v0.9.6  20050322 Calculate nextpos for me and enemy, turn gun at T-1, align the waves with the bullets
 * v0.9.7  20050328 Use 'Orbital gun' as third gun
 * v0.9.8  20050331 Use rolling averages for gun-statistics
 * v0.9.9  20050403 Use other orbital gun as third gun
 * v0.9.10 20050424 First try to do some simple segmentation on distance
 * v1.0.0  20050426 Fresh baseline to start fiddling with movement
 * v1.0.1  20050522 Re-evaluation of movements: take into account me-hitting-him too
 * v1.0.2  20050523 Added orbital movement, use Vuen's ScoreClass as base for movement evaluation
 * v1.0.3  20050524 Improved movement-selection, orbital movement to trick LT+HOT guns
 * v1.0.4  20050530 Tweaked oscillator movement (a la nano.Grofvuil), tweaked orbital movement (def distance)
 * v1.0.5  20050602 Untweaked oscillator movement, replaced orbital movement with sloppy stop 'n go.
 * v1.0.6  20050619 Tweaking Stop'nGo movement (distance controller, wallsmoothing)
 * v1.0.7  20050620 Disabling spinning movement, Stop'nGo should handle it
 * v1.0.8  20050629 Use movement of mini.Gruwel as Oscillator movement
 * v1.1.0  20050705 Adding averaged velocity CT-gun, random gun, and segmentation on near-wall (RRGC investigation)
 * v1.1.1  20050822 Fixing TC-mode, cleanup of startup-code, improved random gun, bugfixing and tweaking Orbital/Stop'nGo
 * v1.1.2  20050826 Fixed improved random gun, introduced slow reaction on enemyfire to prevent 'lastbullethit'
 * v1.1.3  20050927 no slow reaction on enemyfire, several small tweaks
 * v1.2.0  20060223 small bugfix in movementselection and in virtualgun statistics
 *                  support for RobocodeSG
 *                  improved Circular Targeting near walls
 *                  distance dependent dive-in protection oscillator movement
 * v1.2.1  20060302 tweaked dive-in protection for Orbital, StopAndGo and Oscillator movement
 *                  improved wall-behaviour oscillator movement (less wallhits)
 *                  added INI-file (thanx to Loki)
 * v1.2.2  20060310 EndGame improvements (better energymanagement, close-in on non-firing enemies)
 * v1.2.3  20060715 Tryout with Raiko's movement (RRGC) instead of oscillator, just to check if movement-improvements are doable
 * v1.2.4  20060721 v1.2.1 with improved energymanagement
 *                  Fire when someone is threaten to ram you
 */
public class GrubbmGrb extends AdvancedRobot
{
	// some short reference constants
	private static final double PI 			= Math.PI;	// for short reference
	private static final double HALFPI		= PI / 2;	// for short reference
	private static final double SIXTHPI		= PI / 6;	// 30 degrees

	// some scanning values
	private static final  long SCANTIMEOUT		 = 4;	// the number of ticks I received no scanevent (lock is lost)

	// movement related stuff
	private static final int WALLMARGIN			= 17;	// the thickness of the border minus 1
	private static final double FULLSTOP		= 0.0;	// the absolute minimum speed
	private static final double MINSPEED		= 2.0;	// the minimum speed (sharp turn)
	private static final double MEDSPEED		= 4.0;	// the medium speed (fast turn)
	private static final double MAXSPEED		= 8.0;	// the maximum speed
	private static final double FIXPERCDAM		= 9.0;	// if enemy gets a hitpercentage of less than 9%, do not try other movements
	// oscillator
	private static final int WALLTOUCH			= 28;	// the area to avoid near walls
	private static final double BLINDMANSTICK	= 130;	// the distance when to check wallavoidance
	// orbital
	private static final int DEFDISTANCE		= 350;	// the default distance to keep from enemy
	private static final int MINDISTANCE		= 125;	// the minimum distance for Stop'nGo
	private static final int HOTDISTANCE		= 150;	// when hit within this distance, ignore the hit
	private static final double WALLCHECK		= 12;	// extra walldistancecheck
	private static final double DEFDIVEANGLE	= 10 * PI / 24;	// 75   degrees while default movement
	private static final double DEFHOTANGLE		=  6 * PI / 24;	// 45   degrees while full speed orbital movement
	private static final double DEFLCTANGLE		= 11 * PI / 24;	// 82.5 degrees while StopAndGo movement
	private static final double DEFDIVEDIST		= 600;

	// the available guns at the moment
	private static final int NOGUN	= -1;	// No gun selected yet
    private static final int CIRC	= 0;	// Circular Targeting
	private static final int HOT	= 1;	// Head On Targeting
	private static final int ORBIT	= 2;    // Sort of Linear/Orbital targeting of Iiley's Smog 2.5
	private static final int CTRA	= 3;	// Rolling Average Circular Targeting
	private static final int RANDOM	= 4;	// Random gun
	private static final int PM		= 3;	// Pattern Matching
	private static final int FIRSTGUN	= CIRC;		// First available gun
	private static final int LASTGUN	= RANDOM;	// Last available gun

	// the available movements at the moment
	private static final int NOMOVE			= -1;	// no movement selected yet
	private static final int MOVEORBITAL	= 0;	// orbital stop'ngo movement to trick HOT and LT/CT guns
	private static final int MOVEOSCILLATOR	= 1;	// default GrubbmTwo v0.5 movement
	private static final int MOVETOBEDEFINED= 2;	// yet to be defined killer movement
	private static final int FIRSTMOVEMENT	= MOVEORBITAL;		// first available movement
	private static final int LASTMOVEMENT	= MOVEOSCILLATOR;	// last available movement

	// gun related information
	private static double defRASpeed	= 4.0;		// startspeed for CTRA gun
	private EnemyInfo enemy;
	private Vector waveList[];
	private static GunInfo hitRatio[];
	private boolean enemyNearWall;				// set if enemy is near the wall (CT bulletpower adaption)
	private int gunType;			// the current selected gun
	private	double aimFactor = Math.random();	// for random gun

	// movement related information
	private static	MoveInfo movement[];
	private static	boolean	fullSpeed = true;	// true = Musashi-like, false = stop'n'go movement 0
	private Vector	eWaveList;		// enemy waves (more or less)
	private int		currMovement;	// the current selected movement
	private static	int notHotHits = 0;			// the number of notHotHits in Musashi-like movement
	private boolean doHOTCheck;		// do checking if I got hit by HOT
	private boolean	switchDir;		// must I switch direction in the current tick
	private boolean	startupMove;	// true if starting up, false if moving
	private long 	startupTime;	// the moment the movement will be started
	private long	lastEnemyFireTime;	// time that the enemy fired its last bullet
	private boolean endGame;		// true if the air is clear of enemybullets
	private boolean	ramMove;		// true if opponent is disabled and needs to be rammed
	private long	lastSwitch;		// the time I last switched directions

	private double	jiggleOffset;	// for oscillator movement
	private double	powerFired;		// the power the enemy fired at me

	// scanning and locking variables
	private long	lastscan;		// the time the last robotscanned_event was received
	private boolean	lockFound;		// default lock is NOT found
	private long	currTime;		// the time the last robotscanned_event was received

	// some info about myself
	private Point2D.Double	myPos = new Point2D.Double();		// my current position
	private Point2D.Double	myNextPos = new Point2D.Double();	// my next position
	private Point2D.Double	myPrevPos = new Point2D.Double();	// my previous position
	private double			myHeading;			// my current heading
	private double			firePower;			// the optimal firepower
	private double			currDirection;		// 1 if forward, -1 if backward
	private double			lastDirection;		// 1 if forward, -1 if backward
	private double			myCheckDistance;	// my distance to check for wall-avoidance
	private boolean			robothit;			// true if I hit other robot

	// statistics counters
	static int	bullhit;	// the total number of bullets that hit
	static int	bullmis;	// the total number of bullets that missed
	static int	wallhit;	// the total number of times I hit the wall
	static int	skipturn;	// the total number of skipped turns

	// some general info
	private double	fieldWidth;		// the width of the battlefield
	private double	fieldHeight;	// the heigth of the battlefield
    private RoundRectangle2D.Double playField;	// the playable battlefield
	private Rectangle2D.Double fireField;		// the shootable battlefield

	// some flags for the several challanges, intially set in readIniFile() !!
	private static boolean SG_flag;	// RobocodeSG graphics
	private static boolean TC_flag;	// Targetting/PM challenge (no movement, always fire 3.0, no energy-management)
	private static boolean MC_flag;	// Movement challenge	(do not fire or ram)
//	private static boolean GC_flag; 	// RR Gun Challenge (Raiko's movement, do not ram disabled bots)
	private static boolean RR_flag;	// RR Release if set, development otherwise

	/**
	 * run: Grubbm_Grb's default behavior
	 */
	public void run() {
		int i;

		// first of all, read the initialisation file
		readIniFile();

		// Give the robot an appealing look
		setColors( Color.red, Color.yellow.brighter(), Color.red.brighter());

		// First set the statistics
		fieldWidth = getBattleFieldWidth();
		fieldHeight = getBattleFieldHeight();
        playField = new RoundRectangle2D.Double( WALLTOUCH, WALLTOUCH,
	    			fieldWidth - 2 * WALLTOUCH, fieldHeight - 2 * WALLTOUCH, 50, 50);
        fireField = new Rectangle2D.Double( WALLMARGIN, WALLMARGIN,
					fieldWidth - 2 * WALLMARGIN, fieldHeight - 2 * WALLMARGIN);

		// Fill in some default info about the Enemy
		enemy = new EnemyInfo();
		// gunInfo is kept over the full battle
		if (hitRatio == null) {
			hitRatio = new GunInfo[LASTGUN + 1];
			for (i = FIRSTGUN; i <= LASTGUN; i++) {
				hitRatio[i] = new GunInfo();
			}
		}
		// moveInfo is kept over the full battle
		if (movement == null) {
			movement = new MoveInfo[LASTMOVEMENT + 1];
			for (i = FIRSTMOVEMENT; i <= LASTMOVEMENT; i++) {
				movement[i] = new MoveInfo();
			}
		}

		waveList = new Vector[LASTGUN + 1];
		for (i = FIRSTGUN; i <= LASTGUN; i++) {
			waveList[i] = new Vector();
		}

		// enemyWave list
		eWaveList = new Vector();

		// do not start moving until enemy fired its gun
		setMaxVelocity( FULLSTOP);
		startupTime = (long)(getGunHeat() / getGunCoolingRate() + 2);

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
		setAdjustRadarForGunTurn( true);

		// Make initial direction less predictable
		currDirection = (Math.random() < 0.5 ? 1 : -1);

		// Select the movement
		startupMove = true;		// initial state, before moving
		gunType = NOGUN;		// no gun selected yet
		currMovement = doSelectMovement();	// select the movement for this round
		movement[currMovement].newRound();

		// Start the mainloop
		while(true) {
			myPrevPos.setLocation( myPos.getX(), myPos.getY());	// Remember current position (to avoid getXXX calls)
			myPos.setLocation( getX(), getY());	// Remember current position (to avoid getXXX calls)
			myNextPos.setLocation( getX() + Math.sin( getHeadingRadians()) * getVelocity(), getY() + Math.cos( getHeadingRadians()) * getVelocity());	// Next position (approx)
			doCheckEnemyFired();	// Check if enemy fired
			doCheckMovement();		// Check if adaptions have to be made for movement
			doCheckRadar();			// Check the state of the radar
			doUpdateGunStats();		// Update the gun statistics with data from the waves
			doUpdateMoveStats();	// Update the movement statistics with data from the 'enemywaves'
			doFireGun();			// Fire the gun (with angle of previous tick)
			doFirePower();			// Select the fire power to use the next tick (gunmovement depends on it)
			doMoveGun();			// Move the gun to the desired position (for next tick)
			execute();				// execute all commands
		}
	}

	/**
	 * doCheckMovement: Check if something has to be done about the movement
	 *
	 * Exceptional situations:
	 *  - TargetingChallenge
	 *  - (nearly) disabled opponents
	 *	- startup-situation
	 *  - all 'stopped by' situations handles by normal movement handling
	 */
	void doCheckMovement() {

		// If TargetingChallenge, do not move
		if (TC_flag == true) return;

		// handle ramming the disabled if necessary
		// EndGame is handled in normal movement
		if ((endGame == true) && (ramMove == true)) {
			if (MC_flag == true) {
				setAhead(0);	// in MovementChallenge, just stop
			}
			else {
				setAhead( 20);
				setTurnRightRadians( enemy.getBearing());
				setMaxVelocity( Math.abs( getTurnRemaining()) > 45 ? MINSPEED : MAXSPEED);
			}
			return;
		}

		// handle startup, do not drive into a wall immediately (todo: try to get best angle to opponent)
		if (startupMove == true) {
			handleStartupMovement();
		}

		// check again, it may have changed
		if (startupMove == true)
			return;

		if (MOVEOSCILLATOR == currMovement) {
			handleOscillatorMovement();
		}
		else if (MOVEORBITAL == currMovement) {
			handleOrbitalMovement();
		}
//		else if (MOVETOBEDEFINED == currMovement) {
//			handleKillerMovement();
//		}
		else handleOscillatorMovement();

	}

	/**
	 * doSelectMovement: Find the right movement for the current opponent
	 *
	 */
	int doSelectMovement() {
		int i, gt;

		gt = FIRSTMOVEMENT;
		// first select best movement
		// if enemy has a hitpercentage of less than 9%, don't even try other movements
		for (i = FIRSTMOVEMENT + 1; i <= LASTMOVEMENT; i++) {
			if (movement[i].getHitRate() < movement[gt].getHitRate()) {
//				out.println("__: "+i+" "+movement[i].getHitRate()+"  "+gt+" "+movement[gt].getHitRate());
				gt = i;
			}
		}
		// Be nice for one-time movements, select it if it is not worse than the worst of the best movement.
		for (i = FIRSTMOVEMENT; i <= LASTMOVEMENT; i++) {
			if ((movement[i].getRound() == 1.0) && (movement[i].getHitRate() < movement[gt].getWorst())) {
//				out.println("I : "+i+" "+movement[i].getHitRate()+"  "+gt+" "+movement[gt].getWorst());
				gt = i;
			}
//			out.println(movement[i].getRound()+"  "+movement[i].getHitRate()+"  "+movement[i].getWorst());
		}
		out.println("Chosen movement: "+gt);
		return gt;
	}

	/**
	 * handleStartupMovement: Handle the movement during startup (until moving)
	 *
	 */
	void handleStartupMovement() {
		// if enemy is too close, start moving
		if (lockFound == true) {
			if ((enemy.getDistance() < MINDISTANCE) || (powerFired > 0.0))
				startupMove = false;
			if ((MOVEORBITAL == currMovement) && (Math.abs(Math.atan( Math.tan( enemy.getBearing()))) < 0.6))
				setTurnRightRadians( Utils.normalRelativeAngle(enemy.getBearing() + Math.PI/2));
		}
	}

	/**
	 * handleOrbitalMovement: Handle the orbital movement to trick HOT and LT/CT guns (Musashi)
	 *
	 */
	void handleOrbitalMovement() {
		double	myTurnAngle;		// the adapted turnangle (distancecompensation, wallsmoothing)
		double  newposx, newposy;	// my future (end)position
		double	perpAngle;			// angle to reach perpendicularity
		double	diveAngle;			// angle to compensate for distance
		double	tmpAngle;			// working variable
		double	distDiff;
		double	mySpeed;			// my current absolute speed
		double  maxDiveAngle;		// the maximum dive-in angle

		mySpeed = Math.abs( getVelocity());

		// calculate angle for perpendicularity, I might need it
		tmpAngle = Utils.normalRelativeAngle(enemy.getBearing() + HALFPI);
		perpAngle = Math.atan(Math.tan(tmpAngle));

		// Now do some dynamic distancing, stay at approx DEFDISTANCE.
		// Note that wallavoidance can override !
		// Stay roughly perpendicular to the opponent
		// This routine should be within one line, so I can use it in micro !!
		distDiff = enemy.getDistance() - DEFDISTANCE;
		if (Math.abs( distDiff) > 5) {
			diveAngle = (Math.abs(tmpAngle) < HALFPI ? currDirection : -currDirection);
			if (distDiff < 0) {
				diveAngle *= (Math.sqrt(Math.abs(distDiff)) / 25);
			}
			else {
				diveAngle *= -(Math.sqrt(Math.abs(distDiff)) / 60);
			}
//			out.println("  Distance: "+e.getDistance()+" adjust: "+Math.toDegrees(adjustAngle));
		}
		else {
			diveAngle = 0;
		}
		// adjust turnangle with distance-compensation
		myTurnAngle = perpAngle + diveAngle;

		// perform some wall-avoidance, try to slide away if moving near wall
		myHeading = getHeadingRadians() + myTurnAngle;
		myCheckDistance = BLINDMANSTICK * currDirection;
		newposx = myPos.getX() + Math.sin( myHeading) * myCheckDistance;	// calculate end-of-stick position
		newposy = myPos.getY() + Math.cos( myHeading) * myCheckDistance;
		// if endposition not in field, adapt
		if (!playField.contains( newposx, newposy)) {
			tmpAngle = Math.atan( Math.tan( calcAngleToWall()));
			// if adaption is too big (dive-in protection), just bounce
//			if ((fullSpeed == true) || (mySpeed == FULLSTOP)) {
//				if (PI / 2.7 < Math.abs(adjustAngle)) {
//					switchDir = true;
////					out.println("Switchdir Wallavoidance X: "+myPos.getX()+" Y: "+myPos.getY());
//				}
//				else
//					myTurnAngle += adjustAngle;
//			}
//			else
				myTurnAngle += tmpAngle;
		}
//		out.println("turn: "+Math.toDegrees(myTurnAngle));

		// perform some dive-in protection here
		myTurnAngle = Utils.normalRelativeAngle( myTurnAngle);
		if ((fullSpeed == true) || (mySpeed == FULLSTOP)) {
			if (fullSpeed == true)
				maxDiveAngle = DEFHOTANGLE - (enemy.getDistance() / DEFDIVEDIST) * SIXTHPI;
			else
				maxDiveAngle = DEFLCTANGLE - (enemy.getDistance() / DEFDIVEDIST) * SIXTHPI;
			if ((Math.abs(enemy.getBearing() - myTurnAngle) < maxDiveAngle) && (currDirection == 1)) {
//			if ((Math.abs(enemy.getBearing() - myTurnAngle) < maxDiveAngle)) {
				switchDir = true;
				myTurnAngle = perpAngle;
			}
			if ((Math.abs(enemy.getBearing() - myTurnAngle) > (PI - maxDiveAngle)) && (currDirection == -1)) {
//			if ((Math.abs(enemy.getBearing() - myTurnAngle) > (PI - maxDiveAngle))) {
				switchDir = true;
				myTurnAngle = perpAngle;
			}
		}

		// check if I have to turn to much (more than 90 degrees)
		if (switchDir == false) {
			if (Math.abs( myTurnAngle) > HALFPI) {
				switchDir = true;
				myTurnAngle = perpAngle;
			}
		}

		// Do some extra wallchecking to prevent wallhits
		if (switchDir == false) {
			myHeading = getHeadingRadians();
			myCheckDistance = WALLCHECK * currDirection;
			newposx = myPos.getX() + Math.sin( myHeading) * myCheckDistance;	// calculate end-of-stick position
			newposy = myPos.getY() + Math.cos( myHeading) * myCheckDistance;
			// if endposition not in field, adapt
			if (!playField.contains( newposx, newposy)) {
				myHeading = getHeadingRadians() + perpAngle;
				newposx = myPos.getX() - Math.sin( myHeading) * myCheckDistance;	// calculate end-of-stick position
				newposy = myPos.getY() - Math.cos( myHeading) * myCheckDistance;
				// if other side is in field, just bounce. Otherwise let default wallavoidance handle it
				if (playField.contains( newposx, newposy)) {
					switchDir = true;
//					out.println("Switchdir Extra wallcheck"+myPos.getX()+" Y: "+myPos.getY());
				}
//				else
//					out.println("Do nothing . . .");
			}
		}

		// when enemy is too close or I am not hit with !HOT, use full speed
		// otherwise switch speed when enemy fired
		if ((fullSpeed == false) && (enemy.getDistance() > MINDISTANCE)) {
			// Check if enemy fired a bullet. If so, swap speed (FULLSPEED <-> FULLSTOP)
			if (powerFired > 0.0) {
				setMaxVelocity( (mySpeed == FULLSTOP ? MAXSPEED : FULLSTOP));
			}
		}
		// fullspeed or very close, so move at fullspeed
		else {
			setMaxVelocity( MAXSPEED);
		}

		// finally set the preferred direction and turnangle
		if (switchDir == true) { currDirection = -currDirection; lastSwitch = getTime(); }
		if (TC_flag == false) setAhead( Double.POSITIVE_INFINITY * currDirection);
		setTurnRightRadians( myTurnAngle);
		switchDir = false;
	}

	/**
	 * handleOscillatorMovement: Handle the (double) oscillator movement
	 *
	 */
	void handleOscillatorMovement() {
		double  myDistance;			// my current distance to travel 
		double	myTurnAngle;		// the desired turnangle
		double  newposx, newposy;	// my future (end)position
		double	perpAngle;			// angle to reach perpendicularity
		double  maxDiveAngle;		// the maximum dive-in angle

		// calculate angle for perpendicularity, I might need it
		myTurnAngle = Utils.normalRelativeAngle(enemy.getBearing() + HALFPI);
		perpAngle = Math.atan(Math.tan(myTurnAngle));

		// Check if nearly at end of movement
		if (Math.abs(getDistanceRemaining()) < 1) { 
			myDistance = (Math.sin( currTime/13 + getRoundNum()) * Math.cos( currTime/29))*270 + (Math.random()*100 - 50);
			currDirection = (myDistance > 0 ? 1 : -1);
			// move away after robothit (to avoid rambots)
			if (true == robothit) {
				if (currDirection == lastDirection) {
					currDirection = -currDirection;
					myDistance = -myDistance;
				}
				robothit = false;
			}
			lastDirection = currDirection;
			setAhead( myDistance);

			jiggleOffset = Math.toRadians(Math.random()*46-23);
			myTurnAngle = Math.atan( Math.tan( perpAngle + jiggleOffset));
			setTurnRightRadians( myTurnAngle);
		}

		// perform some wall-avoidance, try to slide away if moving near wall
		myHeading = getHeadingRadians();
		myDistance = Math.abs( getDistanceRemaining());
		myCheckDistance = Math.min( BLINDMANSTICK, myDistance) * currDirection;
		newposx = myPos.getX() + Math.sin( myHeading) * myCheckDistance;	// calculate end-of-stick position
		newposy = myPos.getY() + Math.cos( myHeading) * myCheckDistance;
		// if endposition not in field, adapt
		if (!playField.contains( newposx, newposy)) {
			myTurnAngle = Math.atan( Math.tan( calcAngleToWall()));
			setTurnRightRadians( myTurnAngle);
//			out.println("turn: "+Math.toDegrees(myTurnAngle));
		}

		// perform some dive-in protection here
		maxDiveAngle = DEFDIVEANGLE - (enemy.getDistance() / DEFDIVEDIST) * SIXTHPI;
		myTurnAngle = Math.atan( Math.tan( perpAngle + jiggleOffset));
		myDistance += (Math.random() * 40);	// when reversing, move a bit more away from the wall
//		out.println("Dist: "+enemy.getDistance()+" maxDA: "+Math.toDegrees(maxDiveAngle));
		if ((Math.abs(enemy.getBearing() - getTurnRemainingRadians()) < maxDiveAngle) && (currDirection == 1)) {
//			out.println("F: perpAngle: "+perpAngle+" currDir: "+ currDirection);
			currDirection = -currDirection;
			setAhead( myDistance * currDirection);
			setTurnRightRadians( myTurnAngle);
		}
		if ((Math.abs(enemy.getBearing() - getTurnRemainingRadians()) > (PI - maxDiveAngle)) && (currDirection == -1)) {
//			out.println("B: perpAngle: "+perpAngle+" currDir: "+ currDirection);
			currDirection = -currDirection;
			setAhead( myDistance * currDirection);
			setTurnRightRadians( myTurnAngle);
		}

		// If a wallhit is eminent (next position not inside 'safezone'), turn sharply by reducing speed
		myCheckDistance = Math.min( WALLCHECK, myDistance) * currDirection;
		newposx = myPos.getX() + Math.sin( myHeading) * myCheckDistance;	// calculate end-of-stick position
		newposy = myPos.getY() + Math.cos( myHeading) * myCheckDistance;
		// if endposition not in field, adapt
		if ((!playField.contains( newposx, newposy)) && (Math.abs( getTurnRemaining()) > 10)) {
//			setMaxVelocity( Math.abs( getTurnRemaining()) > 15 ? MINSPEED : MAXSPEED);
			setMaxVelocity( MINSPEED);
		}
		else {
			setMaxVelocity( MAXSPEED);
		}
	}

	/**
	 * calcAngleToWall: Calculate the wall'smoothing' (or something that looks like it)
	 */
	public double calcAngleToWall() {
		double di;

		for (di = 0; di < PI; di += PI/72) {
			if (checkInPlayField( di) == true) {
				break;
			}
			if (checkInPlayField( -di) == true) {
				di = -di;
				break;
			}
		}
		return ( di);
	}

	/**
	 * checkInPlayField: Check if my future position is in the playField
	 */
	public boolean checkInPlayField( double di) {
		double px, py;

		px = myPos.getX() + Math.sin( myHeading + di) * myCheckDistance;
		py = myPos.getY() + Math.cos( myHeading + di) * myCheckDistance;
		return (playField.contains( px, py));
	}

	/**
	 * doCheckEnemyFired: Check if the enemy has fired a bullet. If so, rememeber it
	 */
	void doCheckEnemyFired() {

		powerFired = enemy.checkBulletFired();
		if (powerFired > 0.0) {
			eWaveList.add( new EnemyWave( powerFired));
			lastEnemyFireTime = getTime();
//			out.println("Added to wavelist: "+powerFired);
		}
		// check if enemy stopped firing and there are no bullets in the air anymore
		endGame = (eWaveList.size() == 0);
	}

	/**
	 * doCheckRadar: Check if rescanning has to be done. If not, just lock on target
	 *
	 * Exceptional situations:
	 *	- no lock anymore
	 */
	void doCheckRadar() {
		double radarmove;

		currTime = getTime();

	    // I have seemed to lost my lock, just find any opponent
		if ((lockFound == true) && ((currTime - lastscan) > SCANTIMEOUT)) {
			lockFound = false;
		}

		// I am in need of an(other) opponent, find one !
		if (lockFound == false) {
			setTurnRadarRight( Double.POSITIVE_INFINITY);
		}
		else {
			// make a nice 'botwide' scanarc
			radarmove = Utils.normalRelativeAngle( getHeadingRadians() + enemy.getBearing() - getRadarHeadingRadians());
			radarmove += (radarmove < 0.0 ? Math.atan( -20 / enemy.getDistance()) : Math.atan( 20 / enemy.getDistance()));
			setTurnRadarRightRadians( radarmove);
		}
	}

	/**
	 * doFirePower: Select the optimal firepower for the current target
	 */
	void doFirePower() {

		if (lockFound == false) {
			firePower = 0.0;
			return;
		}
		if (TC_flag == true) {
			firePower = Math.min( 3.0, getEnergy());
			return;
		}

		if (enemy.getDistance() > 850)
			firePower = 0.1;
		else if (enemy.getDistance() > 700)
		    firePower = 0.49;
		else if (enemy.getDistance() > 250)
		    firePower = 1.9;
		else
			firePower = 3;
		// awful place for this tweak, but it seems to help
		if ( gunType != NOGUN) {
			if (hitRatio[gunType].getRatio(enemy.getDistance()) > 0.33) {
				firePower *= 3;
				firePower = Math.min( 3.0, firePower);
			}
		}
		// do energymanagement
		if (enemy.getDistance() > MINDISTANCE)
			firePower = Math.min( getEnergy()/5-0.4, firePower);	// try to hold some reserve (0.4)
		else {
			firePower = Math.min( getEnergy()/3+0.5, firePower);	// but shoot when enemy gets too close
			firePower = Math.min( getEnergy(), firePower);			// shoot maximum is own energy
		}
		firePower = Math.min( enemy.getEnergy()/4 + 0.1, firePower);	// just kill enemy
		firePower = Math.max( 0.1, firePower);				// but always shoot at least 0.1

		// do not disable myself, only against rammers
		if ((enemy.getDistance() > MINDISTANCE) && (getEnergy() <= 0.1)) firePower = 0.0;

		if (MC_flag == true) firePower = 0.0;

		// do not shoot at disabled enemies, try to ram them
		// Introduces also occasional chasing just for fun !!
		// If ramming takes too long, just fire to collect bonus
		ramMove = false;
		if ((enemy.getEnergy() <= 0.1) && ((getTime() < (lastEnemyFireTime + 350)) || (MC_flag == true))) {
			firePower = 0.0;
			ramMove = true;
		}
	}

	/**
	 * doUpdateGunStats: Update gun statistics with data from the waves
	 */
	void doUpdateGunStats() {
		int i, j;
		// for every available gun
		for (i = FIRSTGUN; i <= LASTGUN; i++) {
			// for every still available wave
			for (j = waveList[i].size() - 1; j >= 0; j--) {
				if (((Wave)(waveList[i].elementAt(j))).wavehitcheck() == true) {
					hitRatio[i].setVHit( ((Wave)(waveList[i].elementAt(j))).getoriginaldistance(),
										 ((Wave)(waveList[i].elementAt(j))).getnotwall());
					waveList[i].removeElementAt(j);	// remove when virtual hit
				}
				else if (((Wave)(waveList[i].elementAt(j))).wavepasscheck() == true) {
					hitRatio[i].setVMiss( ((Wave)(waveList[i].elementAt(j))).getoriginaldistance(),
										  ((Wave)(waveList[i].elementAt(j))).getnotwall());
					waveList[i].removeElementAt(j);	// remove when virtual miss
				}
			}
		}
	}

	/**
	 * doUpdateMoveStats: Update move statistics with data from the enemywaves
	 */
	void doUpdateMoveStats() {
		int i;
		boolean hothit = false;
		// for every still available wave
		for (i = eWaveList.size() - 1; i >= 0; i--) {
			if (doHOTCheck == true) {
				if (((EnemyWave)(eWaveList.elementAt(i))).wavehitHOTcheck( lastSwitch) == true) {
					hothit = true;
				}
			}
			if (((EnemyWave)(eWaveList.elementAt(i))).wavepasscheck() == true) {
				movement[currMovement].gotFired(((EnemyWave)(eWaveList.elementAt(i))).getoriginalpower());
				eWaveList.removeElementAt(i);
			}
		}
		if (doHOTCheck == true) {
			if ((fullSpeed == true) && (hothit == false)) {
				notHotHits++;
				if (((getRoundNum() + 1) / notHotHits) > 2) {
					out.println("You hit me, but I am a forgiving bot ;-)");
				}
				else {
					out.println("Going to Stop'nGo mode !");
					fullSpeed = false;
				}
			}
			doHOTCheck = false;
		}
	}

	/**
	 * doMoveGun: Move the gun to the optimal angle to hit the target
	 */
	void doMoveGun() {
		int i;
		// if no target is locked, gun movement is done in doCheckRadar()
		if (lockFound == false) return;
		if (MC_flag == true) return;

		// if firepower is zero or gun is hot, warm of luke-warm, move it like HOT-gun
		if (((getGunHeat() / getGunCoolingRate()) > 3.0) || (firePower == 0.0)) {
			setTurnGunRightRadians( Utils.normalRelativeAngle( doGunHOT() - getGunHeadingRadians()));
		}
		else {
			for (i = FIRSTGUN; i <= LASTGUN; i++) {
				hitRatio[i].setBPower( firePower);
			}
			enemyNearWall = false;
			// gun is almost cool, start calculating for every available gun
			if ((FIRSTGUN <= HOT) && (LASTGUN >= HOT))
				hitRatio[HOT].setAngle( doGunHOT());
			if ((FIRSTGUN <= CIRC) && (LASTGUN >= CIRC))
				hitRatio[CIRC].setAngle( doGunCIRC( CIRC));
			if ((FIRSTGUN <= ORBIT) && (LASTGUN >= ORBIT))
				hitRatio[ORBIT].setAngle( doGunOrbit());
			if ((FIRSTGUN <= CTRA) && (LASTGUN >= CTRA))
				hitRatio[CTRA].setAngle( doGunCIRC( CTRA));
			if ((FIRSTGUN <= RANDOM) && (LASTGUN >= RANDOM))
				hitRatio[RANDOM].setAngle( doGunRandom());
			if (NOGUN == gunType) {
				gunType = doSelectGun(); // select gun once just (2-3 ticks) before firing
			}
			setTurnGunRightRadians( Utils.normalRelativeAngle( hitRatio[gunType].getAngle() - getGunHeadingRadians()));
		}
	}

	/**
	 * doFireGun: Fire the gun. Calculate angle for all virtual guns
	 */
	void doFireGun() {
		int i;

		if (MC_flag == true) return;
		// Only fire if possible and target locked.
		if ((getGunHeat() == 0.0) && (gunType != NOGUN) && (getGunTurnRemaining() == 0.0)) {
			// In TC-mode, just fire power 3.0 bullets
			if (TC_flag == true) {
				for (i = FIRSTGUN; i <= LASTGUN; i++) {
					hitRatio[i].setBPower( Math.min( getEnergy(), 3.0));
				}
			}
			else {
				// if not in TC-mode, do not disable yourself
				if (getEnergy() <= 0.1)
					hitRatio[gunType].setBPower( 0.0);
			}
			// fire only if bulletpower is set, so don't disable yourself !
			if (hitRatio[gunType].getBPower() >= 0.1) {
				setFire( hitRatio[gunType].getBPower());
				hitRatio[gunType].setFired();
//				out.println("Fire ! !");

				// for random gun
				double oldAimFactor = aimFactor;
				while(Math.abs(oldAimFactor - aimFactor) < .08) aimFactor = Math.random();

				for (i = FIRSTGUN; i <= LASTGUN; i++) {
					waveList[i].add( new Wave( Utils.normalRelativeAngle(hitRatio[i].getAngle() - getGunHeadingRadians()), hitRatio[i].getBPower(), !enemyNearWall));
				}
				gunType = NOGUN; // reset guntype, for every shot the guntype is re-evaluated
			}
		}
	}

	/**
	 * doSelectGun: Determine which gun is best in current circumstance
	 */
	int doSelectGun() {
		int i, gt;
		double dist;

		gt = FIRSTGUN;
		if ((bullhit + bullmis) < 8) return gt; // default always best gun ( == FIRSTGUN)
		dist = myPos.distance( enemy.getPos());
		for (i = FIRSTGUN + 1; i <= LASTGUN; i++) {
//			if (hitRatio[i].getVRatio() > hitRatio[gt].getVRatio())
			if (hitRatio[i].getRatio( dist, !enemyNearWall) > hitRatio[gt].getRatio( dist, !enemyNearWall))
				gt = i;
		}
		return gt;
	}

	/**
	 * doGunHOT: Calculate angle using Head On Targeting.
	 */
	double doGunHOT() {
		double tmpx = enemy.getNextPos().getX();
		double tmpy = enemy.getNextPos().getY();
		return( (PI / 2) - Math.atan2( tmpy - myNextPos.getY(), tmpx - myNextPos.getX()) );
	}

	/**
	 * doGunOrbit: Assume enemy is orbiting me with constant speed and distance.
	 *				Gun taken from Iiley's Smog 2.5
	 */
	double doGunOrbit() {
		double tmpb = getHeadingRadians() + enemy.getBearing() - enemy.getHeading() + 0.5*PI;
		if (enemy.getSpeed( CIRC) < 0) tmpb -= PI;
		return( getHeadingRadians() + enemy.getBearing() + Math.atan2(Math.cos(tmpb) * Math.abs(enemy.getSpeed( CIRC)), 14 + Math.sin(tmpb)));
	}

	/**
	 * doGunRandom: Random gun developed by myself
	 *				Take into account current lateral speed and direction change of target
	 */
	double doGunRandom() {
		double bspeed = (20.0 - 3.0 * hitRatio[RANDOM].getBPower());
//	gun from wiki.Shark, replaced by a slightly better own development
//		double maxEscapeAngle = Math.abs(8.0 / bspeed);
//		double fireAngle = maxEscapeAngle * (aimFactor * 2.0 - 1);

		long neededTicks = Math.round((enemy.getDistance() - 18) / bspeed);
		double sumspeed = 0;
		double speedsign = Math.sin( enemy.getHeading() - (enemy.getBearing() + getHeadingRadians()));
		double cspeed = enemy.getSpeed( CIRC) * speedsign;
		for (int i = 0; i < neededTicks; i++) {
			if (cspeed != -8) {
				cspeed -= 1;
				if (cspeed >= 0)
					cspeed -= 1;
				else if (cspeed < -8)
					cspeed = -8;
			}
			sumspeed += cspeed;
		}
		double minEscapeAngle = (sumspeed / neededTicks) / bspeed;

		sumspeed = 0;
 		cspeed = enemy.getSpeed( CIRC) * speedsign;
		for (int i = 0; i < neededTicks; i++) {
			if (cspeed != 8) {
				cspeed += 1;
				if (cspeed <= 0)
					cspeed += 1;
				else if (cspeed > 8)
					cspeed = 8;
			}
			sumspeed += cspeed;
		}
		double maxEscapeAngle = (sumspeed / neededTicks) / bspeed;
		double fireAngle = ((maxEscapeAngle - minEscapeAngle) * aimFactor) + minEscapeAngle;

//		out.println("MaxEA="+Math.toDegrees(maxEscapeAngle)+" MinEA="+Math.toDegrees(minEscapeAngle));
		return( getHeadingRadians() + enemy.getBearing() + fireAngle);
	}

	/**
	 * doGunCIRC: Calculate angle using a Circular targeting with current speed.
	 *			  ToDo: put all code here and not in EnemyInfo
	 */
	double doGunCIRC(int guntype) {
		Point2D.Double point = new Point2D.Double();
		double gunAngle;
		double bpower;
		double head, chead, speed, bspeed;
		double tmpx, tmpy;
		int i = 0;
		int deltahittime;

		// perform an iteration to find a reasonable accurate expected position
		bpower = hitRatio[guntype].getBPower();

		head = enemy.getHeading();
		chead = enemy.getHeadingChange();
		speed = enemy.getSpeed(guntype);
		// nextposition is calculated (by hittime = -1), because of headingchange
		tmpx = enemy.getPos().getX();
		tmpy = enemy.getPos().getY();
		point.setLocation( tmpx, tmpy);
		deltahittime = -1;
		do {
			tmpx += Math.sin( head) * speed;
			tmpy +=	Math.cos( head) * speed;
			head += chead;
			deltahittime++;
			// if position not in field, adapt
			if ((!fireField.contains( tmpx, tmpy)) && (deltahittime > 0)) {
				enemyNearWall = true;
				point.setLocation( warpPoint( point));
				bspeed = (myNextPos.distance( point) -18) / deltahittime;
				bpower = Math.max( Math.min( (20 - bspeed) / 3.0, 3.0), 0.1);
				break;
			}
			point.setLocation( tmpx, tmpy);
		} while ( (int)Math.round( (myNextPos.distance( point) - 18) / (20 - (3 * bpower))) > deltahittime);
		// put enemylocation into field
		point.setLocation( warpPoint( point));

		hitRatio[guntype].setBPower( bpower);	// save adjusted firePower

		// Return gunAngle
		gunAngle = ( (PI / 2) - Math.atan2( point.getY() - myNextPos.getY(), point.getX() - myNextPos.getX()) );
		return gunAngle;
	}

	/**
	 * warpPoint: Put point into field with a margin of 2 * WALLMARGIN (34)
	 */
	public Point2D.Double warpPoint( Point2D.Double p)
	{
		Point2D.Double pos = new Point2D.Double();

		pos.x = Math.min( fieldWidth  - 2 * WALLMARGIN, Math.max( 2 * WALLMARGIN, p.x));
		pos.y = Math.min( fieldHeight - 2 * WALLMARGIN, Math.max( 2 * WALLMARGIN, p.y));
		return pos;
	}

	/**
	 * onScannedRobot: What to do when you see another robot (just save everything and do nothing here)
	 *
	 */
	public void onScannedRobot( ScannedRobotEvent e) {
		// only put the data into the EnemyInfo class
		// interpretation is done in the mainloop
//		ScoreInfo.onScannedRobot( e);

		// prevent a 'lock lost' event
		lastscan = getTime();
	    lockFound = true;
		enemy.onScannedRobot( e);
		enemy.myInfo( getHeadingRadians(), getX(), getY(), lastscan);
	}

	/**
	 * onRobotDeath: What to do when someone else dies
	 */
//	do nothing if someone else dies, you have won!

	/**
	 * onHitRobot: Bounce off !
	 */	
	public void onHitRobot(HitRobotEvent e) {
		// if I hit the opponent, just bounce off
		enemy.onHitRobot();
		if (e.isMyFault()) {
			robothit = true;
			switchDir = true;
		}
	}

	/**
	 * onHitWall:  Handle collision with wall.
	 */
	public void onHitWall(HitWallEvent e)
	{
		wallhit++;
//		should be handled by movement itself
//		if (MOVEORBITAL == currMovement) {
//			switchDir = true;
//		}

//		out.println("Hit wall");
	}

	/**
	 * onHitByBullet:  I am hit, handle some (movement) statistics.
	 */
	public void onHitByBullet(HitByBulletEvent e)
	{
		movement[currMovement].gotHit( e.getPower());
		enemy.onHitByBullet( e);
		// If I do a fullspeed Orbital movement, check if I got hit by HOT
		if ((MOVEORBITAL == currMovement) && (fullSpeed == true))
			 doHOTCheck = true;
//		out.println("I am Hit ! !");
	}

	/**
	 * onSkippedTurn:  Handle a skipped turn.
	 */
	public void onSkippedTurn(SkippedTurnEvent e)
	{
		skipturn++;
		out.println("AAaaaarghhhhh");
	}

	/**
	 * onWin: Show my private victory dance
	 */
	public void onWin(WinEvent e)
	{
		// remember some variables
		defRASpeed = Math.abs(enemy.getSpeed(CTRA));	// remember Rolling Average speed for next round
		printStats();
		//Victory dance	
		setTurnGunRight( Double.POSITIVE_INFINITY);
		setTurnRadarLeft( Double.POSITIVE_INFINITY);
		setTurnRight(10);
		ahead(20);
		waitFor(new RadarTurnCompleteCondition(this));
	}

	/**
	 * onDeath: Show some statistics
	 */
	public void onDeath(DeathEvent e)
	{
		// remember some variables
		defRASpeed = Math.abs(enemy.getSpeed(CTRA));	// remember Rolling Average speed for next round
		// Show some statistics	
		printStats();
	}

	/**
	 * printStats: Print statistics on end of round
	 */
	public void printStats( )
	{
		int i,j,k;
		boolean lastRound;

		// close current movement
		movement[currMovement].closeRound();
		lastRound = (getRoundNum() == (getNumRounds() - 1));
		if (MC_flag == false) {
			out.println("Bullets HIT:" + bullhit);
			out.println("Bullets mis:" + bullmis);
			out.println("Promillage :" + (bullhit*1000)/(bullhit + bullmis));
		}
		if (lastRound == true) {
			out.println("Wall hits  :" + wallhit);
			out.println("SkipTurns  :" + skipturn);
		}

		if (MC_flag == false) {
			if (lastRound == true) {
				for (i = FIRSTGUN; i<=LASTGUN; i++) out.println("Gun "+i+" Fired: "+hitRatio[i].getFired());
				out.println("Virtual results :");
				if (RR_flag == true) {
					for (i = FIRSTGUN; i<=LASTGUN; i++) out.println("Gun "+i+"  "+hitRatio[i].getVRatio());
					for (j = 0; j < 4; j++) {
						k = FIRSTGUN;
						for (i = FIRSTGUN+1; i <=LASTGUN; i++) {
							if (hitRatio[i].getRatio(j*300) > hitRatio[k].getRatio(j*300))
								k = i;
						}
						out.println("Dist "+j+" --> Gun "+k+"  "+hitRatio[k].getRatio(j*300));
					}
				}
				else {
					for (j = 0; j < 4; j++)
						for (i = FIRSTGUN; i<=LASTGUN; i++)
							// first print farwall, then nearwall
							out.println("Dist "+j+" Gun "+i+"  "+Math.round(hitRatio[i].getRatio(j*300, true)*10000)/10000.0+"  "+Math.round(hitRatio[i].getRatio(j*300, false)*10000)/10000.0);
//							out.println("Dist "+j+" Gun "+i+"  "+Math.round(hitRatio[i].getRatio(j*300)*10000)/10000.0);
					out.println("Virtual RA results :");
					for (i = FIRSTGUN; i<=LASTGUN; i++) out.println("Gun "+i+"  "+hitRatio[i].getRatio( -1));
				}
			}
		}

		out.println("Movement results : (hitpercentage)");
		for (i = FIRSTMOVEMENT; i<=LASTMOVEMENT; i++) out.println("Move "+i+"  "+movement[i].getRound()+"  "+movement[i].getHitRate());

		// remove all existing waves
		for (i = FIRSTGUN; i <= LASTGUN; i++) {
			// for every still available wave
			for (j = waveList[i].size() - 1; j >= 0; j--) {
				waveList[i].removeElementAt(j);
			}
		}
	}

	/**
	 * onBulletHit: Yes, I hit somebody
	 */
	public void onBulletHit(BulletHitEvent e)
	{
		enemy.onBulletHit( e);
		bullhit++;
	}

	/**
	 * onBulletMissed: Damn, I missed (again)
	 */
	public void onBulletMissed(BulletMissedEvent e)
	{
		bullmis++;
	}

	/**
	 * onPaint: Show some stuff, maybe I am calculating something wrong
	 */
	public void onPaint(java.awt.Graphics2D g) {
		if (SG_flag == true) {
			int i, j;
			double dist;
			Point2D botpos = new Point2D.Double();
			Point2D bulpos = new Point2D.Double();

			// print my waves to check if I am aligned
			if (waveList == null) return;
			for (i = FIRSTGUN; i <= LASTGUN; i++) {
				// for every still available wave
				for (j = waveList[i].size() - 1; j >= 0; j--) {
					dist   = ((Wave)(waveList[i].elementAt(j))).getBulletDistance();
					botpos = ((Wave)(waveList[i].elementAt(j))).getBulletOrigin();
					bulpos = ((Wave)(waveList[i].elementAt(j))).getBulletLocation();
					g.setColor(Color.red);
					g.drawOval((int)(botpos.getX() - dist),(int)(fieldHeight - botpos.getY() - dist),(int)(dist*2),(int)(dist*2));
					g.setColor(Color.yellow);
					g.drawLine((int)botpos.getX(), (int)(fieldHeight - botpos.getY()), (int)bulpos.getX(), (int)(fieldHeight - bulpos.getY()));
				}
			}
//			g.drawOval((int)(getX()-50),(int)(600-getY()-50), 100, 100);
		}
    }

	/**
	 * This routine is copied from Loki's bot Freya !!
	 *
	 * <p>This function reads the GrubbmGrb.ini file for behaviour and graphics.</p>
	 * <p>The parameter "behaviour" may have following values:
	 * <ul>
	 * <li>normal   : moves and schoots (default-value).</li>
	 * <li>reference: sits still, shoots with power 3.0.</li>
	 * <li>challenge: moves normally, does not shoot.</li>
	 * </ul></p>
	 * <p>The parameter "graphics" may have following values:
	 * <ul>
	 * <li>true : enable RobocodeSG graphics.</li>
	 * <li>false: disable RobocodeSG graphics (default value).</li>
	 * </ul></p>
	 * <p>The parameter "logging" may have following values:
	 * <ul>
	 * <li>true : extended type of debug output.</li>
	 * <li>false: release type of debug output (default value).</li>
	 * </ul></p>
	 */
	private void readIniFile() {
		if (getRoundNum() == 0) {

			// set the flags initially
			SG_flag = TC_flag = MC_flag = false;	// No graphics, TargetingChallenge or MovementChallenge
			RR_flag = true;							// Release output to debug window

			try {
				BufferedReader	bufferedreader = new BufferedReader(new FileReader(getDataFile("GrubbmGrb.ini")));
				String			s;
				String			gedrag = null;

				while ((s = bufferedreader.readLine()) != null) {
					s = s.trim();

					if (!s.substring(0, 1).equals("#")) {
						if (s.length() > 11 && s.substring(0, 10).equalsIgnoreCase("behaviour=")) {
							gedrag = s.substring(10).trim();
						}

						if (s.length() > 10 && s.substring(0, 9).equalsIgnoreCase("graphics=")) {
							SG_flag = s.substring(9).trim().equalsIgnoreCase("false") ^ true;
						}

						if (s.length() > 9 && s.substring(0, 8).equalsIgnoreCase("logging=")) {
							RR_flag = s.substring(8).trim().equalsIgnoreCase("true") ^ true;
						}
					}
				}

				if (gedrag != null) {
					TC_flag = gedrag.equalsIgnoreCase("reference") ? true : false;
					MC_flag = gedrag.equalsIgnoreCase("challenge") ? true : false;
				}
			} catch (IOException ioexception) {
				out.println("readIniFile(): IOException reading data GrubbmGrb.ini: " + ioexception);
			}

			if (TC_flag == true) out.println("Targeting Challenge");
			if (MC_flag == true) out.println("Movement Challenge");
			if (SG_flag == true) out.println("RobocodeSG Graphics ON");
		}
	}

	/**
	 * GunInfo: Info about (virtual) hits and misses
	 */
	class GunInfo {
		private double vfired;		// virtual fired
		private double vhit;		// virtual hit
		private double vdfired[]	= new double[4];	// virtual fired, segmented on distance
		private double vdhit[]		= new double[4];	// virtual hit, segmented on distance
		private double radRatio[]	= new double[4];	// hitrato, segmented on distance
		private double vdwfired[][]	= new double[4][2];	// virtual fired, segment distance and nearwall
		private double vdwhit[][]	= new double[4][2];	// virtual hit, segment distance and nearwall
		private double radwRatio[][]= new double[4][2];	// hitratio, segment distance and nearwall
		private double tfired;		// really fired (temporary until each bullet is counted for)
		private double rfired;		// really fired (n.u.y)
		private double rhit;		// really hit (n.u.y)
		private double lastangle;	// the absolute angle to the target
		private double bulletpower;	// the bulletpower to use
		private double raRatio = 0.1;		// rolling average hitratio
		private int    distsegment;	// distance segment
		private int    wallsegment;	// nearwall segment

		public GunInfo() {
			radRatio[0] = radRatio[1] = radRatio[2] = radRatio[3] = 0.1;
			radwRatio[0][0] = radwRatio[1][0] = radwRatio[2][0] = radwRatio[3][0] = 0.1;
			radwRatio[0][1] = radwRatio[1][1] = radwRatio[2][1] = radwRatio[3][1] = 0.1;
		}

		public void setFired( ) {
			tfired++;
		}

		public double getFired( ) {
			return tfired;
		}

//		public void setHit( ) {
//			rfired++;
//			rhit++;
//		}

//		public void setMiss( ) {
//			rfired++;
//		}

		public double getRatio( double dist, boolean notwall) {
//			out.println("Hit = "+vhit+"  Fired = "+vfired);
			if (dist < 0) return raRatio;
			distsegment = distToSegment( dist);
			wallsegment = wallToSegment( notwall);
			// only handle segment mature if 8 evalutations are made
			if (vdwfired[distsegment][wallsegment] < 8)
				if (vdfired[distsegment] < 8)
					return vhit / vfired;
				else
					return vdhit[distsegment] / vdfired[distsegment];
			else
				return vdwhit[distsegment][wallsegment] / vdwfired[distsegment][wallsegment];
		}

		public double getRatio( double dist) {
//			out.println("Hit = "+vhit+"  Fired = "+vfired);
			if (dist < 0) return raRatio;
			distsegment = distToSegment( dist);
			if (vdfired[distsegment] < 8)
				return vhit / vfired;
			else
				return vdhit[distsegment] / vdfired[distsegment];
		}

		public void setVHit( double dist, boolean notwall) {
			vfired++;
			vhit++;
			raRatio = 0.99 * raRatio + 0.01;
			distsegment = distToSegment( dist);
			wallsegment = wallToSegment( notwall);
			vdfired[distsegment]++;
			vdwfired[distsegment][wallsegment]++;
			vdhit[distsegment]++;
			vdwhit[distsegment][wallsegment]++;
			radRatio[distsegment] = 0.99 * radRatio[distsegment] + 0.01;
			radwRatio[distsegment][wallsegment] = 0.99 * radwRatio[distsegment][wallsegment] + 0.01;
		}

		public void setVMiss( double dist, boolean notwall) {
			vfired++;
			raRatio = 0.99 * raRatio;
			distsegment = distToSegment( dist);
			wallsegment = wallToSegment( notwall);
			vdfired[distsegment]++;
			vdwfired[distsegment][wallsegment]++;
			radRatio[distsegment] = 0.99 * radRatio[distsegment];
			radwRatio[distsegment][wallsegment] = 0.99 * radwRatio[distsegment][wallsegment];
		}

		public double getVRatio( ) {
			return vhit / vfired;
		}

		public void setAngle( double angle) {
			lastangle = angle;
		}

		public double getAngle( ) {
			return lastangle;
		}

		public void setBPower( double bpower) {
			bulletpower = bpower;
		}

		public double getBPower( ) {
			return bulletpower;
		}

		private int distToSegment( double dist) {
			if (dist < 250) return 0;
			if (dist < 500) return 1;
			if (dist < 750) return 2;
			return 3;
		}

		private int wallToSegment( boolean notwall) {
			if (notwall == true) return 0;
			return 1;
		}
	}

	/**
	 * MoveInfo: Info about movements
	 */
	class MoveInfo {
		private double round;		// number of rounds played
		private double tpower;		// total potential damage fired
		private double tdamage;		// total damage taken
//		private double tratio;		// total damagepercentage
		private double rpower;		// potential damage fired this round
		private double rdamage;		// damage taken this round
//		private double rhotdam;		// damage taken this round from HOT bullets
		private double rratio;		// round damagepercentage
		private double wratio = FIXPERCDAM;	// worst damagepercentage

		public void newRound( ) {
			round++;
			rdamage = 0;
			rpower = 0;
		}

		public void closeRound( ) {
			if (rpower != 0) {
				rratio = (rdamage * 100) / rpower;
				if (rratio > wratio) wratio = rratio;
			}
//			if (rdamage > wdamage) wdamage = rdamage;
		}

		public double getRound( ) {
			return round;
		}

		public void gotHit( double power) {
			double dam = 4 * power + 2 * Math.max( power - 1, 0);
			tdamage += dam;
			rdamage += dam;
//			out.println(getTime()+"I got hit with power "+power);
		}

		public void gotFired( double power) {
			double dam = 4 * power + 2 * Math.max( power - 1, 0);
			tpower += dam;
			rpower += dam;
//			out.println(getTime()+"Bullet missed with power "+power);
		}

		public double getHitRate( ) {
			if (round == 0) return FIXPERCDAM;
//			out.println("TDam: "+tdamage+" TPow: "+tpower);
			return ( (tdamage * 100) / tpower);
		}
		
//		public double getHitRatio( ) {
//			if (0 == round) return 0;
////			out.println("Rounds: "+round+" damage: "+tdamage);
//			return (tdamage / round);
//		}

		public double getWorst( ) {
			if (round == 0) return FIXPERCDAM;
			return wratio;
		}
	}

	/**
	 * EnemyInfo: Info about enemies
	 */
	class EnemyInfo {
		/*
		 * Storage for all sorts of info about enemy.
		 */
		private double  enemyBearing;
		private double  enemyHeading;
		private double  enemyHeadingChange;
		private double  enemySpeed;
		private double  enemyDistance;
		private double	enemyEnergy = 100.0;
		private double	enemyPrevSpeed;
		private double	enemyPrevEnergy;
		private double	energyDropCorrection;
		private double	enemyBulletPower;
		private double  enemySpeedRA = defRASpeed;
		private double	enemyLastSpeedSign = 1;
		private long    scantime;	// game time of newest scan
		private long	lastscan;	// game time of previous scan
		private boolean	collision;	// if true, robot collision. if false, possible enemy hit wall
//		private long	hittime;	// game time of latest bullethit
		private Point2D.Double enemyPos	= new Point2D.Double();
		private Point2D.Double enemyNextPos	= new Point2D.Double();
		private Point2D.Double enemyPrevPos = new Point2D.Double();

		public EnemyInfo() {
		}

		public void onScannedRobot (ScannedRobotEvent e) {
//			if (enemyName == NULL) enemyName = e.getName();
			enemyBearing = e.getBearingRadians();
			enemyHeadingChange = robocode.util.Utils.normalRelativeAngle(e.getHeadingRadians() - enemyHeading);
			enemyHeading = e.getHeadingRadians();
			enemyDistance = e.getDistance();
			enemyPrevSpeed = enemySpeed;
			enemySpeed = e.getVelocity();
//			out.println("EnemySpeed = "+enemySpeed);
			enemyPrevEnergy = enemyEnergy;
			enemyEnergy = e.getEnergy();
		}

		public void myInfo ( double myhead, double posx, double posy, long curtime) {
			double absbearing_rad = myhead + enemyBearing;
			lastscan = scantime;
			scantime = curtime;
			enemyPrevPos.setLocation( enemyPos.getX(), enemyPos.getY());
			enemyPos.setLocation( posx + Math.sin( absbearing_rad) * enemyDistance, posy + Math.cos( absbearing_rad) * enemyDistance);
			enemyNextPos.setLocation( enemyPos.getX() + Math.sin( enemyHeading) * enemySpeed, enemyPos.getY() + Math.cos( enemyHeading) * enemySpeed);

			// Trying to circumvent 'Cool Movement', use previous speed instead of current speed
			if (Math.abs(enemyPrevSpeed) > 0.4) {
				enemyLastSpeedSign = enemyPrevSpeed / Math.abs(enemyPrevSpeed);
			}
			if ((Math.abs(enemySpeed) > 0.4) || (scantime > startupTime))
				enemySpeedRA = (enemySpeedRA * 49 + Math.abs(enemySpeed)) / 50;
	
		}
		
		public void onBulletHit( BulletHitEvent e) {
			double damage = 4 * e.getBullet().getPower() + 2 * Math.max( e.getBullet().getPower() - 1, 0);
			energyDropCorrection += damage;
//			out.println(hittime+" Bullet hit with damage "+damage);
		}

		public void onHitByBullet( HitByBulletEvent e) {
			double damage = -3 * e.getPower();
			energyDropCorrection += damage;
//			out.println(hittime+" Hit by bullet with energy gain "+damage);
		}

		public void onHitRobot() {
//			energyDropCorrection += 0.6;
			collision = true;
		}

		public double checkBulletFired() {
			double bpower = 0.0;
			if (enemyEnergy != enemyPrevEnergy) {
				if (collision == true) {
					energyDropCorrection += 0.6;
				}
				if ((collision == false) && (enemySpeed == 0) && (Math.abs( enemySpeed - enemyPrevSpeed) > 2)) {
					energyDropCorrection += (Math.abs(enemyPrevSpeed / 2) - 0.5);
				}
				bpower = enemyPrevEnergy - (enemyEnergy + energyDropCorrection);
				if (bpower < 0.09) bpower = 0.0;
			}
//			if (bpower != 0.0) out.println(currTime+"Bullet Fired !!! Energydrop: "+bpower);
			energyDropCorrection = 0.0;
			collision = false;
			enemyBulletPower = bpower;
			return bpower;
		}

		public double getBearing() {
			return enemyBearing;
		}

		public double getHeading() {
			return enemyHeading;
		}

		public double getHeadingChange() {
			return enemyHeadingChange;
		}

		public double getSpeed( int guntype) {
			if (CTRA == guntype)
				return enemySpeedRA * enemyLastSpeedSign;
			else
				return enemySpeed;
		}

		public double getEnergy( ) {
			return enemyEnergy;
		}

		public double getDistance( ) {
			return enemyDistance;
		}

		public double getBulletPower( ) {
			return enemyBulletPower;
		}

		public Point2D.Double getPos( ) {
			return enemyPos;
		}

		public Point2D.Double getNextPos( ) {
			return enemyNextPos;
		}

		public Point2D.Double getPrevPos( ) {
			return enemyPrevPos;
		}
	}

	/**
	 * Wave: Info about (virtual) bullets
	 *
	 * Inspired by PEZ, although completely different implementation
	 */
	class Wave {
		private long time;
		private double bulletVelocity;
		private double fireAngle;
		private boolean notWall;
		private Point2D oldRLocation = new Point2D.Double();
		private Point2D oldELocation = new Point2D.Double();
		private Point2D bulletLocation = new Point2D.Double();

		public Wave( double dangle, double bpower, boolean farwall) {
		    this.time = currTime;
		    this.bulletVelocity = 20 - (3 * bpower);
		    this.fireAngle = getGunHeadingRadians() + dangle;
	    	this.oldRLocation.setLocation( myPos);
		    this.oldELocation.setLocation( enemy.getPos());
			this.notWall = farwall;
		}

		/**
		 * wavehitcheck: Check if wave hit the current position of the enemy
		 */
		public boolean wavehitcheck() {
		    if (getOthers() == 0) {
				return false;
		    }
		    double bulletDistance = bulletVelocity * ( currTime + 1 - time);
//			out.println(getTime()+"VBullet: "+ (oldRLocation.getX() + Math.sin( fireAngle) * bulletDistance)+", "+(oldRLocation.getY() + Math.cos( fireAngle) * bulletDistance)+", "+Math.toDegrees(fireAngle));
	    	if (bulletDistance > oldRLocation.distance( enemy.getNextPos()) - 29) {
				bulletLocation.setLocation( oldRLocation.getX() + Math.sin( fireAngle) * bulletDistance,
					oldRLocation.getY() + Math.cos( fireAngle) * bulletDistance);
				if (Math.abs( bulletLocation.getX() - enemy.getNextPos().getX()) < 19) {
					if (Math.abs( bulletLocation.getY() - enemy.getNextPos().getY()) < 19) {
						return true;
					}
				}
			}
			return false;
		}

		/**
		 * wavepasscheck: Check if wave passed the current position of the enemy
		 */
		public boolean wavepasscheck() {
		    if (getOthers() == 0) {
				return false;
		    }
		    double bulletDistance = bulletVelocity * ( currTime + 1 - time);
	    	if (bulletDistance > oldRLocation.distance( enemy.getNextPos()) + 30) {
				return true;
			}
			return false;
		}

		/**
		 * getoriginaldistance: Get the distance when bullet was fired, necessary for 'segmentation'
		 */
		public double getoriginaldistance() {
		    return oldRLocation.distance( oldELocation);
		}

		/**
		 * getnotwall: Check if bullet fired when enemy was not near a wall, necessary for 'segmentation'
		 */
		public boolean getnotwall() {
		    return notWall;
		}

		/**
		 * getBulletDistance: return the distance the bullet has travelled since firing
		 *                    for onPaint()
		 */
		public double getBulletDistance() {
		    return (bulletVelocity * ( currTime + 1 - time));
		}

		/**
		 * getBulletOrigin: return the point the bullet was fired from
		 *                  for onPaint()
		 */
		public Point2D getBulletOrigin() {
		    return (oldRLocation);
		}

		/**
		 * getBulletLocation: return the current location of the virtual bullet
		 *                 for onPaint()
		 */
		public Point2D getBulletLocation() {
		    double bulletDistance = bulletVelocity * ( currTime + 1 - time);
			bulletLocation.setLocation( oldRLocation.getX() + Math.sin( fireAngle) * bulletDistance,
					oldRLocation.getY() + Math.cos( fireAngle) * bulletDistance);
		    return (bulletLocation);
		}
	}

	/**
	 * EnemyWave: Info about enemy bullets
	 *
	 * At the moment only for movement-selection
	 */
	class EnemyWave {
		private long time;
		private double bulletVelocity;
		private double bulletPower;
		private double fireAngle;
		private Point2D oldRLocation = new Point2D.Double();
		private Point2D oldELocation = new Point2D.Double();
		private Point2D bulletLocation = new Point2D.Double();

		public EnemyWave( double bpower) {
		    this.time = currTime - 1;
		    this.bulletVelocity = 20 - (3 * bpower);
			this.bulletPower = bpower;
//		    this.fireAngle = Utils.normalRelativeAngle(enemy.getBearing() + getHeadingRadians() + PI);
	    	this.oldRLocation.setLocation( myPrevPos);
		    this.oldELocation.setLocation( enemy.getPrevPos());
		    this.fireAngle = ( (PI / 2) - Math.atan2( oldRLocation.getY() - oldELocation.getY(), oldRLocation.getX() - oldELocation.getX()) );
		}

		/**
		 * wavehitHOTcheck: Check if enemywave hit me with a HOT bullet
		 */
		public boolean wavehitHOTcheck( long switchtime) {
		    if (getOthers() == 0) {
				return true;
		    }
		    double bulletDistance = bulletVelocity * ( currTime - time);
	    	if (bulletDistance > oldELocation.distance( myPrevPos) - 28) {
				if (bulletDistance < HOTDISTANCE) {
					out.println(currTime+" Too close hit !");
					return true;
				}
				if (switchtime > time) {
					out.println(currTime+" Switch hit !!");
					return true;
				}
			}
			return false;
		}

		/**
		 * wavepasscheck: Check if enemywave passed my current position
		 */
		public boolean wavepasscheck() {
		    if (getOthers() == 0) {
				return false;
		    }
		    double bulletDistance = bulletVelocity * ( currTime - time);
	    	if (bulletDistance > oldELocation.distance( myPos) + 30) {
				return true;
			}
			return false;
		}

		/**
		 * getoriginalpower: Get the power of the fired bullet
		 */
		public double getoriginalpower() {
		    return bulletPower;
		}
	}
}

