package nat.ether.wavesurfing;

import java.util.*;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.*;

import nat.ether.utils.M;
import nat.ether.utils.Store;
import nat.ether.utils.Wave;
import robocode.*;

// XXX tick wave
public class WaveManager {
	// Painting config
	public static boolean EnemyWavePaintingEnabled = true;

	public final ArrayList<Wave> activeFiringWaves;
	private double enemyEnergy = 100d;
	private LinkedList<Point2D> positionList = new LinkedList<Point2D>();
	private LinkedList<Double> angleList = new LinkedList<Double>();
	private LinkedList<Store> scanList = new LinkedList<Store>();

	public WaveManager() {
		activeFiringWaves = new ArrayList<Wave>();
	}

	public void onTick(ScannedRobotEvent e, Point2D enemyPosition, Point2D myLocation, Store scan) {

		// Check for new firing wave
		double energyDiff = enemyEnergy - e.getEnergy();
		enemyEnergy = e.getEnergy();

		if (!M.isNear(energyDiff, 0d)) {
			if (energyDiff > 0.09 && energyDiff < 3.01d || M.isNear(enemyEnergy, 0d)) {
				Wave wave = new Wave(energyDiff, positionList.getFirst(), e.getHeadingRadians(), e.getVelocity(), angleList.get(1), scanList.getFirst(), e.getTime() - 1l);
				activeFiringWaves.add(wave);
			} else {
				System.out.println("Unknown energy drop detected.");
			}
		}

		positionList.addFirst(enemyPosition);
		angleList.addFirst(M.getAngle(enemyPosition, myLocation));
		scanList.add(scan);

		// Update each wave status
		for (int i = 0; i < activeFiringWaves.size(); i++) {
			Wave wave = activeFiringWaves.get(i);
			if (wave.status == Wave.PASSED) {
				activeFiringWaves.remove(i--);
			}
			wave.update(myLocation);
		}
	}

	public void onBulletHit(BulletHitEvent e) {
		enemyEnergy -= Rules.getBulletDamage(e.getBullet().getPower());
	}

	public void onHitByBullet(HitByBulletEvent e) {
		enemyEnergy += Rules.getBulletHitBonus(e.getBullet().getPower());
	}

	public ArrayList<Wave> getSurfableWaves(Point2D robotPosition) {
		ArrayList<Wave> surfableWaves = new ArrayList<Wave>(activeFiringWaves.size());
		for (Wave wave : activeFiringWaves) {
			if (wave.fireLocation.distanceSq(robotPosition) > wave.distanceTraveled * wave.distanceTraveled) {
				surfableWaves.add(wave);
			}
		}
		return surfableWaves;
	}

	public Wave findAndRemoveWave(robocode.Bullet b) {
		Wave w = null;

		Point2D loc = new Point2D.Double(b.getX(), b.getY());
		for (Wave wave : activeFiringWaves) {
			if (Math.abs(wave.distanceTraveled - loc.distance(wave.fireLocation)) < 80 && Math.abs(b.getVelocity() - wave.bulletVelocity) < 0.5) {
				w = wave;
				break;
			}
		}

		if (w != null)
			activeFiringWaves.remove(w);

		return w;
	}

	public void onPaint(Graphics2D g) {
		if (!EnemyWavePaintingEnabled)
			return;

		for (Wave wave : activeFiringWaves) {
			switch (wave.status) {
			case Wave.PASSED:
				g.setColor(Color.red.darker().darker().darker());
				break;
			case Wave.ACTIVE:
				g.setColor(Color.darkGray);
				break;
			case Wave.INTERSECTION:
				g.setColor(Color.green.darker().darker());
			}
			Point2D GF0 = M.project(wave.fireLocation, wave.getAngle(0d), wave.distanceTraveled);
			g.draw(M.point(wave.fireLocation, 3d));
			g.draw(M.point(wave.fireLocation, wave.distanceTraveled));
			g.draw(new Line2D.Double(wave.fireLocation, GF0));
			g.draw(M.point(GF0, 6d));

			if (wave.status == Wave.INTERSECTION) {
				g.setColor(Color.red.darker().darker());
				Point2D upper = M.project(wave.fireLocation, wave.getAngle(wave.hitRange.upper), wave.distanceTraveled);
				Point2D lower = M.project(wave.fireLocation, wave.getAngle(wave.hitRange.lower), wave.distanceTraveled);
				g.draw(new Line2D.Double(wave.fireLocation, upper));
				g.draw(new Line2D.Double(wave.fireLocation, lower));
			}
		}

	}
}
